"""
Unit tests for ChalkML Engine
"""

import pytest
import pandas as pd
from pathlib import Path
import tempfile
import shutil

from chalkml import get_chalkml_engine


@pytest.fixture
def temp_dir():
    """Create temporary directory for tests"""
    temp = tempfile.mkdtemp()
    yield temp
    shutil.rmtree(temp)


@pytest.fixture
def sample_csv(temp_dir):
    """Create sample CSV for testing"""
    df = pd.DataFrame({
        'name': ['Alice', 'Bob', 'Charlie'],
        'age': [25, 30, 35],
        'salary': [50000, 60000, 75000],
        'city': ['NYC', 'LA', 'Chicago']
    })
    
    csv_path = Path(temp_dir) / 'test_data.csv'
    df.to_csv(csv_path, index=False)
    return str(csv_path)


class TestPositionParsing:
    """Test position notation parsing"""
    
    def test_parse_left_position(self):
        engine = get_chalkml_engine()
        # 01N = index 0 (1st column)
        assert engine._parse_position('01N', 10) == 0
        assert engine._parse_position('02N', 10) == 1
        assert engine._parse_position('10N', 10) == 9
    
    def test_parse_right_position(self):
        engine = get_chalkml_engine()
        # N01 = last (-1)
        assert engine._parse_position('N01', 10) == 9
        assert engine._parse_position('N02', 10) == 8
        assert engine._parse_position('N10', 10) == 0
    
    def test_parse_range(self):
        engine = get_chalkml_engine()
        # 01N:05N = indices 0 to 4 (inclusive)
        start, end = engine._parse_range('01N:05N', 10)
        assert start == 0
        assert end == 5  # Exclusive end


class TestRemoveOperations:
    """Test remove operations"""
    
    def test_remove_column(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.remove_column(sample_csv, '01N')
        
        assert success is True
        assert 'removed' in message.lower()
        
        # Verify column was removed
        df = pd.read_csv(sample_csv)
        assert len(df.columns) == 3
        assert 'name' not in df.columns
    
    def test_remove_last_column(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.remove_column(sample_csv, None)
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert len(df.columns) == 3
        assert 'city' not in df.columns
    
    def test_remove_row(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.remove_row(sample_csv, '01N')
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert len(df) == 2


class TestMoveOperations:
    """Test move and copy operations"""
    
    def test_move_column(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.move_column(sample_csv, '01N', 'N01')
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        # 'name' should now be last column
        assert df.columns[-1] == 'name'
    
    def test_copy_column(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.copy_column(sample_csv, '01N', '02N')
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert len(df.columns) == 5  # Original 4 + 1 copy


class TestSmartFill:
    """Test smart fill strategies"""
    
    def test_fill_mean(self, temp_dir):
        # Create data with missing values
        df = pd.DataFrame({
            'values': [10, 20, None, 40, 50]
        })
        csv_path = Path(temp_dir) / 'missing.csv'
        df.to_csv(csv_path, index=False)
        
        engine = get_chalkml_engine()
        success, message = engine.fill_smart(str(csv_path), '01N', 'mean')
        
        assert success is True
        
        df_result = pd.read_csv(csv_path)
        assert df_result['values'].isnull().sum() == 0
        # Mean of [10, 20, 40, 50] = 30
        assert df_result['values'].iloc[2] == 30.0
    
    def test_fill_forward(self, temp_dir):
        df = pd.DataFrame({
            'values': [10, None, None, 40]
        })
        csv_path = Path(temp_dir) / 'forward.csv'
        df.to_csv(csv_path, index=False)
        
        engine = get_chalkml_engine()
        success, message = engine.fill_smart(str(csv_path), '01N', 'forward')
        
        assert success is True
        
        df_result = pd.read_csv(csv_path)
        assert df_result['values'].iloc[1] == 10.0
        assert df_result['values'].iloc[2] == 10.0


class TestFeatureEngineering:
    """Test feature engineering operations"""
    
    def test_derive_column(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.derive_column(
            sample_csv,
            'salary_per_age',
            'col:03N/col:02N'
        )
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert 'salary_per_age' in df.columns
        # Alice: 50000/25 = 2000
        assert df['salary_per_age'].iloc[0] == 2000.0
    
    def test_one_hot_encode(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.one_hot_encode(sample_csv, '04N')
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        # Should have one-hot encoded columns
        assert any('city_' in col for col in df.columns)


class TestDesignPatterns:
    """Test design pattern operations"""
    
    def test_map_pattern(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.map_pattern(sample_csv, '02N', 'x * 2')
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert 'age_mapped' in df.columns
    
    def test_reduce_pattern(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.reduce_pattern(
            sample_csv,
            ['02N', '03N'],
            'sum',
            'total'
        )
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert 'total' in df.columns
        # Alice: 25 + 50000 = 50025
        assert df['total'].iloc[0] == 50025
    
    def test_scan_pattern(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.scan_pattern(sample_csv, '02N', 'cumsum')
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        assert 'age_scan' in df.columns
        # Cumulative sum: 25, 55, 90
        assert df['age_scan'].iloc[2] == 90


class TestControlFlow:
    """Test control flow operations"""
    
    def test_if_else(self, sample_csv):
        engine = get_chalkml_engine()
        success, message = engine.apply_if_else(
            sample_csv,
            '03N',
            'col:02N>28',
            'Senior',
            'Junior'
        )
        
        assert success is True
        
        df = pd.read_csv(sample_csv)
        # Alice (25) should be 'Junior'
        assert df['salary'].iloc[0] == 'Junior'
        # Bob (30) should be 'Senior'
        assert df['salary'].iloc[1] == 'Senior'


class TestUndo:
    """Test undo functionality"""
    
    def test_undo_operation(self, sample_csv):
        engine = get_chalkml_engine()
        
        # Read original data
        df_original = pd.read_csv(sample_csv)
        original_cols = len(df_original.columns)
        
        # Remove a column
        engine.remove_column(sample_csv, '01N')
        df_after = pd.read_csv(sample_csv)
        assert len(df_after.columns) == original_cols - 1
        
        # Undo
        success, message = engine.undo_last()
        assert success is True
        
        df_restored = pd.read_csv(sample_csv)
        assert len(df_restored.columns) == original_cols


if __name__ == '__main__':
    pytest.main([__file__, '-v'])
