"""
Generator Objectives Module
============================
EELT7030 — Operation and Expansion Planning of Electric Power Systems  
Federal University of Paraná (UFPR)

Author
------
Augusto Mathias Adams <augusto.adams@ufpr.br>

Summary
-------
This module defines the **objective function** for the generator
subsystem within the MDI framework. The formulation represents the
total operational and investment cost to be minimized across the
planning horizon.

Description
-----------
The generator objective aggregates both the **variable operating cost**
and the **investment cost** associated with generator units.  
The total objective function is defined as follows:

\[
\min_{P, x}
\; C_{gen} =
\sum_{g,t,p} \big( c^{op}_g \, P_{g,t,p}
\;+\;
c^{inv}_g \, x_{g,t} \big)
\]

where:
- \(c^{op}_g\) — variable operating cost of generator \(g\);
- \(c^{inv}_g\) — investment cost of generator \(g\);
- \(P_{g,t,p}\) — power generated by unit \(g\) at time \(t\) and load level \(p\);
- \(x_{g,t}\) — binary or continuous variable indicating the generator's active state.

The resulting expression is assigned as a Pyomo `Objective` with sense
`minimize`.

Functions
---------
set_objective_generator(m)
    Builds and attaches the total generator cost objective to the Pyomo model.

Notes
-----
- The objective expression is scalar and represents the sum of all
  operational and investment expenditures within the generator subsystem.
- All required variables and parameters must be defined in the model prior
  to calling this function.
- This formulation can be combined with additional objectives (e.g.,
  emissions minimization, reliability indices) within a multi-objective
  optimization framework.

References
----------
[1] CEPEL. *DESSEM — Manual de Metodologia*, 2023.  
[2] Unsihuay Vila, C. *Introdução aos Sistemas de Energia Elétrica*, Lecture Notes,
    EELT7030/UFPR, 2023.

Module Dependencies
-------------------
- **Internal:** Generator data and variable definitions.  
- **External:** ``pyomo.environ`` (Objective, minimize)
"""

from pyomo.environ import Objective, minimize


def set_objective_generator(m):
    """
    Define and attach the generator cost minimization objective.

    Constructs the total cost function for the generator subsystem,
    including operational and investment costs, and registers it as
    a Pyomo objective with sense of minimization.

    Parameters
    ----------
    m : pyomo.environ.ConcreteModel
        Pyomo model instance containing generator sets, parameters,
        and decision variables.

    Returns
    -------
    pyomo.environ.ConcreteModel
        The same model instance with an additional attribute ``OBJ``
        representing the total generator cost objective.

    Notes
    -----
    The objective is mathematically defined as:

    .. math::

        \min_{P,x} C_{gen} =
        \sum_{g,t,p} \big( c^{op}_g \, P_{g,t,p}
        + c^{inv}_g \, x_{g,t} \big)

    This formulation captures both operational and investment costs
    across all generator units, time periods, and demand levels.
    """
    def _obj_rule(m):
        return sum(
            # Operational cost term (proportional to generated energy)
            m.gen_c_op[g] * m.gen_P[g, t, p]
            +
            # Investment cost term (annualized, associated with existence)
            m.gen_c_inv[g] * m.gen_x[g, t]
            for g in m.GU for t in m.T for p in m.P
        )

    m.OBJ = Objective(expr=_obj_rule(m), sense=minimize)
    return m

