import cv2
import numpy as np

from pathlib import Path
import requests
from urllib import parse, request
from PIL import Image
from typing import Union, Optional, Any, Iterable, Callable
from collections import OrderedDict
from multiprocessing.pool import ThreadPool
from multiprocessing import Pool as ProcessPool
from tqdm import tqdm
import time

__all__ = [
    "is_url",
    "url_to_image",
    "is_valid_image",
    "numpy_to_pillow",
    "pillow_to_numpy",
    "parallel_process",
]


def is_url(url: str, check: bool = False) -> bool:
    """
    Validate if the given string is a URL and optionally check if the URL exists online.

    Args:
        url (str): The string to be validated as a URL.
        check (bool, optional): If True, performs an additional check to see if the URL exists online.

    Returns:
        (bool): True for a valid URL. If 'check' is True, also returns True if the URL exists online.

    Examples:
        >>> valid = is_url("https://www.example.com")
        >>> valid_and_exists = is_url("https://www.example.com", check=True)
    """
    try:
        url = str(url)
        result = parse.urlparse(url)
        assert all([result.scheme, result.netloc])  # check if is url
        if check:
            with request.urlopen(url) as response:
                return response.getcode() == 200  # check if exists online
        return True
    except Exception:
        return False


def url_to_image(
    url: str, readFlag: int = cv2.IMREAD_COLOR, headers=None
) -> Optional[np.ndarray]:
    """
    Download an image from a URL and decode it into an OpenCV image.

    Args:
        url (str): URL of the image to download.
        readFlag (int, optional): Flag specifying the color type of a loaded image.
            Defaults to cv2.IMREAD_COLOR.

    Returns:
        Optional[np.ndarray]: Decoded image as a numpy array if successful, else None.
    """
    if headers is None:
        headers = {
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36"
        }
    try:
        response = requests.get(url, headers=headers, timeout=10)
        response.raise_for_status()
        image_array = np.frombuffer(response.content, dtype=np.uint8)
        image = cv2.imdecode(image_array, readFlag)
        return image
    except requests.RequestException as e:
        print(f"Request failed: {e}")
        return None
    except Exception as e:
        print(f"Image decode failed: {e}")
        return None


def is_valid_image(path: Union[str, Path]) -> bool:
    """
    Checks whether the given file is a valid image by attempting to open and verify it.

    Args:
        path (Union[str, Path]): Path to the image file.

    Returns:
        bool: True if the image is valid, False otherwise.

    Raises:
        None: All exceptions are caught internally and False is returned.
    """
    try:
        with Image.open(path) as img:
            img.verify()  # Verify that it is, in fact, an image
        return True
    except:
        return False


class Cache:
    def __init__(self, capacity: int):
        if capacity <= 0:
            raise ValueError("capacity must be a positive integer")
        self._capacity = capacity
        self._cache = OrderedDict()

    def put(self, key: Any, value: Any) -> None:
        if key in self._cache:
            return
        if len(self._cache) >= self._capacity:
            self._cache.popitem(last=False)
        self._cache[key] = value

    def get(self, key: Any, default: Optional[Any] = None) -> Any:
        return self._cache.get(key, default)


def pillow_to_numpy(img):
    img_numpy = np.asarray(img)
    if not img_numpy.flags.writeable:
        img_numpy = np.array(img)
    return img_numpy


def numpy_to_pillow(img, mode=None):
    return Image.fromarray(img, mode=mode)


def parallel_process(
    func: Callable,
    data: Iterable,
    use_threads: bool = False,
    num_workers: int = 4,
    store_results: bool = True,
    show_progress: bool = True,
    prog_desc: Optional[str] = None,
    prog_leave: bool = True,
):
    """
    General-purpose function for parallel processing using multiple processes.

    Args:
        func (Callable):
            The function to execute in parallel. It should accept a single input item
            (each element of `data`) as its argument.

        data (Iterable):
            An iterable of input items, each passed as the argument to `func`.
            Can be any iterable — e.g. list, generator, or range.

        use_threads (bool, optional): 
            If True, use ThreadPool. Defaults to False (ProcessPool).

        num_workers (int, optional):
            Number of worker processes to use. Defaults to 4.

        store_results (bool, optional):
            Whether to collect and return the function outputs.
            If False, results are discarded after execution (useful for side-effect-only tasks).
            Defaults to True.

        show_progress (bool): Whether to display a progress bar during processing.

        prog_desc (str, optional):
            Custom description text for the progress bar (from tqdm). Defaults to None.

        prog_leave (bool, optional):
            Whether to leave the progress bar on screen after completion. Defaults to True.

    Returns:
        Tuple[List[Any] | None, float]:
            - results: List of all outputs from `func`, if `store_results=True`. Otherwise, None.
            - duration: Total wall-clock execution time in seconds.

    Raises:
        TypeError:
            If `func` is not callable.

        Note:
            Exceptions raised inside child processes are caught and logged internally.
            The main process will not crash due to worker errors.
    """

    if not callable(func):
        raise TypeError("func must be a callable function.")

    start_time = time.time()

    results = [] if store_results else None

    PoolClass = ThreadPool if use_threads else ProcessPool

    total_items = len(data) if hasattr(data, "__len__") else None

    with PoolClass(processes=num_workers) as pool:
        iterator = pool.imap_unordered(func, data)
        if show_progress:
            iterator = tqdm(iterator, desc=prog_desc, leave=prog_leave, total=total_items)
        for res in iterator:
            if store_results and res != None:
                results.append(res)

    duration = time.time() - start_time

    return (results, duration)
