from importlib.resources import files as package_files
import re
import sys
import functools
import argparse
from SCons.Action import Action, CommandAction
from SCons.Builder import Builder
from SCons.Script import Delete
from SCons.Variables import Variables, BoolVariable, EnumVariable

import SCons
import SCons.Util
import subprocess
import logging
import shlex
import os.path
import os

from SCons.Script.Main import main as scons_main, python_version_unsupported, version_string, path_string, revert_io, _exec_main, memory_stats, count_stats, SConsPrintHelpException, print_objects, print_memoizer, print_time, _main, _scons_user_error, exit_status, _scons_syntax_error, _scons_internal_error
from SCons.Script.Main import AddOption
from steamroller.engines import registry
from steamroller.environment import Environment
try:
    from SCons.Script.Main import ENABLE_JSON
except:
    ENABLE_JSON=False

def _exec_main(parser, values) -> None:
    sconsflags = os.environ.get('SCONSFLAGS', '')
    all_args = sconsflags.split() + sys.argv[1:]
    
    options, args = parser.parse_args(all_args, values)

    if hasattr(options, "verbose"):
        options.set_option("no_progress", False)
    else:
        options.set_option("no_progress", True)
    
    if hasattr(options, "new_project"):
        directories = [
            ("data", "'data/' is for materials *not* generated by the experiments, i.e. the initial inputs"),
            ("scripts", "'scripts/' is for defining the steps that make up the experimental pipeline"),
            ("work", "'work/' is for everything generated by the experiments, i.e. *ephemeral and reproducible*")
        ]
        files = [
            ("SConstruct", "SConstruct", "The 'SConstruct' file defines experimental structure by defining and linking together steps from the scripts/ directory"),
            ("custom.py.example", "custom.py.example", "The 'custom.py.example' file itself isn't read by steamroller, but can be copied to 'custom.py' and used to override variables in the 'SConstruct' file to fit a particular environment or desired experiments"),
            ("gitignore", ".gitignore", "The '.gitignore' file (note the initial dot) tells git to ignore certain file-patterns that are ephemeral or environment-specific"),
        ]
        scripts = [
            ("preprocess_data.py", "scripts/preprocess_data.py", ""),
            ("shuffle_data.py", "scripts/shuffle_data.py", ""),
            ("train_model.py", "scripts/train_model.py", ""),
            ("apply_model.py", "scripts/apply_model.py", ""),
            ("generate_report.py", "scripts/generate_report.py", ""),            
        ]

        all_code = []
        
        sys.stderr.write("Creating a standard directory structure\n\n")
        for d, e in directories:
            sys.stderr.write("\t" + e + "\n")
            if os.path.exists(d):
                sys.stderr.write("\tDirectory '{}' already exists, not creating it\n".format(d))
            else:
                os.mkdir(d)
            sys.stderr.write("\n")

        sys.stderr.write("Creating basic steamroller files\n\n")
        for src, tgt, e in files:
            sys.stderr.write("\t" + e + "\n")
            if os.path.exists(tgt):
                sys.stderr.write("\tFile '{}' already exists, not creating it\n".format(tgt))
            else:
                all_code.append(package_files("steamroller").joinpath("data/" + src).read_text())
                with open(tgt, "wt") as ofd:
                    ofd.write(all_code[-1])
            sys.stderr.write("\n")

        sys.stderr.write("Creating example script files, each of which will perform one step in the experiment\n\n")
        for src, tgt, e in scripts:            
            if os.path.exists(tgt):
                sys.stderr.write("\tFile '{}' already exists, not creating it\n".format(tgt))
            else:
                sys.stderr.write("\tCreating '{}'\n".format(tgt))
                all_code.append(package_files("steamroller").joinpath("data/" + src).read_text())
                with open(tgt, "wt") as ofd:
                    ofd.write(all_code[-1])
            sys.stderr.write("\n")

        sys.stderr.write("Creating example data\n\n")
        if os.path.exists("data/materials.txt"):
            sys.stderr.write("\tFile 'data/materials.txt' already exists, not creating it\n\n")
        else:
            sys.stderr.write("\tCreating 'data/materials.txt'\n\n")
            with open("data/materials.txt", "wt") as ofd:
                ofd.write("\n".join(all_code))

        sys.stderr.write("You should now be able to invoke 'steamroller -n' to see what the experiments would run, and remove the '-n' to actually run them.\n\n")
        sys.exit()
    
    if isinstance(options.debug, list) and "pdb" in options.debug:
        import pdb

        class SConsPdb(pdb.Pdb):
            """Specialization of Pdb to help find SConscript files."""

            def lookupmodule(self, filename: str) -> Optional[str]:
                """Helper function for break/clear parsing -- SCons version.

                Translates (possibly incomplete) file or module name
                into an absolute file name. The "possibly incomplete"
                means adding a ``.py`` suffix if not present, which breaks
                picking breakpoints in sconscript files, which often don't
                have a suffix. This version fixes for some known names of
                sconscript files that don't have the suffix.

                .. versionadded:: 4.6.0
                """
                if os.path.isabs(filename) and os.path.exists(filename):
                    return filename
                f = os.path.join(sys.path[0], filename)
                if os.path.exists(f) and self.canonic(f) == self.mainpyfile:
                    return f
                root, ext = os.path.splitext(filename)
                base = os.path.split(filename)[-1]
                if ext == '' and base not in KNOWN_SCONSCRIPTS:  # SCons mod
                    filename = filename + '.py'
                if os.path.isabs(filename):
                    return filename
                for dirname in sys.path:
                    while os.path.islink(dirname):
                        dirname = os.readlink(dirname)
                    fullname = os.path.join(dirname, filename)
                    if os.path.exists(fullname):
                        return fullname
                return None

        SConsPdb().runcall(_main, parser)

    elif options.profile_file:
        from cProfile import Profile

        prof = Profile()
        try:
            prof.runcall(_main, parser)
        finally:
            prof.dump_stats(options.profile_file)

    else:
        _main(parser)

    


def main() -> None:
    global OptionsParser
    global exit_status
    global first_command_start
    global ENABLE_JSON

    # Check up front for a Python version we do not support.  We
    # delay the check for deprecated Python versions until later,
    # after the SConscript files have been read, in case they
    # disable that warning.
    if python_version_unsupported():
        msg = "scons: *** SCons version %s does not run under Python version %s.\n"
        sys.stderr.write(msg % (SConsVersion, python_version_string()))
        sys.stderr.write("scons: *** Minimum Python version is %d.%d.%d\n" %minimum_python_version)
        sys.exit(1)

    parts = ["SCons by Steven Knight et al.:\n"]
    try:
        import SCons
        parts.append(version_string("SCons", SCons))
    except (ImportError, AttributeError):
        # On Windows there is no scons.py, so there is no
        # __main__.__version__, hence there is no script version.
        pass
    parts.append(path_string("SCons", SCons))
    parts.append(SCons.__copyright__)
    version = ''.join(parts)

    from SCons.Script import SConsOptions
    parser = SConsOptions.Parser(version)
    values = SConsOptions.SConsValues(parser.get_default_values())
    parser.add_option(
        "--verbose",
        "-V",
        dest="verbose",
        default=False,
        action="store_true",
        help="Show 'Reading/Building' progress messages"
    )
    parser.add_option(
        "--new_project",
        dest="new_project",
        default=False,
        action="store_true",
        help="If set, all other options are ignored, and steamroller will create a new template project in the current directory.  If existing files conflict with template files, they will *not* be overwritten."
    )
    OptionsParser = parser

    try:
        try:
            _exec_main(parser, values)
        finally:
            revert_io()
    except SystemExit as s:
        if s:
            exit_status = s.code
    except KeyboardInterrupt:
        print("scons: Build interrupted.")
        sys.exit(2)
    except SyntaxError as e:
        _scons_syntax_error(e)
    except SCons.Errors.InternalError:
        _scons_internal_error()
    except SCons.Errors.UserError as e:
        _scons_user_error(e)
    except SConsPrintHelpException:
        parser.print_help()
        exit_status = 0
    except SCons.Errors.BuildError as e:
        print(e)
        exit_status = e.exitstatus
    except:
        # An exception here is likely a builtin Python exception Python
        # code in an SConscript file.  Show them precisely what the
        # problem was and where it happened.
        SCons.Script._SConscript.SConscript_exception()
        sys.exit(2)

    memory_stats.print_stats()
    count_stats.print_stats()

    if print_objects:
        SCons.Debug.listLoggedInstances('*')
        #SCons.Debug.dumpLoggedInstances('*')

    if print_memoizer:
        SCons.Memoize.Dump("Memoizer (memory cache) hits and misses:")

    # Dump any development debug info that may have been enabled.
    # These are purely for internal debugging during development, so
    # there's no need to control them with --debug= options; they're
    # controlled by changing the source code.
    SCons.Debug.dump_caller_counts()
    SCons.Taskmaster.dump_stats()

    if print_time:
        total_time = time.time() - SCons.Script.start_time
        if num_jobs == 1:
            ct = cumulative_command_time
        else:
            if last_command_end is None or first_command_start is None:
                ct = 0.0
            else:
                ct = last_command_end - first_command_start
        scons_time = total_time - sconscript_time - ct
        print("Total build time: %f seconds"%total_time)
        print("Total SConscript file execution time: %f seconds"%sconscript_time)
        print("Total SCons execution time: %f seconds"%scons_time)
        print("Total command execution time: %f seconds"%ct)
        time_stats.total_times(total_time, sconscript_time, scons_time, ct)


    if ENABLE_JSON:
        write_scons_stats_file()

    sys.exit(exit_status)







