"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = exports.YamlOutputType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const constructs_1 = require("constructs");
const api_object_1 = require("./api-object");
const chart_1 = require("./chart");
const dependency_1 = require("./dependency");
const names_1 = require("./names");
const yaml_1 = require("./yaml");
/** The method to divide YAML output into files */
var YamlOutputType;
(function (YamlOutputType) {
    /** All resources are output into a single YAML file */
    YamlOutputType[YamlOutputType["FILE_PER_APP"] = 0] = "FILE_PER_APP";
    /** Resources are split into seperate files by chart */
    YamlOutputType[YamlOutputType["FILE_PER_CHART"] = 1] = "FILE_PER_CHART";
    /** Each resource is output to its own file */
    YamlOutputType[YamlOutputType["FILE_PER_RESOURCE"] = 2] = "FILE_PER_RESOURCE";
    /** Each chart in its own folder and each resource in its own file */
    YamlOutputType[YamlOutputType["FOLDER_PER_CHART_FILE_PER_RESOURCE"] = 3] = "FOLDER_PER_CHART_FILE_PER_RESOURCE";
})(YamlOutputType = exports.YamlOutputType || (exports.YamlOutputType = {}));
/**
 * Represents a cdk8s application.
 */
class App extends constructs_1.Construct {
    /**
     * Defines an app
     * @param props configuration options
     */
    constructor(props = {}) {
        super(undefined, '');
        this.outdir = props.outdir ?? process.env.CDK8S_OUTDIR ?? 'dist';
        this.outputFileExtension = props.outputFileExtension ?? '.k8s.yaml';
        this.yamlOutputType = props.yamlOutputType ?? YamlOutputType.FILE_PER_CHART;
    }
    /**
     * Synthesize a single chart.
     *
     * Each element returned in the resulting array represents a different ApiObject
     * in the scope of the chart.
     *
     * Note that the returned array order is important. It is determined by the various dependencies between
     * the constructs in the chart, where the first element is the one without dependencies, and so on...
     *
     * @returns An array of JSON objects.
     * @param chart the chart to synthesize.
     * @internal
     */
    static _synthChart(chart) {
        const app = App.of(chart);
        // we must prepare the entire app before synthesizing the chart
        // because the dependency inference happens on the app level.
        resolveDependencies(app);
        // validate the app since we want to call onValidate of the relevant constructs.
        // note this will also call onValidate on constructs from possibly different charts,
        // but thats ok too since we no longer treat constructs as a self-contained synthesis unit.
        validate(app);
        return chartToKube(chart).map(obj => obj.toJson());
    }
    static of(c) {
        const scope = constructs_1.Node.of(c).scope;
        if (!scope) {
            // the app is the only construct without a scope.
            return c;
        }
        return App.of(scope);
    }
    /**
     * Returns all the charts in this app, sorted topologically.
     */
    get charts() {
        const isChart = (x) => x instanceof chart_1.Chart;
        return new dependency_1.DependencyGraph(constructs_1.Node.of(this))
            .topology()
            .filter(isChart);
    }
    /**
     * Synthesizes all manifests to the output directory
     */
    synth() {
        fs.mkdirSync(this.outdir, { recursive: true });
        // Since we plan on removing the distributed synth mechanism, we no longer call `Node.synthesize`, but rather simply implement
        // the necessary operations. We do however want to preserve the distributed validation.
        validate(this);
        // this is kind of sucky, eventually I would like the DependencyGraph
        // to be able to answer this question.
        const hasDependantCharts = resolveDependencies(this);
        const charts = this.charts;
        switch (this.yamlOutputType) {
            case YamlOutputType.FILE_PER_APP:
                let apiObjectList = [];
                for (const chart of charts) {
                    apiObjectList.push(...chartToKube(chart));
                }
                if (charts.length > 0) {
                    yaml_1.Yaml.save(path.join(this.outdir, `app${this.outputFileExtension}`), // There is no "app name", so we just hardcode the file name
                    apiObjectList.map((apiObject) => apiObject.toJson()));
                }
                break;
            case YamlOutputType.FILE_PER_CHART:
                const namer = hasDependantCharts ? new IndexedChartNamer() : new SimpleChartNamer();
                for (const chart of charts) {
                    const chartName = namer.name(chart);
                    const objects = chartToKube(chart);
                    yaml_1.Yaml.save(path.join(this.outdir, chartName + this.outputFileExtension), objects.map(obj => obj.toJson()));
                }
                break;
            case YamlOutputType.FILE_PER_RESOURCE:
                for (const chart of charts) {
                    const apiObjects = chartToKube(chart);
                    apiObjects.forEach((apiObject) => {
                        if (!(apiObject === undefined)) {
                            const fileName = `${`${apiObject.kind}.${apiObject.metadata.name}`
                                .replace(/[^0-9a-zA-Z-_.]/g, '')}`;
                            yaml_1.Yaml.save(path.join(this.outdir, fileName + this.outputFileExtension), [apiObject.toJson()]);
                        }
                    });
                }
                break;
            case YamlOutputType.FOLDER_PER_CHART_FILE_PER_RESOURCE:
                const folderNamer = hasDependantCharts ? new IndexedChartFolderNamer() : new SimpleChartFolderNamer();
                for (const chart of charts) {
                    const chartName = folderNamer.name(chart);
                    const apiObjects = chartToKube(chart);
                    const fullOutDir = path.join(this.outdir, chartName);
                    fs.mkdirSync(fullOutDir, { recursive: true });
                    apiObjects.forEach((apiObject) => {
                        if (!(apiObject === undefined)) {
                            const fileName = `${`${apiObject.kind}.${apiObject.metadata.name}`
                                .replace(/[^0-9a-zA-Z-_.]/g, '')}`;
                            yaml_1.Yaml.save(path.join(fullOutDir, fileName + this.outputFileExtension), [apiObject.toJson()]);
                        }
                    });
                }
                break;
            default:
                break;
        }
    }
    /**
     * Synthesizes the app into a YAML string.
     *
     * @returns A string with all YAML objects across all charts in this app.
     */
    synthYaml() {
        validate(this);
        const charts = this.charts;
        const docs = [];
        for (const chart of charts) {
            const apiObjects = chartToKube(chart);
            docs.push(...apiObjects.map(apiObject => apiObject.toJson()));
        }
        return yaml_1.Yaml.stringify(...docs);
    }
}
exports.App = App;
_a = JSII_RTTI_SYMBOL_1;
App[_a] = { fqn: "cdk8s.App", version: "1.7.16" };
function validate(app) {
    // Note this is a copy-paste of https://github.com/aws/constructs/blob/master/lib/construct.ts#L438.
    const errors = constructs_1.Node.of(app).validate();
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${constructs_1.Node.of(e.source).path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
function resolveDependencies(app) {
    let hasDependantCharts = false;
    for (const dep of constructs_1.Node.of(app).dependencies) {
        // create explicit api object dependencies from implicit construct dependencies
        const targetApiObjects = constructs_1.Node.of(dep.target).findAll().filter(c => c instanceof api_object_1.ApiObject);
        const sourceApiObjects = constructs_1.Node.of(dep.source).findAll().filter(c => c instanceof api_object_1.ApiObject);
        for (const target of targetApiObjects) {
            for (const source of sourceApiObjects) {
                if (target !== source) {
                    constructs_1.Node.of(source).addDependency(target);
                }
            }
        }
        // create an explicit chart dependency from implicit construct dependencies
        const sourceChart = chart_1.Chart.of(dep.source);
        const targetChart = chart_1.Chart.of(dep.target);
        if (sourceChart !== targetChart) {
            constructs_1.Node.of(sourceChart).addDependency(targetChart);
            hasDependantCharts = true;
        }
    }
    const charts = new dependency_1.DependencyGraph(constructs_1.Node.of(app)).topology()
        .filter(x => x instanceof chart_1.Chart);
    for (const parentChart of charts) {
        for (const childChart of constructs_1.Node.of(parentChart).children.filter(x => x instanceof chart_1.Chart)) {
            // create an explicit chart dependency from nested chart relationships
            constructs_1.Node.of(parentChart).addDependency(childChart);
            hasDependantCharts = true;
        }
    }
    return hasDependantCharts;
}
function chartToKube(chart) {
    return new dependency_1.DependencyGraph(constructs_1.Node.of(chart)).topology()
        .filter(x => x instanceof api_object_1.ApiObject)
        .filter(x => chart_1.Chart.of(x) === chart) // include an object only in its closest parent chart
        .map(x => x);
}
class SimpleChartNamer {
    constructor() {
    }
    name(chart) {
        return `${names_1.Names.toDnsLabel(chart)}`;
    }
}
class IndexedChartNamer extends SimpleChartNamer {
    constructor() {
        super();
        this.index = 0;
    }
    name(chart) {
        const name = `${this.index.toString().padStart(4, '0')}-${super.name(chart)}`;
        this.index++;
        return name;
    }
}
class SimpleChartFolderNamer {
    constructor() {
    }
    name(chart) {
        return names_1.Names.toDnsLabel(chart);
    }
}
class IndexedChartFolderNamer extends SimpleChartFolderNamer {
    constructor() {
        super();
        this.index = 0;
    }
    name(chart) {
        const name = `${this.index.toString().padStart(4, '0')}-${super.name(chart)}`;
        this.index++;
        return name;
    }
}
//# sourceMappingURL=data:application/json;base64,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