"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonPatch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fast_json_patch_1 = require("fast-json-patch");
/**
 * Utility for applying RFC-6902 JSON-Patch to a document.
 *
 * Use the the `JsonPatch.apply(doc, ...ops)` function to apply a set of
 * operations to a JSON document and return the result.
 *
 * Operations can be created using the factory methods `JsonPatch.add()`,
 * `JsonPatch.remove()`, etc.
 *
 * @example
 *
 *const output = JsonPatch.apply(input,
 *  JsonPatch.replace('/world/hi/there', 'goodbye'),
 *  JsonPatch.add('/world/foo/', 'boom'),
 *  JsonPatch.remove('/hello'));
 *
 */
class JsonPatch {
    constructor(operation) {
        this.operation = operation;
    }
    /**
     * Applies a set of JSON-Patch (RFC-6902) operations to `document` and returns the result.
     * @param document The document to patch
     * @param ops The operations to apply
     * @returns The result document
     */
    static apply(document, ...ops) {
        const result = fast_json_patch_1.applyPatch(document, ops.map(o => o._toJson()));
        return result.newDocument;
    }
    /**
     * Adds a value to an object or inserts it into an array. In the case of an
     * array, the value is inserted before the given index. The - character can be
     * used instead of an index to insert at the end of an array.
     *
     * @example JsonPatch.add('/biscuits/1', { "name": "Ginger Nut" })
     */
    static add(path, value) { return new JsonPatch({ op: 'add', path, value }); }
    /**
     * Removes a value from an object or array.
     *
     * @example JsonPatch.remove('/biscuits')
     * @example JsonPatch.remove('/biscuits/0')
     */
    static remove(path) { return new JsonPatch({ op: 'remove', path }); }
    /**
     * Replaces a value. Equivalent to a “remove” followed by an “add”.
     *
     * @example JsonPatch.replace('/biscuits/0/name', 'Chocolate Digestive')
     */
    static replace(path, value) { return new JsonPatch({ op: 'replace', path, value }); }
    /**
     * Copies a value from one location to another within the JSON document. Both
     * from and path are JSON Pointers.
     *
     * @example JsonPatch.copy('/biscuits/0', '/best_biscuit')
     */
    static copy(from, path) { return new JsonPatch({ op: 'copy', from, path }); }
    /**
     * Moves a value from one location to the other. Both from and path are JSON Pointers.
     *
     * @example JsonPatch.move('/biscuits', '/cookies')
     */
    static move(from, path) { return new JsonPatch({ op: 'move', from, path }); }
    /**
     * Tests that the specified value is set in the document. If the test fails,
     * then the patch as a whole should not apply.
     *
     * @example JsonPatch.test('/best_biscuit/name', 'Choco Leibniz')
     */
    static test(path, value) { return new JsonPatch({ op: 'test', path, value }); }
    /**
     * Returns the JSON representation of this JSON patch operation.
     *
     * @internal
     */
    _toJson() {
        return this.operation;
    }
}
exports.JsonPatch = JsonPatch;
_a = JSII_RTTI_SYMBOL_1;
JsonPatch[_a] = { fqn: "cdk8s.JsonPatch", version: "1.7.16" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoianNvbi1wYXRjaC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9qc29uLXBhdGNoLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEscURBQXdEO0FBRXhEOzs7Ozs7Ozs7Ozs7Ozs7O0dBZ0JHO0FBQ0gsTUFBYSxTQUFTO0lBMkRwQixZQUFxQyxTQUFvQjtRQUFwQixjQUFTLEdBQVQsU0FBUyxDQUFXO0lBQUcsQ0FBQztJQTFEN0Q7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQWEsRUFBRSxHQUFHLEdBQWdCO1FBQ3BELE1BQU0sTUFBTSxHQUFHLDRCQUFVLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQy9ELE9BQU8sTUFBTSxDQUFDLFdBQVcsQ0FBQztJQUM1QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFZLEVBQUUsS0FBVSxJQUFJLE9BQU8sSUFBSSxTQUFTLENBQUMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUVqRzs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBWSxJQUFJLE9BQU8sSUFBSSxTQUFTLENBQUMsRUFBRSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRXBGOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQVksRUFBRSxLQUFVLElBQUksT0FBTyxJQUFJLFNBQVMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRXpHOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFZLEVBQUUsSUFBWSxJQUFJLE9BQU8sSUFBSSxTQUFTLENBQUMsRUFBRSxFQUFFLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUVwRzs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFZLEVBQUUsSUFBWSxJQUFJLE9BQU8sSUFBSSxTQUFTLENBQUMsRUFBRSxFQUFFLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUVwRzs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBWSxFQUFFLEtBQVUsSUFBSSxPQUFPLElBQUksU0FBUyxDQUFDLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFJbkc7Ozs7T0FJRztJQUNJLE9BQU87UUFDWixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDeEIsQ0FBQzs7QUFwRUgsOEJBcUVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgYXBwbHlQYXRjaCwgT3BlcmF0aW9uIH0gZnJvbSAnZmFzdC1qc29uLXBhdGNoJztcblxuLyoqXG4gKiBVdGlsaXR5IGZvciBhcHBseWluZyBSRkMtNjkwMiBKU09OLVBhdGNoIHRvIGEgZG9jdW1lbnQuXG4gKlxuICogVXNlIHRoZSB0aGUgYEpzb25QYXRjaC5hcHBseShkb2MsIC4uLm9wcylgIGZ1bmN0aW9uIHRvIGFwcGx5IGEgc2V0IG9mXG4gKiBvcGVyYXRpb25zIHRvIGEgSlNPTiBkb2N1bWVudCBhbmQgcmV0dXJuIHRoZSByZXN1bHQuXG4gKlxuICogT3BlcmF0aW9ucyBjYW4gYmUgY3JlYXRlZCB1c2luZyB0aGUgZmFjdG9yeSBtZXRob2RzIGBKc29uUGF0Y2guYWRkKClgLFxuICogYEpzb25QYXRjaC5yZW1vdmUoKWAsIGV0Yy5cbiAqXG4gKiBAZXhhbXBsZVxuICpcbiAqY29uc3Qgb3V0cHV0ID0gSnNvblBhdGNoLmFwcGx5KGlucHV0LFxuICogIEpzb25QYXRjaC5yZXBsYWNlKCcvd29ybGQvaGkvdGhlcmUnLCAnZ29vZGJ5ZScpLFxuICogIEpzb25QYXRjaC5hZGQoJy93b3JsZC9mb28vJywgJ2Jvb20nKSxcbiAqICBKc29uUGF0Y2gucmVtb3ZlKCcvaGVsbG8nKSk7XG4gKlxuICovXG5leHBvcnQgY2xhc3MgSnNvblBhdGNoIHtcbiAgLyoqXG4gICAqIEFwcGxpZXMgYSBzZXQgb2YgSlNPTi1QYXRjaCAoUkZDLTY5MDIpIG9wZXJhdGlvbnMgdG8gYGRvY3VtZW50YCBhbmQgcmV0dXJucyB0aGUgcmVzdWx0LlxuICAgKiBAcGFyYW0gZG9jdW1lbnQgVGhlIGRvY3VtZW50IHRvIHBhdGNoXG4gICAqIEBwYXJhbSBvcHMgVGhlIG9wZXJhdGlvbnMgdG8gYXBwbHlcbiAgICogQHJldHVybnMgVGhlIHJlc3VsdCBkb2N1bWVudFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhcHBseShkb2N1bWVudDogYW55LCAuLi5vcHM6IEpzb25QYXRjaFtdKTogYW55IHtcbiAgICBjb25zdCByZXN1bHQgPSBhcHBseVBhdGNoKGRvY3VtZW50LCBvcHMubWFwKG8gPT4gby5fdG9Kc29uKCkpKTtcbiAgICByZXR1cm4gcmVzdWx0Lm5ld0RvY3VtZW50O1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSB2YWx1ZSB0byBhbiBvYmplY3Qgb3IgaW5zZXJ0cyBpdCBpbnRvIGFuIGFycmF5LiBJbiB0aGUgY2FzZSBvZiBhblxuICAgKiBhcnJheSwgdGhlIHZhbHVlIGlzIGluc2VydGVkIGJlZm9yZSB0aGUgZ2l2ZW4gaW5kZXguIFRoZSAtIGNoYXJhY3RlciBjYW4gYmVcbiAgICogdXNlZCBpbnN0ZWFkIG9mIGFuIGluZGV4IHRvIGluc2VydCBhdCB0aGUgZW5kIG9mIGFuIGFycmF5LlxuICAgKlxuICAgKiBAZXhhbXBsZSBKc29uUGF0Y2guYWRkKCcvYmlzY3VpdHMvMScsIHsgXCJuYW1lXCI6IFwiR2luZ2VyIE51dFwiIH0pXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFkZChwYXRoOiBzdHJpbmcsIHZhbHVlOiBhbnkpIHsgcmV0dXJuIG5ldyBKc29uUGF0Y2goeyBvcDogJ2FkZCcsIHBhdGgsIHZhbHVlIH0pOyB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgYSB2YWx1ZSBmcm9tIGFuIG9iamVjdCBvciBhcnJheS5cbiAgICpcbiAgICogQGV4YW1wbGUgSnNvblBhdGNoLnJlbW92ZSgnL2Jpc2N1aXRzJylcbiAgICogQGV4YW1wbGUgSnNvblBhdGNoLnJlbW92ZSgnL2Jpc2N1aXRzLzAnKVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZW1vdmUocGF0aDogc3RyaW5nKSB7IHJldHVybiBuZXcgSnNvblBhdGNoKHsgb3A6ICdyZW1vdmUnLCBwYXRoIH0pOyB9XG5cbiAgLyoqXG4gICAqIFJlcGxhY2VzIGEgdmFsdWUuIEVxdWl2YWxlbnQgdG8gYSDigJxyZW1vdmXigJ0gZm9sbG93ZWQgYnkgYW4g4oCcYWRk4oCdLlxuICAgKlxuICAgKiBAZXhhbXBsZSBKc29uUGF0Y2gucmVwbGFjZSgnL2Jpc2N1aXRzLzAvbmFtZScsICdDaG9jb2xhdGUgRGlnZXN0aXZlJylcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVwbGFjZShwYXRoOiBzdHJpbmcsIHZhbHVlOiBhbnkpIHsgcmV0dXJuIG5ldyBKc29uUGF0Y2goeyBvcDogJ3JlcGxhY2UnLCBwYXRoLCB2YWx1ZSB9KTsgfVxuXG4gIC8qKlxuICAgKiBDb3BpZXMgYSB2YWx1ZSBmcm9tIG9uZSBsb2NhdGlvbiB0byBhbm90aGVyIHdpdGhpbiB0aGUgSlNPTiBkb2N1bWVudC4gQm90aFxuICAgKiBmcm9tIGFuZCBwYXRoIGFyZSBKU09OIFBvaW50ZXJzLlxuICAgKlxuICAgKiBAZXhhbXBsZSBKc29uUGF0Y2guY29weSgnL2Jpc2N1aXRzLzAnLCAnL2Jlc3RfYmlzY3VpdCcpXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGNvcHkoZnJvbTogc3RyaW5nLCBwYXRoOiBzdHJpbmcpIHsgcmV0dXJuIG5ldyBKc29uUGF0Y2goeyBvcDogJ2NvcHknLCBmcm9tLCBwYXRoIH0pOyB9XG5cbiAgLyoqXG4gICAqIE1vdmVzIGEgdmFsdWUgZnJvbSBvbmUgbG9jYXRpb24gdG8gdGhlIG90aGVyLiBCb3RoIGZyb20gYW5kIHBhdGggYXJlIEpTT04gUG9pbnRlcnMuXG4gICAqXG4gICAqIEBleGFtcGxlIEpzb25QYXRjaC5tb3ZlKCcvYmlzY3VpdHMnLCAnL2Nvb2tpZXMnKVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBtb3ZlKGZyb206IHN0cmluZywgcGF0aDogc3RyaW5nKSB7IHJldHVybiBuZXcgSnNvblBhdGNoKHsgb3A6ICdtb3ZlJywgZnJvbSwgcGF0aCB9KTsgfVxuXG4gIC8qKlxuICAgKiBUZXN0cyB0aGF0IHRoZSBzcGVjaWZpZWQgdmFsdWUgaXMgc2V0IGluIHRoZSBkb2N1bWVudC4gSWYgdGhlIHRlc3QgZmFpbHMsXG4gICAqIHRoZW4gdGhlIHBhdGNoIGFzIGEgd2hvbGUgc2hvdWxkIG5vdCBhcHBseS5cbiAgICpcbiAgICogQGV4YW1wbGUgSnNvblBhdGNoLnRlc3QoJy9iZXN0X2Jpc2N1aXQvbmFtZScsICdDaG9jbyBMZWlibml6JylcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgdGVzdChwYXRoOiBzdHJpbmcsIHZhbHVlOiBhbnkpIHsgcmV0dXJuIG5ldyBKc29uUGF0Y2goeyBvcDogJ3Rlc3QnLCBwYXRoLCB2YWx1ZSB9KTsgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBvcGVyYXRpb246IE9wZXJhdGlvbikge31cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgSlNPTiByZXByZXNlbnRhdGlvbiBvZiB0aGlzIEpTT04gcGF0Y2ggb3BlcmF0aW9uLlxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9Kc29uKCk6IGFueSB7XG4gICAgcmV0dXJuIHRoaXMub3BlcmF0aW9uO1xuICB9XG59Il19