"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Yaml = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const YAML = require("yaml");
const MAX_DOWNLOAD_BUFFER = 10 * 1024 * 1024;
// Set default YAML schema to 1.1. This ensures saved YAML is backward compatible with other parsers, such as PyYAML
// It also ensures that octal numbers in the form `0775` will be parsed
// correctly on YAML load. (see https://github.com/eemeli/yaml/issues/205)
YAML.defaultOptions.version = '1.1';
/**
 * YAML utilities.
 */
class Yaml {
    /**
     * Utility class.
     */
    constructor() {
        return;
    }
    /**
     * @deprecated use `stringify(doc[, doc, ...])`
     */
    static formatObjects(docs) {
        return this.stringify(...docs);
    }
    /**
     * Saves a set of objects as a multi-document YAML file.
     * @param filePath The output path
     * @param docs The set of objects
     */
    static save(filePath, docs) {
        const data = this.stringify(...docs);
        fs.writeFileSync(filePath, data, { encoding: 'utf8' });
    }
    /**
     * Stringify a document (or multiple documents) into YAML
     *
     * We convert undefined values to null, but ignore any documents that are
     * undefined.
     *
     * @param docs A set of objects to convert to YAML
     * @returns a YAML string. Multiple docs are separated by `---`.
     */
    static stringify(...docs) {
        return docs.map(r => r === undefined ? '\n' : YAML.stringify(r, { keepUndefined: true, lineWidth: 0 })).join('---\n');
    }
    /**
     * Saves a set of YAML documents into a temp file (in /tmp)
     *
     * @returns the path to the temporary file
     * @param docs the set of documents to save
     */
    static tmp(docs) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk8s-'));
        const filePath = path.join(tmpdir, 'temp.yaml');
        Yaml.save(filePath, docs);
        return filePath;
    }
    /**
     * Downloads a set of YAML documents (k8s manifest for example) from a URL or
     * a file and returns them as javascript objects.
     *
     * Empty documents are filtered out.
     *
     * @param urlOrFile a URL of a file path to load from
     * @returns an array of objects, each represents a document inside the YAML
     */
    static load(urlOrFile) {
        const body = loadurl(urlOrFile);
        const objects = YAML.parseAllDocuments(body);
        const result = new Array();
        for (const obj of objects.map(x => x.toJSON())) {
            // skip empty documents
            if (obj === undefined) {
                continue;
            }
            if (obj === null) {
                continue;
            }
            if (Array.isArray(obj) && obj.length === 0) {
                continue;
            }
            if (typeof (obj) === 'object' && Object.keys(obj).length === 0) {
                continue;
            }
            result.push(obj);
        }
        return result;
    }
}
exports.Yaml = Yaml;
_a = JSII_RTTI_SYMBOL_1;
Yaml[_a] = { fqn: "cdk8s.Yaml", version: "1.7.16" };
/**
 * Loads a url (or file) and returns the contents.
 * This method spawns a child process in order to perform an http call synchronously.
 */
function loadurl(url) {
    const script = path.join(__dirname, '_loadurl.js');
    return child_process_1.execFileSync(process.execPath, [script, url], {
        encoding: 'utf-8',
        maxBuffer: MAX_DOWNLOAD_BUFFER,
    }).toString();
}
//# sourceMappingURL=data:application/json;base64,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