# pylint: disable=import-error
import uuid
from datetime import datetime
from pymongo import MongoClient
import os
from dotenv import load_dotenv
from bson import json_util, ObjectId
import json
import re
from math import ceil
from flask import g


class Database:
    _instance = None
    _db = None
    _collections = {}
    _uri = None
    _db_name = None

    @classmethod
    def initialize(cls):
        """Initialize the database connection using environment variables."""
        if cls._instance is None:
            cls._instance = cls()
            load_dotenv()  # Load environment variables
            cls._uri = os.getenv('MONGO')
            cls._db_name = os.getenv('DB_NAME')

            if not cls._uri:
                raise Exception("MongoDB URI is not set in environment variables")

            client = MongoClient(cls._uri)
            cls._db = client[cls._db_name] if cls._db_name else client.get_default_database()

            if cls._db is None:
                raise Exception("Failed to connect to MongoDB")
            print("Database connection initialized successfully.")

    @classmethod
    def get_collection(cls, collection_name):
        """Get a collection, initializing the database if necessary."""
        if cls._db is None:
            cls.initialize()  # Lazily initialize the DB on first collection access

        if collection_name not in cls._collections:
            cls._collections[collection_name] = cls._db[collection_name]

        return cls._collections[collection_name]


# Expose db instance
db = Database()


def docs(data):
    """Convert MongoDB documents to JSON-serializable format."""
    parsed_data = json.loads(json_util.dumps(data))
    return parsed_data


class BaseSchema:
    def __init__(self, collection):
        self.collection = collection
        self.create = self.Create(self)
        self.read = self.Read(self)
        self.update = self.Update(self)
        self.delete = self.Delete(self)
        self.custom = {}
    
    class Create:
        def __init__(self, parent):
            self.parent = parent
            self.collection = parent.collection

        def one(self, document, apply_user_filter=True):
            try:
                idd = str(uuid.uuid4())
                document['id'] = idd
                document['date_updated'] = datetime.utcnow()
                document['date_created'] = datetime.utcnow()
                
                # Add user_id from request context if available and filter is applied
                if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                    document['user_id'] = g.user_id

                self.collection.insert_one(document)

                # Return the document directly instead of using docs()
                return document
            except Exception as e:
                print(f"BASE MODEL CREATE ONE ERROR: {str(e)}")
                import traceback
                traceback.print_exc()
                return None

        def batch(self, documents, apply_user_filter=True):
            # Add user_id and UUID to each document
            documents_with_id = []
            for doc in documents:
                doc_with_id = {**doc, 'id': str(uuid.uuid4())}
                if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                    doc_with_id['user_id'] = g.user_id
                documents_with_id.append(doc_with_id)
                
            self.collection.insert_many(documents_with_id)
            return docs(documents_with_id)

    class Read:
        def __init__(self, parent):
            self.parent = parent
            self.collection = parent.collection

        def al(self, query={}, projection=None, sort=None, apply_user_filter=True):
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
                
            data = self.collection.find(query, projection)
            if sort:
                data = data.sort(sort)
            return docs(data)

        def al_id(self, query={}, projection=None, sort=None, apply_user_filter=True):
            """Get all documents with string IDs
            
            Args:
                query (dict): MongoDB query filter
                projection (dict, optional): Fields to include in the results
                sort (dict, optional): Sort criteria, e.g. {'field': 1} for ascending, {'field': -1} for descending
                apply_user_filter (bool, optional): Whether to filter by user_id, defaults to True
            """
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
            
            pipeline = [
                {
                    '$match': query
                },
                {
                    # First convert _id to string
                    '$addFields': {
                        '_id': { '$toString': '$_id' }
                    }
                }
            ]
            
            # Add sort if specified
            if sort:
                pipeline.append({
                    '$sort': sort
                })
            
            # Add projection after _id conversion if specified
            if projection:
                pipeline.append({
                    '$project': {
                        '_id': 1,  # Keep the already converted string _id
                        **projection
                    }
                })

            data = self.collection.aggregate(pipeline)
            return docs(data)

        def one(self, query, projection=None, apply_user_filter=True):
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
                
            data = self.collection.find_one(query, projection)
            return docs(data)

        def one_id(self, id_string, projection=None, apply_user_filter=True):
            """Get single document by string ID with ObjectId conversion"""
            try:
                # For one_id, we need to add user_id to the pipeline if requested
                user_filter = {}
                if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                    user_filter = {'user_id': g.user_id}
                    
                pipeline = [
                    {
                        '$match': {
                            '_id': ObjectId(id_string),
                            **user_filter
                        }
                    }
                ]

                # Only add projection stage if needed
                if projection:
                    project_stage = {
                        '_id': { '$toString': '$_id' }
                    }
                    if isinstance(projection, list):
                        for field in projection:
                            project_stage[field] = 1
                    elif isinstance(projection, dict):
                        project_stage.update(projection)
                        
                    pipeline.append({
                        '$project': project_stage
                    })
                else:
                    # Just convert _id to string, keep everything else as is
                    pipeline.append({
                        '$addFields': {
                            '_id': { '$toString': '$_id' }
                        }
                    })

                data = list(self.collection.aggregate(pipeline))
                return docs(data[0] if data else None)
            except Exception as e:
                print(f"Error in one_id: {str(e)}")
                return None

        def path(self, path, projection=None, apply_user_filter=True):
            # Apply user_id filter to path query if requested
            query = {'path': {'$regex': path}}
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query['user_id'] = g.user_id
            data = self.collection.find(query, projection)
            return docs(data)

        def paginate(self, query=None, page=1, limit=10, sort_field='created_at', sort_order=-1,
                     search_field=None, search=None, select=None, apply_user_filter=True):
            if query is None:
                query = {}
                
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query['user_id'] = g.user_id

            # Apply search filter if provided
            if search and search_field:
                query[search_field] = {'$regex': search, '$options': 'i'}

            # Get total count for pagination
            total_docs = self.collection.count_documents(query)
            total_pages = -(-total_docs // limit)
            skip = (page - 1) * limit

            cursor = self.collection.find(query)
            cursor.sort(sort_field, sort_order)
            cursor.skip(skip).limit(limit)

            if select:
                cursor.project(dict.fromkeys(select, 1))
            
            results = cursor.to_list(length=None)

            has_next_page = page < total_pages
            has_prev_page = page > 1

            return {
                'docs': results,
                'page': page,
                'hasNextPage': has_next_page,
                'hasPrevPage': has_prev_page,
                'nextPage': page + 1 if has_next_page else None,
                'prevPage': page - 1 if has_prev_page else None,
                'pagingCounter': skip + 1,
                'limit': limit,
                'totalDocs': total_docs,
                'totalPages': total_pages
            }

    class Update:
        def __init__(self, parent):
            self.parent = parent
            self.collection = parent.collection

        def one(self, query, update, apply_user_filter=True):
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
                
            update['date_updated'] = datetime.utcnow()
            result = self.collection.update_one(query, {'$set': update})
            return result.modified_count > 0

        def many(self, query, update, apply_user_filter=True):
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
                
            update['date_updated'] = datetime.utcnow()
            result = self.collection.update_many(query, {'$set': update})
            return result.modified_count > 0

    class Delete:
        def __init__(self, parent):
            self.parent = parent
            self.collection = parent.collection

        def one(self, query, apply_user_filter=True):
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
                
            result = self.collection.delete_one(query)
            return result.deleted_count > 0

        def many(self, query, apply_user_filter=True):
            # Apply user_id filter if requested
            if apply_user_filter and hasattr(g, 'user_id') and g.user_id:
                query = {**query, 'user_id': g.user_id}
                
            result = self.collection.delete_many(query)
            return result.deleted_count > 0



async def paginate_model(ModelToPaginate, query, page=1, limit=10, sort_field="createdAt", sort_order=-1, search_field=None, search=None, select=None, distinct=None):
    search_query = query
    
    if search:
        search_query = {
            "$and": [
                {"$or": [{search_field: {"$regex": re.compile(search, re.IGNORECASE)}}]},
                query,
            ],
        }
    
    if distinct:
        distinct_values = ModelToPaginate.distinct(distinct, query)
        search_query = {distinct: {"$in": distinct_values}, **query}
    
    total_docs = ModelToPaginate.count_documents(search_query)
    total_pages = ceil(total_docs / limit)
    skip = (page - 1) * limit
    sort = {sort_field: sort_order} if sort_field else None
    
    if select:
        results = ModelToPaginate.find(search_query).select(select).sort(sort).skip(skip).limit(limit)
    else:
        results = ModelToPaginate.find(search_query).sort(sort).skip(skip).limit(limit)
    
    has_next_page = page < total_pages
    has_prev_page = page > 1
    
    return {
        "docs": results,
        "page": page,
        "hasNextPage": has_next_page,
        "hasPrevPage": has_prev_page,
        "nextPage": page + 1 if has_next_page else None,
        "prevPage": page - 1 if has_prev_page else None,
        "pagingCounter": skip + 1,
        "limit": limit,
        "totalDocs": total_docs,
        "totalPages": total_pages,
    }

def paginate_array(array_to_paginate, page=1, limit=10, sort_field="createdAt", sort_order=-1, search_field=None, search=None):
    filtered_data = array_to_paginate
    if search and search_field:
        filtered_data = [item for item in array_to_paginate if search_field in item and search.lower() in item[search_field].lower()]
    
    filtered_data.sort(key=lambda x: sort_order * (x[sort_field] > x[sort_field] if sort_field in x else 0))
    
    total_docs = len(filtered_data)
    total_pages = ceil(total_docs / limit)
    start_index = (page - 1) * limit
    end_index = start_index + limit
    paginated_items = filtered_data[start_index:end_index]
    
    return {
        "docs": paginated_items,
        "page": page,
        "limit": limit,
        "totalDocs": total_docs,
        "totalPages": total_pages,
        "hasNextPage": page < total_pages,
        "hasPrevPage": page > 1,
        "nextPage": page + 1 if page < total_pages else None,
        "prevPage": page - 1 if page > 1 else None,
        "pagingCounter": start_index + 1,
    }
