"""Collection of functions and classes designed to load data."""

import numpy as np
import pandas as pd


def load_w2v_format(fname):
    """Load data from non-binary word2vec format file without need for
    gensim.

    The returned numpy.array `syn0` corresponds to `syn0` in the
    gensim word2vec model.

    Parameters
    ----------
    fname: string
      Filename where the w2v model is stored.

    Returns
    -------
    word2idx, dict{str: int}
      A dictionary to look up the row number of the
      representation vector for a given word.

    syn0: numpy.array 2d
      Contains the vector represantation for words in every row. The
      row number for a given word can be found with `word2idx`.

    """
    header = pd.read_csv(fname, sep=" ", nrows=1, header=None).values[0]
    num_features = header.astype(int).tolist()[1]

    w2v_data = pd.read_csv(fname, sep=" ", skiprows=1, header=None)
    word2idx = {word: idx for idx, word in enumerate(w2v_data[0])}
    syn0 = np.array(w2v_data[np.arange(1, num_features + 1)])

    return word2idx, syn0


def load_w2v_vocab(fname):
    """Load vocabulary file generated by gensim as a dictionary.

    Note that this does not correspond to the gensim vocabulary
    object.

    Parameters
    ----------
    fname: string
      Filename where the w2v model is stored.

    Returns
    -------
    vocab: dict{str: int}
      A dictionary containing the words as keys and their counts as
      values.

    """
    lst = []
    with open(fname, 'r') as f:
        for line in f:
            word, count = line.split()
            lst.append((word, int(count)))
    vocab = dict(lst)
    return vocab
