"""
Test for the ES futures infinite restart/hang bug during backtesting.

BUG DESCRIPTION:
The ES futures strategy was hanging/restarting infinitely during backtesting.

ROOT CAUSE:
strategy_executor.py wasn't advancing broker datetime to next trading day 
after market close for non-24/7 markets like "us_futures".

FIX:
Added datetime advancement logic for non-24/7 markets in strategy_executor.py.

STATUS: ✅ FIXED - ES futures now complete normally (1 restart vs infinite)
"""

import unittest
from unittest.mock import patch
from datetime import datetime, timedelta

import pandas as pd

from lumibot.credentials import DATABENTO_CONFIG
from lumibot.strategies import Strategy
from lumibot.entities import Asset, TradingFee, Bars, Order
from lumibot.backtesting import BacktestingBroker, DataBentoDataBacktesting

DATABENTO_API_KEY = DATABENTO_CONFIG.get("API_KEY")


class ESFuturesTestStrategy(Strategy):
    """Simple ES futures strategy to test the hang bug fix"""
    
    def initialize(self):
        self.asset = Asset("ES", asset_type=Asset.AssetType.CONT_FUTURE)
        self.set_market("us_futures")
        self.sleeptime = "1M"
        
    def on_trading_iteration(self):
        pass


class TestESFuturesHangBug(unittest.TestCase):
    """Test that ES futures strategies no longer hang/restart infinitely"""
    
    def setUp(self):
        if not DATABENTO_API_KEY or DATABENTO_API_KEY == "<your key here>":
            self.skipTest("DataBento API key required for DataBento backtesting tests")
        self.backtesting_params = {
            'datasource_class': DataBentoDataBacktesting,
            'backtesting_start': datetime(2025, 6, 5),
            'backtesting_end': datetime(2025, 6, 6),
            'api_key': DATABENTO_API_KEY,
            'show_plot': False,
            'show_tearsheet': False,
            'show_indicators': False,
            'save_tearsheet': False,
            'save_logfile': False
        }

    def test_es_futures_no_infinite_restart(self):
        """
        MAIN TEST: Verify ES futures strategies don't restart infinitely.
        
        Before fix: Would restart 100s-1000s of times (infinite loop)
        After fix: Should restart only 1-2 times (normal behavior)
        """
        restart_count = 0
        original_method = None
        
        def count_restarts(self):
            nonlocal restart_count, original_method
            restart_count += 1
            
            # Fail if infinite restart detected
            if restart_count > 5:
                raise AssertionError(f"INFINITE RESTART BUG DETECTED: {restart_count} restarts")
            
            # Call the original method to maintain proper execution flow
            if original_method:
                return original_method(self)
            return None
        
        from lumibot.strategies.strategy_executor import StrategyExecutor
        
        # Store the original method before patching
        original_method = StrategyExecutor._run_trading_session
        
        with patch.object(StrategyExecutor, '_run_trading_session', count_restarts):
            
            try:
                ESFuturesTestStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **self.backtesting_params)
            except Exception as e:
                if "INFINITE RESTART BUG DETECTED" in str(e):
                    raise
                # Ignore other errors (like visualization issues)
        
        # Assert fix is working
        self.assertLessEqual(
            restart_count, 
            3, 
            f"ES futures should restart ≤3 times but had {restart_count} (infinite loop?)"
        )
        
        print(f"✅ ES futures test PASSED: {restart_count} restart(s) - no infinite loop")

    def test_different_sleeptime_combinations(self):
        """Test various sleeptime values don't cause infinite loops"""
        sleeptimes = ["1S", "30S", "1M", "5M", "15M", "1H"]  # Skip 1D to keep tests fast
        
        for sleeptime in sleeptimes:
            with self.subTest(sleeptime=sleeptime):
                print(f"\n🕐 Testing sleeptime: {sleeptime}")
                
                class TestSleeptimeStrategy(Strategy):
                    def initialize(self):
                        self.asset = Asset("ES", asset_type=Asset.AssetType.CONT_FUTURE)
                        self.set_market("us_futures")
                        self.sleeptime = sleeptime
                        self.iteration_count = 0
                        
                    def on_trading_iteration(self):
                        self.iteration_count += 1
                        # Stop after 3 iterations to keep tests fast
                        if self.iteration_count >= 3:
                            self._executor.stop_event.set()
                
                try:
                    result = TestSleeptimeStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **self.backtesting_params)
                    self.assertIsNotNone(result)
                    print(f"✅ Sleeptime {sleeptime}: Success")
                                
                except Exception as e:
                    # If there's an infinite restart bug, it would hang or restart many times
                    if "infinite" in str(e).lower() or "restart" in str(e).lower():
                        self.fail(f"Sleeptime {sleeptime} caused infinite restart: {e}")
                    # Other errors (like data issues) are acceptable for this test
                    print(f"⚠️  Sleeptime {sleeptime}: {e} (non-critical)")

    def test_different_market_types(self):
        """Test various market types work correctly"""
        market_configs = [
            ("us_futures", "ES", Asset.AssetType.CONT_FUTURE, DataBentoDataBacktesting),
        ]
        
        for market, symbol, asset_type, datasource in market_configs:
            with self.subTest(market=market, symbol=symbol):
                print(f"\n🏪 Testing market: {market} with {symbol}")
                
                class TestMarketStrategy(Strategy):
                    def initialize(self):
                        self.asset = Asset(symbol, asset_type=asset_type)
                        self.set_market(market)
                        self.sleeptime = "1M"
                        self.iteration_count = 0
                        
                    def on_trading_iteration(self):
                        self.iteration_count += 1
                        # Stop after 3 iterations
                        if self.iteration_count >= 3:
                            self._executor.stop_event.set()
                
                try:
                    params = dict(self.backtesting_params)
                    params['datasource_class'] = datasource
                    result = TestMarketStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **params)
                    self.assertIsNotNone(result)
                    print(f"✅ Market {market}/{symbol}: Success")
                            
                except Exception as e:
                    if "infinite" in str(e).lower() or "restart" in str(e).lower():
                        self.fail(f"Market {market}/{symbol} caused infinite restart: {e}")
                    # Skip if no data available for that symbol
                    if "No data" in str(e) or "not found" in str(e):
                        print(f"⚠️  Skipping {market}/{symbol} - no data available")
                        continue
                    print(f"⚠️  Market {market}/{symbol}: {e} (non-critical)")

    def test_continuous_vs_non_continuous_markets(self):
        """Test that continuous and non-continuous markets behave differently"""
        
        class ContinuousMarketStrategy(Strategy):
            def initialize(self):
                self.asset = Asset("ES", asset_type=Asset.AssetType.CONT_FUTURE)
                self.set_market("us_futures")  # This should be treated as continuous
                self.sleeptime = "1M"
                self.iteration_count = 0
                
            def on_trading_iteration(self):
                self.iteration_count += 1
                if self.iteration_count >= 3:
                    self._executor.stop_event.set()
        
        # Test continuous market (futures)
        print("\n🔄 Testing continuous market (futures)")
        try:
            result = ContinuousMarketStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **self.backtesting_params)
            self.assertIsNotNone(result)
            print("✅ Continuous market test completed")
        except Exception as e:
            if "infinite" in str(e).lower() or "restart" in str(e).lower():
                self.fail(f"Continuous market caused infinite restart: {e}")
            print(f"⚠️  Continuous market: {e} (acceptable for this test)")
        
        print("✅ Market behavior tests completed")

    def test_comprehensive_diagnostic_scenarios(self):
        """Comprehensive diagnostic test covering multiple scenarios"""
        print("\n🔍 Running comprehensive diagnostic scenarios")
        
        # Test 1: Different sleeptime formats
        sleeptime_formats = ["1S", "30S", "1M", "5M", "15M", "1H"]
        
        for sleeptime in sleeptime_formats:
            with self.subTest(test="sleeptime_formats", sleeptime=sleeptime):
                print(f"🕐 Testing sleeptime format: {sleeptime}")
                
                class DiagnosticSleeptimeStrategy(Strategy):
                    def initialize(self):
                        self.asset = Asset("ES", asset_type=Asset.AssetType.CONT_FUTURE)
                        self.set_market("us_futures")
                        self.sleeptime = sleeptime
                        self.iteration_count = 0
                        
                    def on_trading_iteration(self):
                        self.iteration_count += 1
                        # Stop early to avoid long test times
                        if self.iteration_count >= 3:
                            self._executor.stop_event.set()
                            return
                
                try:
                    result = DiagnosticSleeptimeStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **self.backtesting_params)
                    self.assertIsNotNone(result)
                    print(f"✅ Sleeptime {sleeptime}: Success")
                except Exception as e:
                    if "infinite" in str(e).lower() or "restart" in str(e).lower():
                        self.fail(f"Sleeptime {sleeptime} caused infinite restart: {e}")
                    print(f"⚠️  Sleeptime {sleeptime}: {e} (non-critical)")

        # Test 2: Different asset types
        asset_configs = [
            ("ES", Asset.AssetType.CONT_FUTURE, "ES Continuous Future"),
            ("ESM24", Asset.AssetType.FUTURE, "ES June 2024 Future"),
        ]
        
        for symbol, asset_type, description in asset_configs:
            with self.subTest(test="asset_types", symbol=symbol, asset_type=asset_type):
                print(f"📈 Testing asset: {description}")
                
                class DiagnosticAssetStrategy(Strategy):
                    def initialize(self):
                        self.asset = Asset(symbol, asset_type=asset_type)
                        self.set_market("us_futures")
                        self.sleeptime = "1M"
                        self.iteration_count = 0
                        
                    def on_trading_iteration(self):
                        self.iteration_count += 1
                        # Stop early
                        if self.iteration_count >= 3:
                            self._executor.stop_event.set()
                            return
                
                try:
                    result = DiagnosticAssetStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **self.backtesting_params)
                    self.assertIsNotNone(result)
                    print(f"✅ Asset {symbol}: Success")
                except Exception as e:
                    if "infinite" in str(e).lower() or "restart" in str(e).lower():
                        self.fail(f"Asset {symbol} caused infinite restart: {e}")
                    print(f"⚠️  Asset {symbol}: {e} (acceptable)")

        # Test 3: Longer backtest period (stress test)
        print("📅 Testing longer backtest period (stress test)")
        
        class StressTestStrategy(Strategy):
            def initialize(self):
                self.asset = Asset("ES", asset_type=Asset.AssetType.CONT_FUTURE)
                self.set_market("us_futures")
                self.sleeptime = "15M"  # Use 15 minute intervals for faster execution
                self.iteration_count = 0
                
            def on_trading_iteration(self):
                self.iteration_count += 1
                # Stop after reasonable number of iterations
                if self.iteration_count >= 20:  # Reduced from 50 to keep tests fast
                    self._executor.stop_event.set()
                    return
        
        try:
            # Use a longer period for stress testing
            stress_params = dict(self.backtesting_params)
            stress_params['backtesting_end'] = datetime(2025, 6, 9)  # 4 days
            
            result = StressTestStrategy.backtest(quote_asset=Asset("USD", Asset.AssetType.FOREX), **stress_params)
            self.assertIsNotNone(result)
            print("✅ Stress test: Success")
        except Exception as e:
            if "infinite" in str(e).lower() or "restart" in str(e).lower():
                self.fail(f"Stress test caused infinite restart: {e}")
            print(f"⚠️  Stress test: {e} (acceptable)")
        
        print("✅ All diagnostic scenarios completed")


if __name__ == '__main__':
    print("🧪 Testing ES Futures hang bug fix...")
    unittest.main(verbosity=2)


def test_broker_timeshift_guard():
    captured = []

    class StubDataSource:
        SOURCE = "DATABENTO_POLARS"
        IS_BACKTESTING_DATA_SOURCE = True

        def __init__(self):
            self._datetime = datetime(2025, 6, 5, 14, 30)

        def get_historical_prices(self, asset, length, quote=None, timeshift=None, **kwargs):
            captured.append(timeshift)
            index = pd.DatetimeIndex([self._datetime - timedelta(minutes=1)])
            frame = pd.DataFrame(
                {
                    'open': [4300.0],
                    'high': [4301.0],
                    'low': [4299.5],
                    'close': [4300.5],
                    'volume': [1500],
                },
                index=index,
            )
            target_asset = asset[0] if isinstance(asset, tuple) else asset
            return Bars(frame, self.SOURCE, target_asset, raw=frame)

        def get_datetime(self):
            return self._datetime

    broker = BacktestingBroker(data_source=StubDataSource())
    broker._datetime = broker.data_source.get_datetime()

    order = Order(
        strategy="stub",
        asset=Asset("MES", asset_type=Asset.AssetType.CONT_FUTURE),
        quantity=1,
        side=Order.OrderSide.BUY,
    )
    order.order_type = Order.OrderType.MARKET
    order.quote = Asset("USD", asset_type=Asset.AssetType.FOREX)
    broker._new_orders.append(order)

    class StubStrategy:
        name = "stub"
        buy_trading_fees = []
        sell_trading_fees = []
        timestep = 'minute'
        bars_lookback = 1

        def __init__(self, broker):
            self.broker = broker
            self.cash = 100000.0
            self.quote_asset = Asset('USD', asset_type=Asset.AssetType.FOREX)

        def log_message(self, *args, **kwargs):
            return None

        def _set_cash_position(self, value):
            self.cash = value

    broker.process_pending_orders(strategy=StubStrategy(broker))

    assert captured, "BacktestingBroker did not request historical data"
    assert captured[0] == timedelta(minutes=-2)
