import { FC, memo, RefObject } from "react";
import {
  ApprovalEvent,
  ErrorEvent,
  InfoEvent,
  InputEvent,
  LoggerEvent,
  ModelEvent,
  SampleInitEvent,
  SampleLimitEvent,
  SandboxEvent,
  ScoreEditEvent,
  ScoreEvent,
  SpanBeginEvent,
  StateEvent,
  StepEvent,
  StoreEvent,
  SubtaskEvent,
  ToolEvent,
} from "../../../@types/log";
import { ApprovalEventView } from "./ApprovalEventView";
import { ErrorEventView } from "./ErrorEventView";
import { InfoEventView } from "./InfoEventView";
import { InputEventView } from "./InputEventView";
import { LoggerEventView } from "./LoggerEventView";
import { ModelEventView } from "./ModelEventView";
import { SampleInitEventView } from "./SampleInitEventView";
import { SampleLimitEventView } from "./SampleLimitEventView";
import { SandboxEventView } from "./SandboxEventView";
import { ScoreEventView } from "./ScoreEventView";
import { StateEventView } from "./state/StateEventView";
import { StepEventView } from "./StepEventView";
import { SubtaskEventView } from "./SubtaskEventView";
import { ToolEventView } from "./ToolEventView";
import { EventNode } from "./types";

import { VirtuosoHandle } from "react-virtuoso";
import { ScoreEditEventView } from "./ScoreEditEventView";
import { SpanEventView } from "./SpanEventView";
import { TranscriptVirtualListComponent } from "./TranscriptVirtualListComponent";

interface TranscriptVirtualListProps {
  id: string;
  eventNodes: EventNode[];
  listHandle: RefObject<VirtuosoHandle | null>;
  initialEventId: string | null;
  offsetTop?: number;
  scrollRef?: RefObject<HTMLDivElement | null>;
  running?: boolean;
  className?: string | string[];
}

/**
 * Renders the Transcript Virtual List.
 */
export const TranscriptVirtualList: FC<TranscriptVirtualListProps> = memo(
  (props) => {
    let {
      id,
      scrollRef,
      eventNodes,
      listHandle,
      running,
      initialEventId,
      offsetTop,
      className,
    } = props;

    return (
      <TranscriptVirtualListComponent
        id={id}
        listHandle={listHandle}
        eventNodes={eventNodes}
        initialEventId={initialEventId}
        offsetTop={offsetTop}
        scrollRef={scrollRef}
        running={running}
        className={className}
      />
    );
  },
);

export interface EventNodeContext {
  hasToolEvents?: boolean;
}

interface RenderedEventNodeProps {
  node: EventNode;
  next?: EventNode;
  className?: string | string[];
  context?: EventNodeContext;
}
/**
 * Renders the event based on its type.
 */
export const RenderedEventNode: FC<RenderedEventNodeProps> = memo(
  ({ node, next, className, context }) => {
    switch (node.event.event) {
      case "sample_init":
        return (
          <SampleInitEventView
            eventNode={node as EventNode<SampleInitEvent>}
            className={className}
          />
        );

      case "sample_limit":
        return (
          <SampleLimitEventView
            eventNode={node as EventNode<SampleLimitEvent>}
            className={className}
          />
        );

      case "info":
        return (
          <InfoEventView
            eventNode={node as EventNode<InfoEvent>}
            className={className}
          />
        );

      case "logger":
        return (
          <LoggerEventView
            eventNode={node as EventNode<LoggerEvent>}
            className={className}
          />
        );

      case "model":
        return (
          <ModelEventView
            eventNode={node as EventNode<ModelEvent>}
            showToolCalls={next?.event.event !== "tool"}
            className={className}
            context={context}
          />
        );

      case "score":
        return (
          <ScoreEventView
            eventNode={node as EventNode<ScoreEvent>}
            className={className}
          />
        );

      case "score_edit":
        return (
          <ScoreEditEventView
            eventNode={node as EventNode<ScoreEditEvent>}
            className={className}
          />
        );

      case "state":
        return (
          <StateEventView
            eventNode={node as EventNode<StateEvent>}
            className={className}
          />
        );

      case "span_begin":
        return (
          <SpanEventView
            eventNode={node as EventNode<SpanBeginEvent>}
            children={node.children}
            className={className}
          />
        );

      case "step":
        return (
          <StepEventView
            eventNode={node as EventNode<StepEvent>}
            children={node.children}
            className={className}
          />
        );

      case "store":
        return (
          <StateEventView
            eventNode={node as EventNode<StoreEvent>}
            className={className}
          />
        );

      case "subtask":
        return (
          <SubtaskEventView
            eventNode={node as EventNode<SubtaskEvent>}
            className={className}
            children={node.children}
          />
        );

      case "tool":
        return (
          <ToolEventView
            eventNode={node as EventNode<ToolEvent>}
            className={className}
            children={node.children}
          />
        );

      case "input":
        return (
          <InputEventView
            eventNode={node as EventNode<InputEvent>}
            className={className}
          />
        );

      case "error":
        return (
          <ErrorEventView
            eventNode={node as EventNode<ErrorEvent>}
            className={className}
          />
        );

      case "approval":
        return (
          <ApprovalEventView
            eventNode={node as EventNode<ApprovalEvent>}
            className={className}
          />
        );

      case "sandbox":
        return (
          <SandboxEventView
            eventNode={node as EventNode<SandboxEvent>}
            className={className}
          />
        );

      default:
        return null;
    }
  },
);
