# devibe

Remove auto-generated comments from your codebase. A simple, extensible library for cleaning up AI-generated and IDE-generated comments.

## Installation

### PyPI (Coming Soon!)

🚀 **PyPI package coming soon!** We're preparing to publish devibe to the Python Package Index. Once available, you'll be able to install it with:

```bash
# Coming soon!
pip install devibe
```

Once published, you'll be able to run `devibe ./myfile.py` directly from anywhere.

### Local Development Install

Clone the repository and install in development mode:

```bash
git clone https://github.com/nhunter0/devibe.git
cd devibe
pip install -e .
```

**After installation, you can run devibe either:**

1. **As a Python module** (works anywhere):

   ```bash
   python -m devibe myfile.py
   ```

2. **Direct command** (if pip Scripts directory is in PATH):
   ```bash
   devibe myfile.py
   ```

## Usage

### Command Line

Clean a single file:

```bash
devibe myfile.py
```

Clean a directory recursively:

```bash
devibe ./src -r
```

**Dry run to see exactly what lines would be removed:**

```bash
devibe myfile.py --dry-run
```

This will show output like:

```
myfile.py - Would remove 3 line(s):

  1. # Generated by Claude
  2.     x = 1  # ✅ AI-generated
  3. # ... rest of implementation
```

Process only specific file types:

```bash
devibe ./src -r -e .py .js .java
```

### Python API

```python
from devibe import CommentRemover

remover = CommentRemover()

# Clean a single file
removed_count, removed_comments = remover.clean_file("myfile.py")

# Clean a directory
stats = remover.clean_directory("./src", recursive=True, extensions=[".py", ".js"])
print(f"Removed {stats['comments_removed']} comments from {stats['files_modified']} files")
```

## Default Rules

devibe comes with intelligent rules to detect and remove:

### AI-Generated Comments

- `generated by Claude`, `AI-generated`, `Created by GPT`
- Comments from Copilot, Codeium, TabNine, and other AI assistants

### IDE-Generated Comments

- Auto-generated code stubs and boilerplate
- IDE template comments from VSCode, IntelliJ, Eclipse

### Status Markers & Emojis

- Emoji indicators (✅, 🔒, 🚀, etc.)
- Status prefixes (`FIXED:`, `REMOVED:`, `ENHANCED:`, `SIMPLIFIED:`)
- Case-insensitive matching with or without colons

### Placeholder Comments

- `... rest of code`, `... existing methods`
- `your code here`, `insert logic here`
- `TODO: implement`, `stub`, `placeholder`

List active rules:

```bash
devibe --list-rules
```

## Adding Custom Rules

I have definetly missed rules so if you have any feel free to go to: `/devibe/rules.py` add a custom rule send a PR.

### Via Command Line

Add custom patterns on the fly:

```bash
devibe myfile.py --add-pattern "my_rule" "pattern_to_match"
```

### Via Python

```python
from devibe import CommentRemover, PatternRule

remover = CommentRemover()

# Add a custom pattern rule
custom_rule = PatternRule(
    name="custom_assistant",
    description="Remove specific assistant comments",
    patterns=[r"generated by my-assistant", r"my-assistant wrote this"]
)
remover.rule_registry.add_rule(custom_rule)

# Now clean files with your custom rule active
remover.clean_file("myfile.py")
```

### Creating Your Own Rule Class

```python
from devibe.rules import Rule

class KeywordRule(Rule):
    def __init__(self, keywords):
        super().__init__("keyword_rule", "Remove comments with specific keywords")
        self.keywords = [k.lower() for k in keywords]

    def matches(self, comment):
        comment_lower = comment.lower()
        return any(keyword in comment_lower for keyword in self.keywords)

# Use your custom rule
remover = CommentRemover()
remover.rule_registry.add_rule(KeywordRule(["autogen", "placeholder"]))
```

## Requirements

- Python 3.7+
- No external dependencies for core functionality

## Development

Install in development mode with:

```bash
git clone https://github.com/nhunter0/devibe.git
cd devibe
pip install -e .
```

Run tests:

```bash
pytest tests/
```

## Contributing

We welcome contributions! To add support for new languages or improve detection rules:

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Add your parser or rule to the appropriate registry
4. Write tests for your changes
5. Submit a pull request

## Roadmap

- [ ] PyPI package publication
- [ ] Devibe UI & UX !
- [ ] VS Code extension
- [ ] More language-specific rules

## License

MIT

## Author

Created with ❤️ to keep codebases clean and professional.

---
