import re
from typing import List, Pattern, Optional
from abc import ABC, abstractmethod


class Rule(ABC):
    """Base class for comment removal rules"""
    
    def __init__(self, name: str, description: str):
        self.name = name
        self.description = description
    
    @abstractmethod
    def matches(self, comment: str) -> bool:
        """Check if this rule matches the given comment"""
        pass


class PatternRule(Rule):
    """Rule that matches comments based on regex patterns"""
    
    def __init__(self, name: str, description: str, patterns: List[str]):
        super().__init__(name, description)
        self.patterns = [re.compile(pattern, re.IGNORECASE) for pattern in patterns]
    
    def matches(self, comment: str) -> bool:
        return any(pattern.search(comment) for pattern in self.patterns)


class RuleRegistry:
    """Registry for managing comment removal rules"""
    
    def __init__(self):
        self.rules: List[Rule] = []
        self._load_default_rules()
    
    def _load_default_rules(self):
        """Load default rules for common auto-generated comments"""
        
        # Removes: Comments containing AI-generated markers like "generated by Claude", "AI-generated", etc.
        self.add_rule(PatternRule(
            "ai_generated",
            "Comments generated by AI coding assistants",
            [
                r"generated\s+(?:by|with)\s+(?:ai|claude|gpt|copilot|codeium|tabnine)",
                r"ai[- ]generated",
                r"auto[- ]generated\s+by\s+(?:ai|assistant)",
                r"created\s+by\s+(?:ai|claude|gpt)",
                r"written\s+by\s+(?:ai|claude|gpt)",
                r"(?:claude|gpt|copilot)\s+(?:generated|created|wrote)",
            ]
        ))
        
        # Removes: Comments generated by IDEs like "auto-generated by VSCode", "generated by IDE", etc.
        self.add_rule(PatternRule(
            "ide_generated",
            "Comments generated by IDEs and code generators",
            [
                r"auto[- ]generated\s+(?:code|comment|by\s+ide)",
                r"generated\s+by\s+(?:ide|editor|vscode|intellij|eclipse)",
                r"created\s+by\s+(?:ide|editor)",
            ]
        ))
        
        # Removes: All emoji characters and status markers like ✅, 🔒, 🚀, etc.
        self.add_rule(PatternRule(
            "emoji_markers",
            "Emoji markers and status indicators",
            [
                r"✅",
                r"❌",
                r"⚠️",
                r"🔐",
                r"🔒",
                r"🔑",
                r"⏳",
                r"🔄",
                r"🚪",
                r"🔍",
                r"🛡️",
                r"🎨",
                r"💾",
                r"🔥",
                r"🚀",
                r"🧪",
                r"📝",
                r"💡",
                r"🛠️",
                r"🔧",
                r"✨",
                r"⚡",
                r"🎉",
                r"🎯",
            ]
        ))
        
        # Removes: Status prefixes like "Real:", "REMOVED:", "FIXED:", "ENHANCED:", "SIMPLIFIED:" (with/without colon, case-insensitive)
        self.add_rule(PatternRule(
            "status_prefixes",
            "Status prefix comments",
            [
                r"(?i)//\s*real:?",
                r"(?i)//\s*removed:?",
                r"(?i)//\s*fixed:?",
                r"(?i)//\s*enhanced:?",
                r"(?i)//\s*simplified:?",
                r"(?i)^\s*real:?",
                r"(?i)^\s*removed:?",
                r"(?i)^\s*fixed:?",
                r"(?i)^\s*enhanced:?",
                r"(?i)^\s*simplified:?",
            ]
        ))
        
        # Removes: Placeholder comments with ellipsis like "... rest of code", "... existing methods"
        self.add_rule(PatternRule(
            "placeholder_ellipsis",
            "Placeholder comments with ellipsis",
            [
                r"\.\.\.\s*rest\s+of\s+(?:your\s+)?(?:methods|functions|code|implementation)",
                r"\.\.\.\s*existing\s+(?:fields|methods|code|implementation)",
                r"\.\.\.\s*rest\s+of\s+(?:the\s+)?(?:function|method|class|code)",
                r"\.\.\.\s*other\s+(?:methods|functions|code)",
                r"\.\.\.\s*(?:more\s+)?(?:code|methods|functions)\s+(?:here|goes\s+here)?",
                r"//\s*\.\.\.",
                r"^\s*\.\.\.$",
            ]
        ))
        
        # Removes: Boilerplate placeholder comments like "your code here", "insert code here", "stub"
        self.add_rule(PatternRule(
            "boilerplate",
            "Common boilerplate comments",
            [
                r"^\s*your\s+code\s+(?:here|goes\s+here)$",
                r"^\s*insert\s+(?:code|logic)\s+here$",
                r"^\s*placeholder\s*$",
                r"^\s*stub\s*$",
            ]
        ))
    
    def add_rule(self, rule: Rule):
        """Add a new rule to the registry"""
        self.rules.append(rule)
    
    def remove_rule(self, name: str):
        """Remove a rule by name"""
        self.rules = [r for r in self.rules if r.name != name]
    
    def should_remove(self, comment: str) -> bool:
        """Check if a comment should be removed based on all rules"""
        return any(rule.matches(comment) for rule in self.rules)
    
    def list_rules(self) -> List[tuple]:
        """List all registered rules"""
        return [(rule.name, rule.description) for rule in self.rules]
    
    def get_detailed_rules(self) -> List[dict]:
        """Get detailed information about all rules including patterns"""
        detailed_rules = []
        for rule in self.rules:
            rule_info = {
                'name': rule.name,
                'description': rule.description,
                'type': type(rule).__name__
            }
            
            # Add patterns for PatternRule instances
            if hasattr(rule, 'patterns'):
                rule_info['patterns'] = [pattern.pattern for pattern in rule.patterns]
            
            detailed_rules.append(rule_info)
        
        return detailed_rules