import pytest
import os
import tempfile
from pathlib import Path
from devibe.remover import CommentRemover
from devibe.rules import PatternRule


class TestCommentRemover:
    def setup_method(self):
        self.remover = CommentRemover()
        self.temp_dir = tempfile.mkdtemp()
    
    def teardown_method(self):
        import shutil
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    def test_initialization(self):
        remover = CommentRemover()
        assert remover.rule_registry is not None
        assert remover.parser_registry is not None
        assert len(remover.rule_registry.rules) > 0
    
    def test_clean_file_python(self):
        test_file = os.path.join(self.temp_dir, "test.py")
        original_content = '''
def foo():
    # Generated by Claude
    x = 1
    # Normal comment
    return x
'''
        with open(test_file, 'w') as f:
            f.write(original_content)
        
        removed_count, removed_comments = self.remover.clean_file(test_file)
        
        with open(test_file, 'r') as f:
            cleaned_content = f.read()
        
        assert "Generated by Claude" not in cleaned_content
        assert "Normal comment" in cleaned_content
        assert removed_count == 1
        assert len(removed_comments) == 1
    
    def test_clean_file_javascript(self):
        test_file = os.path.join(self.temp_dir, "test.js")
        original_content = '''
function foo() {
    // Generated by GPT
    let x = 1;
    // Calculate result
    return x;
}
'''
        with open(test_file, 'w') as f:
            f.write(original_content)
        
        removed_count, removed_comments = self.remover.clean_file(test_file)
        
        with open(test_file, 'r') as f:
            cleaned_content = f.read()
        
        assert "Generated by GPT" not in cleaned_content
        assert "Calculate result" in cleaned_content
        assert removed_count == 1
    
    def test_clean_file_with_emojis(self):
        test_file = os.path.join(self.temp_dir, "test.py")
        original_content = '''
def foo():
    # ✅ Task completed
    x = 1
    # Important function
    return x
'''
        with open(test_file, 'w', encoding='utf-8') as f:
            f.write(original_content)
        
        removed_count, removed_comments = self.remover.clean_file(test_file)
        
        with open(test_file, 'r', encoding='utf-8') as f:
            cleaned_content = f.read()
        
        assert "✅" not in cleaned_content
        assert "Important function" in cleaned_content
        assert removed_count == 1
    
    def test_clean_file_with_status_prefixes(self):
        test_file = os.path.join(self.temp_dir, "test.py")
        original_content = '''
def foo():
    # FIXED: bug resolved
    x = 1
    # ENHANCED: performance improved
    # Check for errors
    return x
'''
        with open(test_file, 'w') as f:
            f.write(original_content)
        
        removed_count, removed_comments = self.remover.clean_file(test_file)
        
        with open(test_file, 'r') as f:
            cleaned_content = f.read()
        
        assert "FIXED:" not in cleaned_content
        assert "ENHANCED:" not in cleaned_content
        assert "Check for errors" in cleaned_content
        assert removed_count == 2
    
    def test_clean_file_dry_run(self):
        test_file = os.path.join(self.temp_dir, "test.py")
        original_content = '''
def foo():
    # Generated by AI
    return 1
'''
        with open(test_file, 'w') as f:
            f.write(original_content)
        
        removed_count, removed_comments = self.remover.clean_file(test_file, dry_run=True)
        
        with open(test_file, 'r') as f:
            content_after = f.read()
        
        assert content_after == original_content
        assert removed_count == 1
        assert len(removed_comments) == 1
    
    def test_clean_directory(self):
        # Create test files
        py_file = os.path.join(self.temp_dir, "test.py")
        with open(py_file, 'w') as f:
            f.write('# Generated by Claude\nx = 1')
        
        js_file = os.path.join(self.temp_dir, "test.js")
        with open(js_file, 'w') as f:
            f.write('// Created by GPT\nlet x = 1;')
        
        txt_file = os.path.join(self.temp_dir, "test.txt")
        with open(txt_file, 'w') as f:
            f.write('Some text file')
        
        stats = self.remover.clean_directory(self.temp_dir)
        
        assert stats['files_processed'] == 2
        assert stats['files_modified'] == 2
        assert stats['comments_removed'] == 2
    
    def test_clean_directory_recursive(self):
        # Create nested directory structure
        sub_dir = os.path.join(self.temp_dir, "subdir")
        os.makedirs(sub_dir)
        
        root_file = os.path.join(self.temp_dir, "root.py")
        with open(root_file, 'w') as f:
            f.write('# AI-generated\nx = 1')
        
        sub_file = os.path.join(sub_dir, "sub.py")
        with open(sub_file, 'w') as f:
            f.write('# Generated by AI\ny = 2')
        
        stats = self.remover.clean_directory(self.temp_dir, recursive=True)
        
        assert stats['files_processed'] == 2
        assert stats['files_modified'] == 2
        assert stats['comments_removed'] == 2
    
    def test_clean_directory_with_extensions(self):
        # Create files with different extensions
        py_file = os.path.join(self.temp_dir, "test.py")
        with open(py_file, 'w') as f:
            f.write('# Generated by Claude\nx = 1')
        
        js_file = os.path.join(self.temp_dir, "test.js")
        with open(js_file, 'w') as f:
            f.write('// Created by GPT\nlet x = 1;')
        
        stats = self.remover.clean_directory(
            self.temp_dir,
            extensions=['.py']
        )
        
        assert stats['files_processed'] == 1
        assert stats['files_modified'] == 1
        
        with open(js_file, 'r') as f:
            js_content = f.read()
        assert "Created by GPT" in js_content
    
    def test_custom_rule_addition(self):
        custom_rule = PatternRule(
            name="custom_test",
            description="Custom test rule",
            patterns=[r"CUSTOM_MARKER"]
        )
        self.remover.rule_registry.add_rule(custom_rule)
        
        test_file = os.path.join(self.temp_dir, "test.py")
        content = '''
def foo():
    # CUSTOM_MARKER: test
    # Normal comment
    return 1
'''
        with open(test_file, 'w') as f:
            f.write(content)
        
        removed_count, removed_comments = self.remover.clean_file(test_file)
        
        with open(test_file, 'r') as f:
            cleaned_content = f.read()
        
        assert "CUSTOM_MARKER" not in cleaned_content
        assert "Normal comment" in cleaned_content
        assert removed_count == 1
    
    def test_unsupported_file_extension(self):
        test_file = os.path.join(self.temp_dir, "test.unknown")
        with open(test_file, 'w') as f:
            f.write('Some content')
        
        removed_count, removed_comments = self.remover.clean_file(test_file)
        
        assert removed_count == 0
        assert len(removed_comments) == 0
    
    def test_file_not_found(self):
        with pytest.raises(FileNotFoundError):
            self.remover.clean_file("nonexistent.py")