import pytest
from devibe.rules import Rule, PatternRule, RuleRegistry


class TestPatternRule:
    def test_pattern_rule_creation(self):
        rule = PatternRule(
            name="test_rule",
            description="Test description",
            patterns=[r"test_pattern"]
        )
        assert rule.name == "test_rule"
        assert rule.description == "Test description"
        assert len(rule.patterns) == 1
    
    def test_pattern_rule_matches(self):
        rule = PatternRule(
            name="test_rule",
            description="Test description",
            patterns=[r"generated by AI", r"auto-generated"]
        )
        assert rule.matches("This was generated by AI")
        assert rule.matches("This is auto-generated code")
        assert not rule.matches("This is normal code")
    
    def test_pattern_rule_case_insensitive(self):
        rule = PatternRule(
            name="test_rule",
            description="Test description",
            patterns=[r"(?i)claude"]
        )
        assert rule.matches("Generated by Claude")
        assert rule.matches("Generated by claude")
        assert rule.matches("Generated by CLAUDE")


class TestRuleRegistry:
    def test_registry_initialization(self):
        registry = RuleRegistry()
        assert len(registry.rules) > 0
        assert any(rule.name == "ai_generated" for rule in registry.rules)
        assert any(rule.name == "emoji_markers" for rule in registry.rules)
    
    def test_add_rule(self):
        registry = RuleRegistry()
        initial_count = len(registry.rules)
        
        new_rule = PatternRule(
            name="custom_rule",
            description="Custom test rule",
            patterns=[r"custom_pattern"]
        )
        registry.add_rule(new_rule)
        
        assert len(registry.rules) == initial_count + 1
        assert any(rule.name == "custom_rule" for rule in registry.rules)
    
    def test_remove_rule(self):
        registry = RuleRegistry()
        registry.add_rule(PatternRule(
            name="temp_rule",
            description="Temporary rule",
            patterns=[r"temp"]
        ))
        
        initial_count = len(registry.rules)
        registry.remove_rule("temp_rule")
        
        assert len(registry.rules) == initial_count - 1
        assert not any(rule.name == "temp_rule" for rule in registry.rules)
    
    def test_should_remove_ai_comments(self):
        registry = RuleRegistry()
        
        ai_comments = [
            "Generated by Claude",
            "AI-generated code",
            "Created by GPT",
            "Generated by Copilot",
            "auto-generated by AI",
        ]
        
        for comment in ai_comments:
            assert registry.should_remove(comment), f"Should remove: {comment}"
    
    def test_should_remove_emoji_comments(self):
        registry = RuleRegistry()
        
        emoji_comments = [
            "✅ Task completed",
            "🔒 Secure function",
            "🚀 Performance optimized",
            "Code with ⚠️ warning",
        ]
        
        for comment in emoji_comments:
            assert registry.should_remove(comment), f"Should remove: {comment}"
    
    def test_should_remove_status_prefixes(self):
        registry = RuleRegistry()
        
        status_comments = [
            "// FIXED: bug resolved",
            "// REMOVED: old code",
            "// ENHANCED: improved performance",
            "// SIMPLIFIED: cleaner code",
            "Real: implementation",
            "fixed bug here",
            "FIXED bug here",
        ]
        
        for comment in status_comments:
            assert registry.should_remove(comment), f"Should remove: {comment}"
    
    def test_should_remove_placeholder_comments(self):
        registry = RuleRegistry()
        
        placeholder_comments = [
            "... rest of code",
            "... existing methods",
            "// ...",
            "... rest of your implementation",
        ]
        
        for comment in placeholder_comments:
            assert registry.should_remove(comment), f"Should remove: {comment}"
    
    def test_should_not_remove_normal_comments(self):
        registry = RuleRegistry()
        
        normal_comments = [
            "Calculate the sum of two numbers",
            "This function handles user authentication",
            "Important: Check for null values",
            "TODO: Optimize this algorithm",
        ]
        
        for comment in normal_comments:
            assert not registry.should_remove(comment), f"Should keep: {comment}"
    
    def test_list_rules(self):
        registry = RuleRegistry()
        rules_list = registry.list_rules()
        
        assert isinstance(rules_list, list)
        assert all(isinstance(rule, tuple) and len(rule) == 2 for rule in rules_list)
        
        rule_names = [rule[0] for rule in rules_list]
        assert "ai_generated" in rule_names
        assert "emoji_markers" in rule_names
        assert "status_prefixes" in rule_names