# fbdfile/setup.py

"""Fbdfile package Setuptools script."""

import os
import re
import sys

import numpy
from setuptools import Extension, setup

buildnumber = ''

DEBUG = bool(os.environ.get('FBDFILE_DEBUG', False))


def search(pattern: str, string: str, flags: int = 0) -> str:
    """Return first match of pattern in string."""
    match = re.search(pattern, string, flags)
    if match is None:
        raise ValueError(f'{pattern!r} not found')
    return match.groups()[0]


def fix_docstring_examples(docstring: str) -> str:
    """Return docstring with examples fixed for GitHub."""
    start = True
    indent = False
    lines = ['..', '  This file is generated by setup.py', '']
    for line in docstring.splitlines():
        if not line.strip():
            start = True
            indent = False
        if line.startswith('>>> '):
            indent = True
            if start:
                lines.extend(['.. code-block:: python', ''])
                start = False
        lines.append(('    ' if indent else '') + line)
    return '\n'.join(lines)


with open('fbdfile/fbdfile.py', encoding='utf-8') as fh:
    code = fh.read()

version = search(r"__version__ = '(.*?)'", code).replace('.x.x', '.dev0')
version += ('.' + buildnumber) if buildnumber else ''

description = search(r'"""(.*)\.(?:\r\n|\r|\n)', code)

readme = search(
    r'(?:\r\n|\r|\n){2}"""(.*)"""(?:\r\n|\r|\n){2}from __future__',
    code,
    re.MULTILINE | re.DOTALL,
)
readme = '\n'.join(
    [description, '=' * len(description)] + readme.splitlines()[1:]
)

if 'sdist' in sys.argv:
    # update README, LICENSE, and CHANGES files

    with open('README.rst', 'w', encoding='utf-8') as fh:
        fh.write(fix_docstring_examples(readme))

    license = search(
        r'(# Copyright.*?(?:\r\n|\r|\n))(?:\r\n|\r|\n)+""',
        code,
        re.MULTILINE | re.DOTALL,
    )
    license = license.replace('# ', '').replace('#', '')

    with open('LICENSE', 'w', encoding='utf-8') as fh:
        fh.write('BSD-3-Clause license\n\n')
        fh.write(license)

    # revisions = search(
    #     r'(?:\r\n|\r|\n){2}(Revisions.*)- …',
    #     readme,
    #     re.MULTILINE | re.DOTALL,
    # ).strip()

    # with open('CHANGES.rst', encoding='utf-8') as fh:
    #     old = fh.read()

    # old = old.split(revisions.splitlines()[-1])[-1]
    # with open('CHANGES.rst', 'w', encoding='utf-8') as fh:
    #     fh.write(revisions.strip())
    #     fh.write(old)

if DEBUG:
    extra_compile_args = ['/Zi', '/Od']
    extra_link_args = ['-debug:full']
elif sys.platform == 'win32':
    extra_compile_args = ['/openmp']
    extra_link_args = []
elif sys.platform == 'darwin':
    # https://mac.r-project.org/openmp/
    extra_compile_args = ['-Xclang', '-fopenmp']
    extra_link_args = ['-lomp']
else:
    extra_compile_args = ['-fopenmp']
    extra_link_args = ['-fopenmp']

ext_modules = [
    Extension(
        'fbdfile._fbdfile',
        ['fbdfile/_fbdfile.pyx'],
        define_macros=[
            # ('CYTHON_TRACE_NOGIL', '1'),
            # ('CYTHON_LIMITED_API', '1'),
            # ('Py_LIMITED_API', '1'),
            ('NPY_NO_DEPRECATED_API', 'NPY_1_7_API_VERSION'),
        ],
        extra_compile_args=extra_compile_args,
        extra_link_args=extra_link_args,
        include_dirs=[numpy.get_include()],
    ),
]

setup(
    name='fbdfile',
    version=version,
    license='BSD-3-Clause',
    description=description,
    long_description=readme,
    long_description_content_type='text/x-rst',
    author='Christoph Gohlke',
    author_email='cgohlke@cgohlke.com',
    url='https://www.cgohlke.com',
    project_urls={
        'Bug Tracker': 'https://github.com/cgohlke/fbdfile/issues',
        'Source Code': 'https://github.com/cgohlke/fbdfile',
        # 'Documentation': 'https://',
    },
    packages=['fbdfile'],
    package_data={'fbdfile': ['py.typed']},
    entry_points={
        'console_scripts': [
            'fbdfile = fbdfile.__main__:main',
            'fbd2b64 = fbdfile.fbd2b64:main',
        ]
    },
    python_requires='>=3.11',
    install_requires=['numpy'],
    extras_require={'all': ['tifffile', 'matplotlib', 'click']},
    ext_modules=ext_modules,
    zip_safe=False,
    platforms=['any'],
    classifiers=[
        'Development Status :: 3 - Alpha',
        'Intended Audience :: Science/Research',
        'Intended Audience :: Developers',
        'Operating System :: OS Independent',
        'Programming Language :: Cython',
        'Programming Language :: Python :: 3 :: Only',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: 3.12',
        'Programming Language :: Python :: 3.13',
        'Programming Language :: Python :: 3.14',
    ],
)
