import * as fs from 'node:fs';
import * as path from 'node:path';

import type { AttributeJson } from './types';

// Function to read and parse a JSON file
function readJsonFile(filePath: string): AttributeJson {
  const fileContent = fs.readFileSync(filePath, 'utf8');
  return JSON.parse(fileContent) as AttributeJson;
}

// Function to generate a markdown table for a single attribute
function generateAttributeTable(attribute: AttributeJson): string {
  let table = `### ${attribute.key}\n\n`;
  table += `${attribute.brief}\n\n`;
  table += '| Property | Value |\n';
  table += '| --- | --- |\n';
  table += `| Type | \`${attribute.type}\` |\n`;

  const piiValue = attribute.pii.key + (attribute.pii.reason ? ` (${attribute.pii.reason})` : '');
  table += `| Has PII | ${piiValue} |\n`;

  table += `| Exists in OpenTelemetry | ${attribute.is_in_otel ? 'Yes' : 'No'} |\n`;

  if (attribute.has_dynamic_suffix) {
    table += '| Has dynamic suffix | Yes |\n';
  }

  if (attribute.example !== undefined) {
    const exampleValue =
      typeof attribute.example === 'object' ? JSON.stringify(attribute.example) : String(attribute.example);
    table += `| Example | \`${exampleValue}\` |\n`;
  }

  if (attribute.deprecation) {
    if (attribute.deprecation.replacement) {
      table += `| Deprecated | Yes, use \`${attribute.deprecation.replacement}\` instead |\n`;
    } else {
      table += '| Deprecated | Yes, no replacement at this time |\n';
    }
    if (attribute.deprecation.reason) {
      table += `| Deprecation Reason | ${attribute.deprecation.reason} |\n`;
    }
  }

  if (attribute.alias && attribute.alias.length > 0) {
    table += `| Aliases | ${attribute.alias.map((a) => `\`${a}\``).join(', ')} |\n`;
  }

  table += '\n';

  return table;
}

// Main function to generate all markdown docs
export async function generateAttributeDocs() {
  const baseDir = 'model/attributes';
  const outputDir = 'generated/attributes';
  const categories: Record<string, AttributeJson[]> = {};

  // Ensure output directory exists
  if (!fs.existsSync(outputDir)) {
    fs.mkdirSync(outputDir, { recursive: true });
  }

  // Process top-level files (they go into a "general" category)
  const topLevelFiles = fs
    .readdirSync(baseDir)
    .filter((file) => file.endsWith('.json'))
    .map((file) => path.join(baseDir, file));

  if (topLevelFiles.length > 0) {
    categories.general = topLevelFiles.map((file) => readJsonFile(file));
  }

  // Process subdirectories
  const subdirs = fs
    .readdirSync(baseDir, { withFileTypes: true })
    .filter((dirent) => dirent.isDirectory())
    .map((dirent) => dirent.name);

  for (const subdir of subdirs) {
    const categoryDir = path.join(baseDir, subdir);
    const files = fs
      .readdirSync(categoryDir)
      .filter((file) => file.endsWith('.json'))
      .map((file) => path.join(categoryDir, file));

    if (files.length > 0) {
      categories[subdir] = files.map((file) => readJsonFile(file));
    }
  }

  // Generate markdown files for each category
  for (const [category, attributes] of Object.entries(categories)) {
    // Add autogenerated warning
    let markdown = '<!-- THIS FILE IS AUTO-GENERATED. DO NOT EDIT DIRECTLY. -->\n\n';
    markdown += `# ${category.charAt(0).toUpperCase() + category.slice(1)} Attributes\n\n`;

    // Split attributes into stable and deprecated
    const stableAttributes = attributes.filter((attr) => !attr.deprecation);
    const deprecatedAttributes = attributes.filter((attr) => attr.deprecation);

    // Sort both arrays alphabetically by key
    stableAttributes.sort((a, b) => a.key.localeCompare(b.key));
    deprecatedAttributes.sort((a, b) => a.key.localeCompare(b.key));

    if (stableAttributes.length > 0) {
      markdown += '- [Stable Attributes](#stable-attributes)\n';
      for (const attr of stableAttributes) {
        const anchorLink = attr.key.toLowerCase().replaceAll('.', '').replaceAll('-', '').replaceAll('<key>', 'key');
        markdown += `  - [${attr.key}](#${anchorLink})\n`;
      }
    }

    if (deprecatedAttributes.length > 0) {
      markdown += '- [Deprecated Attributes](#deprecated-attributes)\n';
      for (const attr of deprecatedAttributes) {
        const anchorLink = attr.key.toLowerCase().replaceAll('.', '').replaceAll('-', '').replaceAll('<key>', 'key');
        markdown += `  - [${attr.key}](#${anchorLink})\n`;
      }
    }

    markdown += '\n';

    // Generate stable attributes section
    if (stableAttributes.length > 0) {
      markdown += '## Stable Attributes\n\n';
      for (const attribute of stableAttributes) {
        markdown += generateAttributeTable(attribute);
      }
    }

    // Generate deprecated attributes section
    if (deprecatedAttributes.length > 0) {
      markdown += '## Deprecated Attributes\n\n';
      markdown +=
        'These attributes are deprecated and will be removed in a future version. Please use the recommended replacements.\n\n';
      for (const attribute of deprecatedAttributes) {
        markdown += generateAttributeTable(attribute);
      }
    }

    markdown = markdown.replaceAll('<key>', '\\<key\\>');

    // Write the markdown file
    const outputFile = path.join(outputDir, `${category}.md`);
    fs.writeFileSync(outputFile, markdown);
    console.log(`Generated documentation for ${category} attributes: ${outputFile}`);
  }

  // Create index.md file that links to all categories
  let indexContent = '<!-- THIS FILE IS AUTO-GENERATED. DO NOT EDIT DIRECTLY. -->\n\n';
  indexContent += '# Attribute Documentation\n\n';
  indexContent += 'This directory contains documentation for all available attributes.\n\n';
  indexContent += '## Available Categories\n\n';

  // Generate links to each category
  for (const category of Object.keys(categories).sort()) {
    indexContent += `- [\`${category}\` Attributes](./${category}.md)\n`;
  }

  // Write the index.md file
  const indexFile = path.join(outputDir, 'index.md');
  fs.writeFileSync(indexFile, indexContent);
  console.log(`Generated index file: ${indexFile}`);

  console.log('Documentation generation complete!');
}
