from datetime import datetime, timezone
from unittest import mock

from .asserts import time_within_delta


def test_ai_spans_example_transaction(
    mini_sentry, relay, relay_with_processing, spans_consumer
):
    """
    Asserts the span output of an example AI agent workflow.

    This test acts as a regression test for future modifications.
    The example was taken from a test application with a real agentic workflow.
    """
    spans_consumer = spans_consumer()

    project_id = 42
    mini_sentry.add_full_project_config(project_id)

    mini_sentry.global_config["aiModelCosts"] = {
        "version": 2,
        "models": {
            "gpt-4o": {
                "inputPerToken": 0.01,
                "outputPerToken": 0.02,
                "outputReasoningPerToken": 0.03,
                "inputCachedPerToken": 0.0,
            },
        },
    }
    mini_sentry.global_config["aiOperationTypeMap"] = {
        "version": 1,
        "operationTypes": {
            "ai.run.generateText": "agent",
            "ai.run.generateObject": "agent",
            "gen_ai.invoke_agent": "agent",
            "ai.pipeline.generate_text": "agent",
            "ai.pipeline.generate_object": "agent",
            "ai.pipeline.stream_text": "agent",
            "ai.pipeline.stream_object": "agent",
            "gen_ai.create_agent": "agent",
            "gen_ai.execute_tool": "tool",
            "gen_ai.handoff": "handoff",
            "*": "ai_client",
        },
    }

    relay = relay(relay_with_processing())

    ts = datetime.now(timezone.utc)

    transaction = {
        "contexts": {
            "trace": {
                "span_id": "657cf984a6a4e59b",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.source": "custom",
                    "sentry.sample_rate": 1,
                    "sentry.origin": "manual",
                },
                "origin": "manual",
                "status": "ok",
            },
        },
        "spans": [
            {
                "span_id": "13e7c1ffd66981f0",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.vercelai.otel",
                    "sentry.op": "gen_ai.invoke_agent",
                    "operation.name": "ai.generateText weather-chat",
                    "resource.name": "weather-chat",
                    "gen_ai.function_id": "weather-chat",
                    "gen_ai.prompt": "Weather Prompt",
                    "gen_ai.response.model": "gpt-4o",
                    "gen_ai.usage.output_tokens": 65,
                    "gen_ai.usage.input_tokens": 245,
                    "gen_ai.usage.total_tokens": 310,
                    "gen_ai.response.text": "True. \n\n- London: 61°F \n- San Francisco: 13°C",
                    "gen_ai.conversation.id": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                    "vercel.ai.operationId": "ai.generateText",
                    "vercel.ai.telemetry.functionId": "weather-chat",
                    "vercel.ai.model.provider": "openai.responses",
                    "vercel.ai.model.id": "gpt-4o",
                    "vercel.ai.settings.maxRetries": 2,
                    "vercel.ai.request.headers.user-agent": "ai/5.0.71",
                    "vercel.ai.prompt": "Weather Prompt",
                    "vercel.ai.pipeline.name": "generateText",
                    "vercel.ai.streaming": False,
                    "vercel.ai.response.finishReason": "stop",
                    "vercel.ai.response.providerMetadata": '{"openai":{"responseId":"resp_0","serviceTier":"default"}}',
                },
                "description": "generateText weather-chat",
                "parent_span_id": "657cf984a6a4e59b",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "gen_ai.invoke_agent",
                "origin": "auto.vercelai.otel",
            },
            {
                "span_id": "af7b547f4d5f4f49",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.vercelai.otel",
                    "sentry.op": "gen_ai.generate_text",
                    "operation.name": "ai.generateText.doGenerate weather-chat",
                    "resource.name": "weather-chat",
                    "gen_ai.system": "openai.responses",
                    "gen_ai.request.model": "gpt-4o",
                    "gen_ai.function_id": "weather-chat",
                    "gen_ai.response.model": "gpt-4o-2024-08-06",
                    "gen_ai.response.finish_reasons": ["tool-calls"],
                    "gen_ai.response.id": "resp_0c1c943ef2dc8bf9006909e7b649008197a541a144de019abf",
                    "gen_ai.usage.input_tokens": 37,
                    "gen_ai.usage.output_tokens": 46,
                    "gen_ai.usage.total_tokens": 83,
                    "gen_ai.request.messages": "Another weather prompt",
                    "gen_ai.response.tool_calls": "some_tool_calls",
                    "gen_ai.request.available_tools": ["tools1"],
                    "gen_ai.conversation.id": "resp_0c1c943ef2dc8bf9006909e7b649008197a541a144de019abf",
                    "vercel.ai.operationId": "ai.generateText.doGenerate",
                    "vercel.ai.telemetry.functionId": "weather-chat",
                    "vercel.ai.model.provider": "openai.responses",
                    "vercel.ai.model.id": "gpt-4o",
                    "vercel.ai.settings.maxRetries": 2,
                    "vercel.ai.request.headers.user-agent": "ai/5.0.71",
                    "vercel.ai.prompt.toolChoice": '{"type":"auto"}',
                    "vercel.ai.pipeline.name": "generateText.doGenerate",
                    "vercel.ai.streaming": False,
                    "vercel.ai.response.finishReason": "tool-calls",
                    "vercel.ai.response.id": "resp_0c1c943ef2dc8bf9006909e7b649008197a541a144de019abf",
                    "vercel.ai.response.model": "gpt-4o-2024-08-06",
                    "vercel.ai.response.timestamp": "2025-11-04T11:47:02.000Z",
                    "vercel.ai.response.providerMetadata": '{"openai":{"responseId":"resp_0","serviceTier":"default"}}',
                },
                "description": "generate_text gpt-4o",
                "parent_span_id": "13e7c1ffd66981f0",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "gen_ai.generate_text",
                "origin": "auto.vercelai.otel",
            },
            {
                "span_id": "5ff84ff6bf512012",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.http.otel.node_fetch",
                    "sentry.op": "http.client",
                    "http.request.method": "POST",
                    "http.request.method_original": "POST",
                    "url.full": "https://api.openai.com/v1/responses",
                    "url.path": "/v1/responses",
                    "url.query": "",
                    "url.scheme": "https",
                    "server.address": "api.openai.com",
                    "server.port": 443,
                    "user_agent.original": "ai/5.0.71 ai-sdk/provider-utils/3.0.12 runtime/node.js/v20.19.1",
                    "network.peer.address": "162.159.140.245",
                    "network.peer.port": 443,
                    "http.response.status_code": 200,
                    "url": "https://api.openai.com/v1/responses",
                    "otel.kind": "CLIENT",
                },
                "description": "POST https://api.openai.com/v1/responses",
                "parent_span_id": "af7b547f4d5f4f49",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "http.client",
                "origin": "auto.http.otel.node_fetch",
            },
            {
                "span_id": "d2fd68d7cf6eb933",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.vercelai.otel",
                    "sentry.op": "gen_ai.execute_tool",
                    "operation.name": "ai.toolCall weather-chat",
                    "resource.name": "weather-chat",
                    "gen_ai.tool.name": "getWeather",
                    "gen_ai.tool.call.id": "call_jRcFbOh5zNVeV2l2mEqpRiI4",
                    "gen_ai.tool.type": "function",
                    "gen_ai.tool.input": '{"city":"San Francisco"}',
                    "gen_ai.tool.output": '{"location":"San Francisco, United States of America","temperature":"13°C (56°F)","temperatureC":13,"temperatureF":56,"condition":"Haze","humidity":"88%","windSpeed":"4 km/h"}',
                    "vercel.ai.operationId": "ai.toolCall",
                    "vercel.ai.telemetry.functionId": "weather-chat",
                },
                "description": "execute_tool getWeather",
                "parent_span_id": "13e7c1ffd66981f0",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "gen_ai.execute_tool",
                "origin": "auto.vercelai.otel",
            },
            {
                "span_id": "672681a999129905",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.http.otel.node_fetch",
                    "sentry.op": "http.client",
                    "http.request.method": "GET",
                    "http.request.method_original": "GET",
                    "url.full": "https://wttr.in/San%20Francisco?format=j1",
                    "url.path": "/San%20Francisco",
                    "url.query": "?format=j1",
                    "url.scheme": "https",
                    "server.address": "wttr.in",
                    "server.port": 443,
                    "user_agent.original": "node",
                    "network.peer.address": "5.9.243.187",
                    "network.peer.port": 443,
                    "http.response.status_code": 200,
                    "http.response.header.content-length": 50712,
                    "url": "https://wttr.in/San%20Francisco",
                    "http.query": "format=j1",
                    "otel.kind": "CLIENT",
                },
                "description": "GET https://wttr.in/San%20Francisco",
                "parent_span_id": "d2fd68d7cf6eb933",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "http.client",
                "origin": "auto.http.otel.node_fetch",
            },
            {
                "span_id": "e5bb8f1d156e7649",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.vercelai.otel",
                    "sentry.op": "gen_ai.execute_tool",
                    "operation.name": "ai.toolCall weather-chat",
                    "resource.name": "weather-chat",
                    "gen_ai.tool.name": "getWeather",
                    "gen_ai.tool.call.id": "call_yXdgMJXsotHOn19VdlDjrc7c",
                    "gen_ai.tool.type": "function",
                    "gen_ai.tool.input": '{"city":"London"}',
                    "gen_ai.tool.output": '{"location":"London, United Kingdom","temperature":"16°C (61°F)","temperatureC":16,"temperatureF":61,"condition":"Partly cloudy","humidity":"72%","windSpeed":"21 km/h"}',
                    "vercel.ai.operationId": "ai.toolCall",
                    "vercel.ai.telemetry.functionId": "weather-chat",
                },
                "description": "execute_tool getWeather",
                "parent_span_id": "13e7c1ffd66981f0",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "gen_ai.execute_tool",
                "origin": "auto.vercelai.otel",
            },
            {
                "span_id": "3d755d9884113eba",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.http.otel.node_fetch",
                    "sentry.op": "http.client",
                    "http.request.method": "GET",
                    "http.request.method_original": "GET",
                    "url.full": "https://wttr.in/London?format=j1",
                    "url.path": "/London",
                    "url.query": "?format=j1",
                    "url.scheme": "https",
                    "server.address": "wttr.in",
                    "server.port": 443,
                    "user_agent.original": "node",
                    "network.peer.address": "5.9.243.187",
                    "network.peer.port": 443,
                    "http.response.status_code": 200,
                    "http.response.header.content-length": 50728,
                    "url": "https://wttr.in/London",
                    "http.query": "format=j1",
                    "otel.kind": "CLIENT",
                },
                "description": "GET https://wttr.in/London",
                "parent_span_id": "e5bb8f1d156e7649",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "http.client",
                "origin": "auto.http.otel.node_fetch",
            },
            {
                "span_id": "bdf1648756367ee5",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.vercelai.otel",
                    "sentry.op": "gen_ai.generate_text",
                    "operation.name": "ai.generateText.doGenerate weather-chat",
                    "resource.name": "weather-chat",
                    "gen_ai.system": "openai.responses",
                    "gen_ai.request.model": "gpt-4o",
                    "gen_ai.function_id": "weather-chat",
                    "gen_ai.response.model": "gpt-4o-2024-08-06",
                    "gen_ai.response.finish_reasons": ["stop"],
                    "gen_ai.response.id": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                    "gen_ai.usage.input_tokens": 208,
                    "gen_ai.usage.output_tokens": 19,
                    "gen_ai.usage.total_tokens": 227,
                    "gen_ai.request.messages": "Some AI Prompt about the Wheather",
                    "gen_ai.response.text": "True. \n\n- London: 61°F \n- San Francisco: 13°C",
                    "gen_ai.request.available_tools": ["tool_1"],
                    "gen_ai.conversation.id": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                    "vercel.ai.operationId": "ai.generateText.doGenerate",
                    "vercel.ai.telemetry.functionId": "weather-chat",
                    "vercel.ai.model.provider": "openai.responses",
                    "vercel.ai.model.id": "gpt-4o",
                    "vercel.ai.settings.maxRetries": 2,
                    "vercel.ai.request.headers.user-agent": "ai/5.0.71",
                    "vercel.ai.prompt.toolChoice": '{"type":"auto"}',
                    "vercel.ai.pipeline.name": "generateText.doGenerate",
                    "vercel.ai.streaming": False,
                    "vercel.ai.response.finishReason": "stop",
                    "vercel.ai.response.id": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                    "vercel.ai.response.model": "gpt-4o-2024-08-06",
                    "vercel.ai.response.timestamp": "2025-11-04T11:47:04.000Z",
                    "vercel.ai.response.providerMetadata": '{"openai":{"responseId":"resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1","serviceTier":"default"}}',
                },
                "description": "generate_text gpt-4o",
                "parent_span_id": "13e7c1ffd66981f0",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "gen_ai.generate_text",
                "origin": "auto.vercelai.otel",
            },
            {
                "span_id": "203e925b464ad87b",
                "trace_id": "a9351cd574f092f6acad48e250981f11",
                "data": {
                    "sentry.origin": "auto.http.otel.node_fetch",
                    "sentry.op": "http.client",
                    "http.request.method": "POST",
                    "http.request.method_original": "POST",
                    "url.full": "https://api.openai.com/v1/responses",
                    "url.path": "/v1/responses",
                    "url.query": "",
                    "url.scheme": "https",
                    "server.address": "api.openai.com",
                    "server.port": 443,
                    "user_agent.original": "ai/5.0.71 ai-sdk/provider-utils/3.0.12 runtime/node.js/v20.19.1",
                    "network.peer.address": "162.159.140.245",
                    "network.peer.port": 443,
                    "http.response.status_code": 200,
                    "url": "https://api.openai.com/v1/responses",
                    "otel.kind": "CLIENT",
                },
                "description": "POST https://api.openai.com/v1/responses",
                "parent_span_id": "bdf1648756367ee5",
                "start_timestamp": ts.timestamp() - 0.5,
                "timestamp": ts.timestamp(),
                "status": "ok",
                "op": "http.client",
                "origin": "auto.http.otel.node_fetch",
            },
        ],
        "start_timestamp": ts.timestamp() - 0.5,
        "timestamp": ts.timestamp(),
        "transaction": "main",
        "type": "transaction",
        "transaction_info": {"source": "custom"},
        "platform": "node",
        "server_name": "some_machine.local",
        "environment": "production",
    }

    relay.send_transaction(42, transaction)

    assert spans_consumer.get_spans(n=10) == [
        {
            "attributes": {
                "gen_ai.conversation.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                },
                "gen_ai.cost.input_tokens": {"type": "double", "value": 2.45},
                "gen_ai.cost.output_tokens": {"type": "double", "value": 1.3},
                "gen_ai.cost.total_tokens": {"type": "double", "value": 3.75},
                "gen_ai.function_id": {"type": "string", "value": "weather-chat"},
                "gen_ai.operation.type": {"type": "string", "value": "agent"},
                "gen_ai.prompt": {"type": "string", "value": "Weather Prompt"},
                "gen_ai.response.model": {"type": "string", "value": "gpt-4o"},
                "gen_ai.response.text": {
                    "type": "string",
                    "value": "True. \n\n- London: 61°F \n- San Francisco: 13°C",
                },
                "gen_ai.response.tokens_per_second": {"type": "double", "value": 130.0},
                "gen_ai.usage.input_tokens": {"type": "integer", "value": 245},
                "gen_ai.usage.output_tokens": {"type": "integer", "value": 65},
                "gen_ai.usage.total_cost": {"type": "double", "value": 3.75},
                "gen_ai.usage.total_tokens": {"type": "integer", "value": 310},
                "operation.name": {
                    "type": "string",
                    "value": "ai.generateText weather-chat",
                },
                "resource.name": {"type": "string", "value": "weather-chat"},
                "sentry.description": {
                    "type": "string",
                    "value": "generateText weather-chat",
                },
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 0.0},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.op": {"type": "string", "value": "gen_ai.invoke_agent"},
                "sentry.origin": {"type": "string", "value": "auto.vercelai.otel"},
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "vercel.ai.model.id": {"type": "string", "value": "gpt-4o"},
                "vercel.ai.model.provider": {
                    "type": "string",
                    "value": "openai.responses",
                },
                "vercel.ai.operationId": {"type": "string", "value": "ai.generateText"},
                "vercel.ai.pipeline.name": {"type": "string", "value": "generateText"},
                "vercel.ai.prompt": {"type": "string", "value": "Weather Prompt"},
                "vercel.ai.request.headers.user-agent": {
                    "type": "string",
                    "value": "ai/5.0.71",
                },
                "vercel.ai.response.finishReason": {"type": "string", "value": "stop"},
                "vercel.ai.response.providerMetadata": {
                    "type": "string",
                    "value": '{"openai":{"responseId":"resp_0","serviceTier":"default"}}',
                },
                "vercel.ai.settings.maxRetries": {"type": "integer", "value": 2},
                "vercel.ai.streaming": {"type": "boolean", "value": False},
                "vercel.ai.telemetry.functionId": {
                    "type": "string",
                    "value": "weather-chat",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "Generative AI agent operation",
            "organization_id": 1,
            "parent_span_id": "657cf984a6a4e59b",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "13e7c1ffd66981f0",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "_meta": mock.ANY,
            "attributes": {
                "gen_ai.conversation.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b649008197a541a144de019abf",
                },
                "gen_ai.cost.input_tokens": {"type": "double", "value": 0.37},
                "gen_ai.cost.output_tokens": {"type": "double", "value": 0.92},
                "gen_ai.cost.total_tokens": {"type": "double", "value": 1.29},
                "gen_ai.function_id": {"type": "string", "value": "weather-chat"},
                "gen_ai.operation.type": {"type": "string", "value": "ai_client"},
                "gen_ai.request.available_tools": {
                    "type": "string",
                    "value": '["tools1"]',
                },
                "gen_ai.request.messages": {
                    "type": "string",
                    "value": "Another weather prompt",
                },
                "gen_ai.request.model": {"type": "string", "value": "gpt-4o"},
                "gen_ai.response.finish_reasons": {
                    "type": "string",
                    "value": '["tool-calls"]',
                },
                "gen_ai.response.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b649008197a541a144de019abf",
                },
                "gen_ai.response.model": {
                    "type": "string",
                    "value": "gpt-4o-2024-08-06",
                },
                "gen_ai.response.tokens_per_second": {"type": "double", "value": 92.0},
                "gen_ai.response.tool_calls": {
                    "type": "string",
                    "value": "some_tool_calls",
                },
                "gen_ai.system": {"type": "string", "value": "openai.responses"},
                "gen_ai.usage.input_tokens": {"type": "integer", "value": 37},
                "gen_ai.usage.output_tokens": {"type": "integer", "value": 46},
                "gen_ai.usage.total_cost": {"type": "double", "value": 1.29},
                "gen_ai.usage.total_tokens": {"type": "integer", "value": 83},
                "operation.name": {
                    "type": "string",
                    "value": "ai.generateText.doGenerate " "weather-chat",
                },
                "resource.name": {"type": "string", "value": "weather-chat"},
                "sentry.description": {
                    "type": "string",
                    "value": "generate_text gpt-4o",
                },
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 0.0},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.op": {"type": "string", "value": "gen_ai.generate_text"},
                "sentry.origin": {"type": "string", "value": "auto.vercelai.otel"},
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "vercel.ai.model.id": {"type": "string", "value": "gpt-4o"},
                "vercel.ai.model.provider": {
                    "type": "string",
                    "value": "openai.responses",
                },
                "vercel.ai.operationId": {
                    "type": "string",
                    "value": "ai.generateText.doGenerate",
                },
                "vercel.ai.pipeline.name": {
                    "type": "string",
                    "value": "generateText.doGenerate",
                },
                "vercel.ai.prompt.toolChoice": {
                    "type": "string",
                    "value": '{"type":"auto"}',
                },
                "vercel.ai.request.headers.user-agent": {
                    "type": "string",
                    "value": "ai/5.0.71",
                },
                "vercel.ai.response.finishReason": {
                    "type": "string",
                    "value": "tool-calls",
                },
                "vercel.ai.response.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b649008197a541a144de019abf",
                },
                "vercel.ai.response.model": {
                    "type": "string",
                    "value": "gpt-4o-2024-08-06",
                },
                "vercel.ai.response.providerMetadata": {
                    "type": "string",
                    "value": '{"openai":{"responseId":"resp_0","serviceTier":"default"}}',
                },
                "vercel.ai.response.timestamp": {
                    "type": "string",
                    "value": "2025-11-04T11:47:02.000Z",
                },
                "vercel.ai.settings.maxRetries": {"type": "integer", "value": 2},
                "vercel.ai.streaming": {"type": "boolean", "value": False},
                "vercel.ai.telemetry.functionId": {
                    "type": "string",
                    "value": "weather-chat",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "gen_ai.generate_text",
            "organization_id": 1,
            "parent_span_id": "13e7c1ffd66981f0",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "af7b547f4d5f4f49",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "http.request.method": {"type": "string", "value": "POST"},
                "http.request.method_original": {"type": "string", "value": "POST"},
                "http.response.status_code": {"type": "integer", "value": 200},
                "network.peer.address": {"type": "string", "value": "162.159.140.245"},
                "network.peer.port": {"type": "integer", "value": 443},
                "otel.kind": {"type": "string", "value": "CLIENT"},
                "sentry.category": {"type": "string", "value": "http"},
                "sentry.description": {
                    "type": "string",
                    "value": "POST " "https://api.openai.com/v1/responses",
                },
                "sentry.domain": {"type": "string", "value": "*.openai.com"},
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 500.0},
                "sentry.group": {"type": "string", "value": "483aa47139350517"},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.normalized_description": {
                    "type": "string",
                    "value": "POST " "https://*.openai.com",
                },
                "sentry.op": {"type": "string", "value": "http.client"},
                "sentry.origin": {
                    "type": "string",
                    "value": "auto.http.otel.node_fetch",
                },
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.status_code": {"type": "string", "value": "200"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "server.address": {"type": "string", "value": "api.openai.com"},
                "server.port": {"type": "integer", "value": 443},
                "url": {
                    "type": "string",
                    "value": "https://api.openai.com/v1/responses",
                },
                "url.full": {
                    "type": "string",
                    "value": "https://api.openai.com/v1/responses",
                },
                "url.path": {"type": "string", "value": "/v1/responses"},
                "url.scheme": {"type": "string", "value": "https"},
                "user_agent.original": {
                    "type": "string",
                    "value": "ai/5.0.71 "
                    "ai-sdk/provider-utils/3.0.12 "
                    "runtime/node.js/v20.19.1",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "POST",
            "organization_id": 1,
            "parent_span_id": "af7b547f4d5f4f49",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "5ff84ff6bf512012",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "gen_ai.operation.type": {"type": "string", "value": "tool"},
                "gen_ai.tool.call.id": {
                    "type": "string",
                    "value": "call_jRcFbOh5zNVeV2l2mEqpRiI4",
                },
                "gen_ai.tool.input": {
                    "type": "string",
                    "value": '{"city":"San Francisco"}',
                },
                "gen_ai.tool.name": {"type": "string", "value": "getWeather"},
                "gen_ai.tool.output": {
                    "type": "string",
                    "value": '{"location":"San Francisco, '
                    "United States of "
                    'America","temperature":"13°C '
                    '(56°F)","temperatureC":13,"temperatureF":56,"condition":"Haze","humidity":"88%","windSpeed":"4 '
                    'km/h"}',
                },
                "gen_ai.tool.type": {"type": "string", "value": "function"},
                "operation.name": {
                    "type": "string",
                    "value": "ai.toolCall weather-chat",
                },
                "resource.name": {"type": "string", "value": "weather-chat"},
                "sentry.description": {
                    "type": "string",
                    "value": "execute_tool getWeather",
                },
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 0.0},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.op": {"type": "string", "value": "gen_ai.execute_tool"},
                "sentry.origin": {"type": "string", "value": "auto.vercelai.otel"},
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "vercel.ai.operationId": {"type": "string", "value": "ai.toolCall"},
                "vercel.ai.telemetry.functionId": {
                    "type": "string",
                    "value": "weather-chat",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "Generative AI model operation",
            "organization_id": 1,
            "parent_span_id": "13e7c1ffd66981f0",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "d2fd68d7cf6eb933",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "http.query": {"type": "string", "value": "format=j1"},
                "http.request.method": {"type": "string", "value": "GET"},
                "http.request.method_original": {"type": "string", "value": "GET"},
                "http.response.header.content-length": {
                    "type": "integer",
                    "value": 50712,
                },
                "http.response.status_code": {"type": "integer", "value": 200},
                "network.peer.address": {"type": "string", "value": "5.9.243.187"},
                "network.peer.port": {"type": "integer", "value": 443},
                "otel.kind": {"type": "string", "value": "CLIENT"},
                "sentry.category": {"type": "string", "value": "http"},
                "sentry.description": {
                    "type": "string",
                    "value": "GET " "https://wttr.in/San%20Francisco",
                },
                "sentry.domain": {"type": "string", "value": "wttr.in"},
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 500.0},
                "sentry.group": {"type": "string", "value": "2cdcd1b2278e1dd3"},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.normalized_description": {
                    "type": "string",
                    "value": "GET " "https://wttr.in",
                },
                "sentry.op": {"type": "string", "value": "http.client"},
                "sentry.origin": {
                    "type": "string",
                    "value": "auto.http.otel.node_fetch",
                },
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.status_code": {"type": "string", "value": "200"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "server.address": {"type": "string", "value": "wttr.in"},
                "server.port": {"type": "integer", "value": 443},
                "url": {"type": "string", "value": "https://wttr.in/San%20Francisco"},
                "url.full": {
                    "type": "string",
                    "value": "https://wttr.in/San%20Francisco?format=j1",
                },
                "url.path": {"type": "string", "value": "/San%20Francisco"},
                "url.query": {"type": "string", "value": "?format=j1"},
                "url.scheme": {"type": "string", "value": "https"},
                "user_agent.original": {"type": "string", "value": "node"},
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "GET",
            "organization_id": 1,
            "parent_span_id": "d2fd68d7cf6eb933",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "672681a999129905",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "gen_ai.operation.type": {"type": "string", "value": "tool"},
                "gen_ai.tool.call.id": {
                    "type": "string",
                    "value": "call_yXdgMJXsotHOn19VdlDjrc7c",
                },
                "gen_ai.tool.input": {"type": "string", "value": '{"city":"London"}'},
                "gen_ai.tool.name": {"type": "string", "value": "getWeather"},
                "gen_ai.tool.output": {
                    "type": "string",
                    "value": '{"location":"London, United '
                    'Kingdom","temperature":"16°C '
                    '(61°F)","temperatureC":16,"temperatureF":61,"condition":"Partly '
                    'cloudy","humidity":"72%","windSpeed":"21 '
                    'km/h"}',
                },
                "gen_ai.tool.type": {"type": "string", "value": "function"},
                "operation.name": {
                    "type": "string",
                    "value": "ai.toolCall weather-chat",
                },
                "resource.name": {"type": "string", "value": "weather-chat"},
                "sentry.description": {
                    "type": "string",
                    "value": "execute_tool getWeather",
                },
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 0.0},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.op": {"type": "string", "value": "gen_ai.execute_tool"},
                "sentry.origin": {"type": "string", "value": "auto.vercelai.otel"},
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "vercel.ai.operationId": {"type": "string", "value": "ai.toolCall"},
                "vercel.ai.telemetry.functionId": {
                    "type": "string",
                    "value": "weather-chat",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "Generative AI model operation",
            "organization_id": 1,
            "parent_span_id": "13e7c1ffd66981f0",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "e5bb8f1d156e7649",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "http.query": {"type": "string", "value": "format=j1"},
                "http.request.method": {"type": "string", "value": "GET"},
                "http.request.method_original": {"type": "string", "value": "GET"},
                "http.response.header.content-length": {
                    "type": "integer",
                    "value": 50728,
                },
                "http.response.status_code": {"type": "integer", "value": 200},
                "network.peer.address": {"type": "string", "value": "5.9.243.187"},
                "network.peer.port": {"type": "integer", "value": 443},
                "otel.kind": {"type": "string", "value": "CLIENT"},
                "sentry.category": {"type": "string", "value": "http"},
                "sentry.description": {
                    "type": "string",
                    "value": "GET https://wttr.in/London",
                },
                "sentry.domain": {"type": "string", "value": "wttr.in"},
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 500.0},
                "sentry.group": {"type": "string", "value": "2cdcd1b2278e1dd3"},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.normalized_description": {
                    "type": "string",
                    "value": "GET " "https://wttr.in",
                },
                "sentry.op": {"type": "string", "value": "http.client"},
                "sentry.origin": {
                    "type": "string",
                    "value": "auto.http.otel.node_fetch",
                },
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.status_code": {"type": "string", "value": "200"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "server.address": {"type": "string", "value": "wttr.in"},
                "server.port": {"type": "integer", "value": 443},
                "url": {"type": "string", "value": "https://wttr.in/London"},
                "url.full": {
                    "type": "string",
                    "value": "https://wttr.in/London?format=j1",
                },
                "url.path": {"type": "string", "value": "/London"},
                "url.query": {"type": "string", "value": "?format=j1"},
                "url.scheme": {"type": "string", "value": "https"},
                "user_agent.original": {"type": "string", "value": "node"},
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "GET",
            "organization_id": 1,
            "parent_span_id": "e5bb8f1d156e7649",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "3d755d9884113eba",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "_meta": mock.ANY,
            "attributes": {
                "gen_ai.conversation.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                },
                "gen_ai.cost.input_tokens": {"type": "double", "value": 2.08},
                "gen_ai.cost.output_tokens": {"type": "double", "value": 0.38},
                "gen_ai.cost.total_tokens": {"type": "double", "value": 2.46},
                "gen_ai.function_id": {"type": "string", "value": "weather-chat"},
                "gen_ai.operation.type": {"type": "string", "value": "ai_client"},
                "gen_ai.request.available_tools": {
                    "type": "string",
                    "value": '["tool_1"]',
                },
                "gen_ai.request.messages": {
                    "type": "string",
                    "value": "Some AI Prompt about " "the Wheather",
                },
                "gen_ai.request.model": {"type": "string", "value": "gpt-4o"},
                "gen_ai.response.finish_reasons": {
                    "type": "string",
                    "value": '["stop"]',
                },
                "gen_ai.response.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                },
                "gen_ai.response.model": {
                    "type": "string",
                    "value": "gpt-4o-2024-08-06",
                },
                "gen_ai.response.text": {
                    "type": "string",
                    "value": "True. \n"
                    "\n"
                    "- London: 61°F \n"
                    "- San Francisco: 13°C",
                },
                "gen_ai.response.tokens_per_second": {"type": "double", "value": 38.0},
                "gen_ai.system": {"type": "string", "value": "openai.responses"},
                "gen_ai.usage.input_tokens": {"type": "integer", "value": 208},
                "gen_ai.usage.output_tokens": {"type": "integer", "value": 19},
                "gen_ai.usage.total_cost": {"type": "double", "value": 2.46},
                "gen_ai.usage.total_tokens": {"type": "integer", "value": 227},
                "operation.name": {
                    "type": "string",
                    "value": "ai.generateText.doGenerate " "weather-chat",
                },
                "resource.name": {"type": "string", "value": "weather-chat"},
                "sentry.description": {
                    "type": "string",
                    "value": "generate_text gpt-4o",
                },
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 0.0},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.op": {"type": "string", "value": "gen_ai.generate_text"},
                "sentry.origin": {"type": "string", "value": "auto.vercelai.otel"},
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "vercel.ai.model.id": {"type": "string", "value": "gpt-4o"},
                "vercel.ai.model.provider": {
                    "type": "string",
                    "value": "openai.responses",
                },
                "vercel.ai.operationId": {
                    "type": "string",
                    "value": "ai.generateText.doGenerate",
                },
                "vercel.ai.pipeline.name": {
                    "type": "string",
                    "value": "generateText.doGenerate",
                },
                "vercel.ai.prompt.toolChoice": {
                    "type": "string",
                    "value": '{"type":"auto"}',
                },
                "vercel.ai.request.headers.user-agent": {
                    "type": "string",
                    "value": "ai/5.0.71",
                },
                "vercel.ai.response.finishReason": {"type": "string", "value": "stop"},
                "vercel.ai.response.id": {
                    "type": "string",
                    "value": "resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1",
                },
                "vercel.ai.response.model": {
                    "type": "string",
                    "value": "gpt-4o-2024-08-06",
                },
                "vercel.ai.response.providerMetadata": {
                    "type": "string",
                    "value": '{"openai":{"responseId":"resp_0c1c943ef2dc8bf9006909e7b8e3e88197bffb4d0e80187ca1","serviceTier":"default"}}',
                },
                "vercel.ai.response.timestamp": {
                    "type": "string",
                    "value": "2025-11-04T11:47:04.000Z",
                },
                "vercel.ai.settings.maxRetries": {"type": "integer", "value": 2},
                "vercel.ai.streaming": {"type": "boolean", "value": False},
                "vercel.ai.telemetry.functionId": {
                    "type": "string",
                    "value": "weather-chat",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "gen_ai.generate_text",
            "organization_id": 1,
            "parent_span_id": "13e7c1ffd66981f0",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "bdf1648756367ee5",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "http.request.method": {"type": "string", "value": "POST"},
                "http.request.method_original": {"type": "string", "value": "POST"},
                "http.response.status_code": {"type": "integer", "value": 200},
                "network.peer.address": {"type": "string", "value": "162.159.140.245"},
                "network.peer.port": {"type": "integer", "value": 443},
                "otel.kind": {"type": "string", "value": "CLIENT"},
                "sentry.category": {"type": "string", "value": "http"},
                "sentry.description": {
                    "type": "string",
                    "value": "POST " "https://api.openai.com/v1/responses",
                },
                "sentry.domain": {"type": "string", "value": "*.openai.com"},
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 500.0},
                "sentry.group": {"type": "string", "value": "483aa47139350517"},
                "sentry.is_remote": {"type": "boolean", "value": False},
                "sentry.normalized_description": {
                    "type": "string",
                    "value": "POST " "https://*.openai.com",
                },
                "sentry.op": {"type": "string", "value": "http.client"},
                "sentry.origin": {
                    "type": "string",
                    "value": "auto.http.otel.node_fetch",
                },
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.status_code": {"type": "string", "value": "200"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "server.address": {"type": "string", "value": "api.openai.com"},
                "server.port": {"type": "integer", "value": 443},
                "url": {
                    "type": "string",
                    "value": "https://api.openai.com/v1/responses",
                },
                "url.full": {
                    "type": "string",
                    "value": "https://api.openai.com/v1/responses",
                },
                "url.path": {"type": "string", "value": "/v1/responses"},
                "url.scheme": {"type": "string", "value": "https"},
                "user_agent.original": {
                    "type": "string",
                    "value": "ai/5.0.71 "
                    "ai-sdk/provider-utils/3.0.12 "
                    "runtime/node.js/v20.19.1",
                },
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": False,
            "key_id": 123,
            "name": "POST",
            "organization_id": 1,
            "parent_span_id": "bdf1648756367ee5",
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "203e925b464ad87b",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
        {
            "attributes": {
                "sentry.description": {"type": "string", "value": "main"},
                "sentry.environment": {"type": "string", "value": "production"},
                "sentry.exclusive_time": {"type": "double", "value": 0.0},
                "sentry.is_remote": {"type": "boolean", "value": True},
                "sentry.op": {"type": "string", "value": "default"},
                "sentry.origin": {"type": "string", "value": "manual"},
                "sentry.platform": {"type": "string", "value": "node"},
                "sentry.sample_rate": {"type": "integer", "value": 1},
                "sentry.sdk.name": {"type": "string", "value": "raven-node"},
                "sentry.sdk.version": {"type": "string", "value": "2.6.3"},
                "sentry.segment.id": {"type": "string", "value": "657cf984a6a4e59b"},
                "sentry.segment.name": {"type": "string", "value": "main"},
                "sentry.source": {"type": "string", "value": "custom"},
                "sentry.status": {"type": "string", "value": "ok"},
                "sentry.trace.status": {"type": "string", "value": "ok"},
                "sentry.transaction": {"type": "string", "value": "main"},
                "sentry.was_transaction": {"type": "boolean", "value": True},
                "server_name": {"type": "string", "value": "some_machine.local"},
            },
            "downsampled_retention_days": 90,
            "end_timestamp": time_within_delta(),
            "event_id": mock.ANY,
            "is_segment": True,
            "key_id": 123,
            "name": "default",
            "organization_id": 1,
            "project_id": 42,
            "received": time_within_delta(),
            "retention_days": 90,
            "span_id": "657cf984a6a4e59b",
            "start_timestamp": time_within_delta(),
            "status": "ok",
            "trace_id": "a9351cd574f092f6acad48e250981f11",
        },
    ]
