# File generated from our OpenAPI spec by Stainless. See CONTRIBUTING.md for details.

from __future__ import annotations

import os
from typing import Any, Union, Mapping
from typing_extensions import Self, override

import httpx

from . import _exceptions
from ._qs import Querystring
from ._types import (
    NOT_GIVEN,
    Omit,
    Timeout,
    NotGiven,
    Transport,
    ProxiesTypes,
    RequestOptions,
)
from ._utils import is_given, get_async_library
from ._version import __version__
from .resources import ai, apis, events
from ._streaming import Stream as Stream, AsyncStream as AsyncStream
from ._exceptions import APIStatusError, ReveniumMeteringError
from ._base_client import (
    DEFAULT_MAX_RETRIES,
    SyncAPIClient,
    AsyncAPIClient,
)

__all__ = [
    "Timeout",
    "Transport",
    "ProxiesTypes",
    "RequestOptions",
    "ReveniumMetering",
    "AsyncReveniumMetering",
    "Client",
    "AsyncClient",
]


class ReveniumMetering(SyncAPIClient):
    events: events.EventsResource
    apis: apis.APIsResource
    ai: ai.AIResource
    with_raw_response: ReveniumMeteringWithRawResponse
    with_streaming_response: ReveniumMeteringWithStreamedResponse

    # client options
    api_key: str

    def __init__(
        self,
        *,
        api_key: str | None = None,
        base_url: str | httpx.URL | None = None,
        timeout: Union[float, Timeout, None, NotGiven] = NOT_GIVEN,
        max_retries: int = DEFAULT_MAX_RETRIES,
        default_headers: Mapping[str, str] | None = None,
        default_query: Mapping[str, object] | None = None,
        # Configure a custom httpx client.
        # We provide a `DefaultHttpxClient` class that you can pass to retain the default values we use for `limits`, `timeout` & `follow_redirects`.
        # See the [httpx documentation](https://www.python-httpx.org/api/#client) for more details.
        http_client: httpx.Client | None = None,
        # Enable or disable schema validation for data returned by the API.
        # When enabled an error APIResponseValidationError is raised
        # if the API responds with invalid data for the expected schema.
        #
        # This parameter may be removed or changed in the future.
        # If you rely on this feature, please open a GitHub issue
        # outlining your use-case to help us decide if it should be
        # part of our public interface in the future.
        _strict_response_validation: bool = False,
    ) -> None:
        """Construct a new synchronous ReveniumMetering client instance.

        This automatically infers the `api_key` argument from the `REVENIUM_METERING_API_KEY` environment variable if it is not provided.
        """
        if api_key is None:
            api_key = os.environ.get("REVENIUM_METERING_API_KEY")
        if api_key is None:
            raise ReveniumMeteringError(
                "The api_key client option must be set either by passing api_key to the client or by setting the REVENIUM_METERING_API_KEY environment variable"
            )
        self.api_key = api_key

        if base_url is None:
            base_url = os.environ.get("REVENIUM_METERING_BASE_URL")
        if base_url is None:
            base_url = f"https://api.revenium.ai/meter/"

        # Normalize the base URL to ensure it includes /meter
        base_url = self._normalize_base_url(base_url)

        super().__init__(
            version=__version__,
            base_url=base_url,
            max_retries=max_retries,
            timeout=timeout,
            http_client=http_client,
            custom_headers=default_headers,
            custom_query=default_query,
            _strict_response_validation=_strict_response_validation,
        )

        self.events = events.EventsResource(self)
        self.apis = apis.APIsResource(self)
        self.ai = ai.AIResource(self)
        self.with_raw_response = ReveniumMeteringWithRawResponse(self)
        self.with_streaming_response = ReveniumMeteringWithStreamedResponse(self)

    @staticmethod
    def _normalize_base_url(url: str | httpx.URL) -> str:
        """
        Normalize the base URL to ensure it includes the /meter path segment.

        This method handles both input scenarios:
        - If the URL doesn't include /meter, it will be added
        - If the URL already includes /meter, it won't be duplicated

        Args:
            url: The base URL to normalize (can be a string or httpx.URL)

        Returns:
            The normalized base URL as a string with /meter path segment
        """
        # Convert to string if it's an httpx.URL
        url_str = str(url)

        # Remove trailing slash for consistent processing
        url_str = url_str.rstrip('/')

        # Parse the URL to extract components
        parsed = httpx.URL(url_str)
        path = parsed.raw_path.decode('utf-8').rstrip('/')

        # Check if /meter is already in the path (case-insensitive)
        path_lower = path.lower()

        # If /meter is not present, add it
        if not path_lower.endswith('/meter'):
            # Check if there's a /meter somewhere in the middle (shouldn't happen, but handle it)
            if '/meter' in path_lower:
                # Find the position of /meter and keep everything up to and including it
                meter_index = path_lower.rfind('/meter')
                path = path[:meter_index + 6]  # 6 is the length of '/meter'
            else:
                # Add /meter to the path
                path = f"{path}/meter"

        # Reconstruct the URL with the normalized path and trailing slash
        normalized_url = parsed.copy_with(raw_path=f"{path}/".encode('utf-8'))

        return str(normalized_url)

    @property
    @override
    def qs(self) -> Querystring:
        return Querystring(array_format="comma")

    @property
    @override
    def auth_headers(self) -> dict[str, str]:
        api_key = self.api_key
        return {"x-api-key": api_key}

    @property
    @override
    def default_headers(self) -> dict[str, str | Omit]:
        return {
            **super().default_headers,
            "X-Stainless-Async": "false",
            **self._custom_headers,
        }

    def copy(
        self,
        *,
        api_key: str | None = None,
        base_url: str | httpx.URL | None = None,
        timeout: float | Timeout | None | NotGiven = NOT_GIVEN,
        http_client: httpx.Client | None = None,
        max_retries: int | NotGiven = NOT_GIVEN,
        default_headers: Mapping[str, str] | None = None,
        set_default_headers: Mapping[str, str] | None = None,
        default_query: Mapping[str, object] | None = None,
        set_default_query: Mapping[str, object] | None = None,
        _extra_kwargs: Mapping[str, Any] = {},
    ) -> Self:
        """
        Create a new client instance re-using the same options given to the current client with optional overriding.
        """
        if default_headers is not None and set_default_headers is not None:
            raise ValueError("The `default_headers` and `set_default_headers` arguments are mutually exclusive")

        if default_query is not None and set_default_query is not None:
            raise ValueError("The `default_query` and `set_default_query` arguments are mutually exclusive")

        headers = self._custom_headers
        if default_headers is not None:
            headers = {**headers, **default_headers}
        elif set_default_headers is not None:
            headers = set_default_headers

        params = self._custom_query
        if default_query is not None:
            params = {**params, **default_query}
        elif set_default_query is not None:
            params = set_default_query

        http_client = http_client or self._client
        return self.__class__(
            api_key=api_key or self.api_key,
            base_url=base_url or self.base_url,
            timeout=self.timeout if isinstance(timeout, NotGiven) else timeout,
            http_client=http_client,
            max_retries=max_retries if is_given(max_retries) else self.max_retries,
            default_headers=headers,
            default_query=params,
            **_extra_kwargs,
        )

    # Alias for `copy` for nicer inline usage, e.g.
    # client.with_options(timeout=10).foo.create(...)
    with_options = copy

    @override
    def _make_status_error(
        self,
        err_msg: str,
        *,
        body: object,
        response: httpx.Response,
    ) -> APIStatusError:
        if response.status_code == 400:
            return _exceptions.BadRequestError(err_msg, response=response, body=body)

        if response.status_code == 401:
            return _exceptions.AuthenticationError(err_msg, response=response, body=body)

        if response.status_code == 403:
            return _exceptions.PermissionDeniedError(err_msg, response=response, body=body)

        if response.status_code == 404:
            return _exceptions.NotFoundError(err_msg, response=response, body=body)

        if response.status_code == 409:
            return _exceptions.ConflictError(err_msg, response=response, body=body)

        if response.status_code == 422:
            return _exceptions.UnprocessableEntityError(err_msg, response=response, body=body)

        if response.status_code == 429:
            return _exceptions.RateLimitError(err_msg, response=response, body=body)

        if response.status_code >= 500:
            return _exceptions.InternalServerError(err_msg, response=response, body=body)
        return APIStatusError(err_msg, response=response, body=body)


class AsyncReveniumMetering(AsyncAPIClient):
    events: events.AsyncEventsResource
    apis: apis.AsyncAPIsResource
    ai: ai.AsyncAIResource
    with_raw_response: AsyncReveniumMeteringWithRawResponse
    with_streaming_response: AsyncReveniumMeteringWithStreamedResponse

    # client options
    api_key: str

    def __init__(
        self,
        *,
        api_key: str | None = None,
        base_url: str | httpx.URL | None = None,
        timeout: Union[float, Timeout, None, NotGiven] = NOT_GIVEN,
        max_retries: int = DEFAULT_MAX_RETRIES,
        default_headers: Mapping[str, str] | None = None,
        default_query: Mapping[str, object] | None = None,
        # Configure a custom httpx client.
        # We provide a `DefaultAsyncHttpxClient` class that you can pass to retain the default values we use for `limits`, `timeout` & `follow_redirects`.
        # See the [httpx documentation](https://www.python-httpx.org/api/#asyncclient) for more details.
        http_client: httpx.AsyncClient | None = None,
        # Enable or disable schema validation for data returned by the API.
        # When enabled an error APIResponseValidationError is raised
        # if the API responds with invalid data for the expected schema.
        #
        # This parameter may be removed or changed in the future.
        # If you rely on this feature, please open a GitHub issue
        # outlining your use-case to help us decide if it should be
        # part of our public interface in the future.
        _strict_response_validation: bool = False,
    ) -> None:
        """Construct a new async AsyncReveniumMetering client instance.

        This automatically infers the `api_key` argument from the `REVENIUM_METERING_API_KEY` environment variable if it is not provided.
        """
        if api_key is None:
            api_key = os.environ.get("REVENIUM_METERING_API_KEY")
        if api_key is None:
            raise ReveniumMeteringError(
                "The api_key client option must be set either by passing api_key to the client or by setting the REVENIUM_METERING_API_KEY environment variable"
            )
        self.api_key = api_key

        if base_url is None:
            base_url = os.environ.get("REVENIUM_METERING_BASE_URL")
        if base_url is None:
            base_url = f"https://api.revenium.ai/meter/"

        # Normalize the base URL to ensure it includes /meter
        base_url = self._normalize_base_url(base_url)

        super().__init__(
            version=__version__,
            base_url=base_url,
            max_retries=max_retries,
            timeout=timeout,
            http_client=http_client,
            custom_headers=default_headers,
            custom_query=default_query,
            _strict_response_validation=_strict_response_validation,
        )

        self.events = events.AsyncEventsResource(self)
        self.apis = apis.AsyncAPIsResource(self)
        self.ai = ai.AsyncAIResource(self)
        self.with_raw_response = AsyncReveniumMeteringWithRawResponse(self)
        self.with_streaming_response = AsyncReveniumMeteringWithStreamedResponse(self)

    @staticmethod
    def _normalize_base_url(url: str | httpx.URL) -> str:
        """
        Normalize the base URL to ensure it includes the /meter path segment.

        This method handles both input scenarios:
        - If the URL doesn't include /meter, it will be added
        - If the URL already includes /meter, it won't be duplicated

        Args:
            url: The base URL to normalize (can be a string or httpx.URL)

        Returns:
            The normalized base URL as a string with /meter path segment
        """
        # Convert to string if it's an httpx.URL
        url_str = str(url)

        # Remove trailing slash for consistent processing
        url_str = url_str.rstrip('/')

        # Parse the URL to extract components
        parsed = httpx.URL(url_str)
        path = parsed.raw_path.decode('utf-8').rstrip('/')

        # Check if /meter is already in the path (case-insensitive)
        path_lower = path.lower()

        # If /meter is not present, add it
        if not path_lower.endswith('/meter'):
            # Check if there's a /meter somewhere in the middle (shouldn't happen, but handle it)
            if '/meter' in path_lower:
                # Find the position of /meter and keep everything up to and including it
                meter_index = path_lower.rfind('/meter')
                path = path[:meter_index + 6]  # 6 is the length of '/meter'
            else:
                # Add /meter to the path
                path = f"{path}/meter"

        # Reconstruct the URL with the normalized path and trailing slash
        normalized_url = parsed.copy_with(raw_path=f"{path}/".encode('utf-8'))

        return str(normalized_url)

    @property
    @override
    def qs(self) -> Querystring:
        return Querystring(array_format="comma")

    @property
    @override
    def auth_headers(self) -> dict[str, str]:
        api_key = self.api_key
        return {"x-api-key": api_key}

    @property
    @override
    def default_headers(self) -> dict[str, str | Omit]:
        return {
            **super().default_headers,
            "X-Stainless-Async": f"async:{get_async_library()}",
            **self._custom_headers,
        }

    def copy(
        self,
        *,
        api_key: str | None = None,
        base_url: str | httpx.URL | None = None,
        timeout: float | Timeout | None | NotGiven = NOT_GIVEN,
        http_client: httpx.AsyncClient | None = None,
        max_retries: int | NotGiven = NOT_GIVEN,
        default_headers: Mapping[str, str] | None = None,
        set_default_headers: Mapping[str, str] | None = None,
        default_query: Mapping[str, object] | None = None,
        set_default_query: Mapping[str, object] | None = None,
        _extra_kwargs: Mapping[str, Any] = {},
    ) -> Self:
        """
        Create a new client instance re-using the same options given to the current client with optional overriding.
        """
        if default_headers is not None and set_default_headers is not None:
            raise ValueError("The `default_headers` and `set_default_headers` arguments are mutually exclusive")

        if default_query is not None and set_default_query is not None:
            raise ValueError("The `default_query` and `set_default_query` arguments are mutually exclusive")

        headers = self._custom_headers
        if default_headers is not None:
            headers = {**headers, **default_headers}
        elif set_default_headers is not None:
            headers = set_default_headers

        params = self._custom_query
        if default_query is not None:
            params = {**params, **default_query}
        elif set_default_query is not None:
            params = set_default_query

        http_client = http_client or self._client
        return self.__class__(
            api_key=api_key or self.api_key,
            base_url=base_url or self.base_url,
            timeout=self.timeout if isinstance(timeout, NotGiven) else timeout,
            http_client=http_client,
            max_retries=max_retries if is_given(max_retries) else self.max_retries,
            default_headers=headers,
            default_query=params,
            **_extra_kwargs,
        )

    # Alias for `copy` for nicer inline usage, e.g.
    # client.with_options(timeout=10).foo.create(...)
    with_options = copy

    @override
    def _make_status_error(
        self,
        err_msg: str,
        *,
        body: object,
        response: httpx.Response,
    ) -> APIStatusError:
        if response.status_code == 400:
            return _exceptions.BadRequestError(err_msg, response=response, body=body)

        if response.status_code == 401:
            return _exceptions.AuthenticationError(err_msg, response=response, body=body)

        if response.status_code == 403:
            return _exceptions.PermissionDeniedError(err_msg, response=response, body=body)

        if response.status_code == 404:
            return _exceptions.NotFoundError(err_msg, response=response, body=body)

        if response.status_code == 409:
            return _exceptions.ConflictError(err_msg, response=response, body=body)

        if response.status_code == 422:
            return _exceptions.UnprocessableEntityError(err_msg, response=response, body=body)

        if response.status_code == 429:
            return _exceptions.RateLimitError(err_msg, response=response, body=body)

        if response.status_code >= 500:
            return _exceptions.InternalServerError(err_msg, response=response, body=body)
        return APIStatusError(err_msg, response=response, body=body)


class ReveniumMeteringWithRawResponse:
    def __init__(self, client: ReveniumMetering) -> None:
        self.events = events.EventsResourceWithRawResponse(client.events)
        self.apis = apis.APIsResourceWithRawResponse(client.apis)
        self.ai = ai.AIResourceWithRawResponse(client.ai)


class AsyncReveniumMeteringWithRawResponse:
    def __init__(self, client: AsyncReveniumMetering) -> None:
        self.events = events.AsyncEventsResourceWithRawResponse(client.events)
        self.apis = apis.AsyncAPIsResourceWithRawResponse(client.apis)
        self.ai = ai.AsyncAIResourceWithRawResponse(client.ai)


class ReveniumMeteringWithStreamedResponse:
    def __init__(self, client: ReveniumMetering) -> None:
        self.events = events.EventsResourceWithStreamingResponse(client.events)
        self.apis = apis.APIsResourceWithStreamingResponse(client.apis)
        self.ai = ai.AIResourceWithStreamingResponse(client.ai)


class AsyncReveniumMeteringWithStreamedResponse:
    def __init__(self, client: AsyncReveniumMetering) -> None:
        self.events = events.AsyncEventsResourceWithStreamingResponse(client.events)
        self.apis = apis.AsyncAPIsResourceWithStreamingResponse(client.apis)
        self.ai = ai.AsyncAIResourceWithStreamingResponse(client.ai)


Client = ReveniumMetering

AsyncClient = AsyncReveniumMetering
