from http import HTTPStatus
from typing import Any, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error_response import ErrorResponse
from ...models.http_validation_error import HTTPValidationError
from ...models.sso_exchange_request import SSOExchangeRequest
from ...models.sso_exchange_response import SSOExchangeResponse
from ...types import Response


def _get_kwargs(
  *,
  body: SSOExchangeRequest,
) -> dict[str, Any]:
  headers: dict[str, Any] = {}

  _kwargs: dict[str, Any] = {
    "method": "post",
    "url": "/v1/auth/sso-exchange",
  }

  _kwargs["json"] = body.to_dict()

  headers["Content-Type"] = "application/json"

  _kwargs["headers"] = headers
  return _kwargs


def _parse_response(
  *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]:
  if response.status_code == 200:
    response_200 = SSOExchangeResponse.from_dict(response.json())

    return response_200
  if response.status_code == 401:
    response_401 = ErrorResponse.from_dict(response.json())

    return response_401
  if response.status_code == 400:
    response_400 = ErrorResponse.from_dict(response.json())

    return response_400
  if response.status_code == 422:
    response_422 = HTTPValidationError.from_dict(response.json())

    return response_422
  if client.raise_on_unexpected_status:
    raise errors.UnexpectedStatus(response.status_code, response.content)
  else:
    return None


def _build_response(
  *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]:
  return Response(
    status_code=HTTPStatus(response.status_code),
    content=response.content,
    headers=response.headers,
    parsed=_parse_response(client=client, response=response),
  )


def sync_detailed(
  *,
  client: Union[AuthenticatedClient, Client],
  body: SSOExchangeRequest,
) -> Response[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]:
  """SSO Token Exchange

   Exchange SSO token for secure session handoff to target application.

  Args:
      body (SSOExchangeRequest): SSO token exchange request model.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Response[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]
  """

  kwargs = _get_kwargs(
    body=body,
  )

  response = client.get_httpx_client().request(
    **kwargs,
  )

  return _build_response(client=client, response=response)


def sync(
  *,
  client: Union[AuthenticatedClient, Client],
  body: SSOExchangeRequest,
) -> Optional[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]:
  """SSO Token Exchange

   Exchange SSO token for secure session handoff to target application.

  Args:
      body (SSOExchangeRequest): SSO token exchange request model.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]
  """

  return sync_detailed(
    client=client,
    body=body,
  ).parsed


async def asyncio_detailed(
  *,
  client: Union[AuthenticatedClient, Client],
  body: SSOExchangeRequest,
) -> Response[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]:
  """SSO Token Exchange

   Exchange SSO token for secure session handoff to target application.

  Args:
      body (SSOExchangeRequest): SSO token exchange request model.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Response[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]
  """

  kwargs = _get_kwargs(
    body=body,
  )

  response = await client.get_async_httpx_client().request(**kwargs)

  return _build_response(client=client, response=response)


async def asyncio(
  *,
  client: Union[AuthenticatedClient, Client],
  body: SSOExchangeRequest,
) -> Optional[Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]]:
  """SSO Token Exchange

   Exchange SSO token for secure session handoff to target application.

  Args:
      body (SSOExchangeRequest): SSO token exchange request model.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Union[ErrorResponse, HTTPValidationError, SSOExchangeResponse]
  """

  return (
    await asyncio_detailed(
      client=client,
      body=body,
    )
  ).parsed
