import { Construct } from "constructs";
import { TerraformRemoteState } from "../terraform-remote-state";
import { TerraformBackend } from "../terraform-backend";
/**
 * checks whether the given hostname belongs to tfc or (else) to tfe
 * If no hostname is given, it will return tfc, as that's the default in backends.
 * @param hostname e.g. app.terraform.io, app.terraform.io:80, tfe.myorg.org
 * @returns "tfc" or "tfe"
 */
export declare function getHostNameType(hostname?: string): "tfc" | "tfe";
/**
 * The Cloud Backend synthesizes a {@link https://developer.hashicorp.com/terraform/cli/cloud/settings#the-cloud-block cloud block}.
 * The cloud block is a nested block within the top-level terraform settings block.
 * It specifies which Terraform Cloud workspaces to use for the current working directory.
 * The cloud block only affects Terraform CLI's behavior.
 * When Terraform Cloud uses a configuration that contains a cloud block - for example, when a workspace is configured to use a VCS provider directly - it ignores the block and behaves according to its own workspace settings.
 */
export declare class CloudBackend extends TerraformBackend {
    private readonly props;
    constructor(scope: Construct, props: CloudBackendConfig);
    /**
     * Adds this resource to the terraform JSON output.
     */
    toTerraform(): any;
    toMetadata(): any;
    protected synthesizeAttributes(): {
        [name: string]: any;
    };
    getRemoteStateDataSource(scope: Construct, name: string, _fromStack: string): TerraformRemoteState;
}
/**
 * The Cloud Backend synthesizes a {@link https://developer.hashicorp.com/terraform/cli/cloud/settings#the-cloud-block cloud block}.
 * The cloud block is a nested block within the top-level terraform settings block.
 * It specifies which Terraform Cloud workspaces to use for the current working directory.
 * The cloud block only affects Terraform CLI's behavior.
 * When Terraform Cloud uses a configuration that contains a cloud block - for example, when a workspace is configured to use a VCS provider directly - it ignores the block and behaves according to its own workspace settings.
 *
 * https://developer.hashicorp.com/terraform/cli/cloud/settings#arguments
 */
export interface CloudBackendConfig {
    /**
     * The name of the organization containing the workspace(s) the current configuration should use.
     */
    readonly organization: string;
    /**
     * A nested block that specifies which remote Terraform Cloud workspaces to use for the current configuration.
     * The workspaces block must contain exactly one of the following arguments, each denoting a strategy for how workspaces should be mapped:
     */
    readonly workspaces: NamedCloudWorkspace | TaggedCloudWorkspaces;
    /**
     * The hostname of a Terraform Enterprise installation, if using Terraform Enterprise.
     *
     * @default app.terraform.io
     */
    readonly hostname?: string;
    /**
     * The token used to authenticate with Terraform Cloud.
     * We recommend omitting the token from the configuration, and instead using terraform login or manually configuring credentials in the CLI config file.
     */
    readonly token?: string;
}
/**
 * A cloud workspace can either be a single named workspace, or a list of tagged workspaces.
 */
export declare abstract class CloudWorkspace {
    abstract toTerraform(): any;
}
/**
 * The name of a single Terraform Cloud workspace.
 * You will only be able to use the workspace specified in the configuration with this working directory, and cannot manage workspaces from the CLI (e.g. terraform workspace select or terraform workspace new).
 */
export declare class NamedCloudWorkspace extends CloudWorkspace {
    name: string;
    constructor(name: string);
    toTerraform(): any;
}
/**
 *  A set of Terraform Cloud workspace tags. You will be able to use this working directory with any workspaces that have all of the specified tags, and can use the terraform workspace commands to switch between them or create new workspaces. New workspaces will automatically have the specified tags. This option conflicts with name.
 */
export declare class TaggedCloudWorkspaces extends CloudWorkspace {
    tags: string[];
    constructor(tags: string[]);
    toTerraform(): any;
}
