import { IResolvable, IResolveContext } from "./tokens/resolvable";
import { Intrinsic } from "./tokens/private/intrinsic";
import { TerraformStack } from "./terraform-stack";
import { ITerraformDependable } from "./terraform-dependable";
declare class TFExpression extends Intrinsic implements IResolvable {
    protected resolveArg(context: IResolveContext, arg: any): string;
    /**
     * Escape string removes characters from the string that are not allowed in Terraform or JSON
     * It must only be used on non-token values
     */
    protected escapeString(str: string): string;
    private resolveString;
}
export declare function rawString(str: string): IResolvable;
declare class Reference extends TFExpression {
    private identifier;
    private originStack?;
    /**
     * A single reference could be used in multiple stacks,
     * e.g. if we expose the ref directly or as token on the stack.
     * We need to store the identifier for each stack,
     * so that the resolved identifier string matches the stack it's resolved in.
     */
    private crossStackIdentifier;
    constructor(identifier: string, originStack?: TerraformStack | undefined);
    resolve(context: IResolveContext): string;
}
export declare function ref(identifier: string, stack?: TerraformStack): IResolvable;
/**
 * marks the argument as being used in a terraform expression
 * @deprecated
 */
export declare function insideTfExpression(arg: any): any;
declare class PropertyAccess extends TFExpression {
    private target;
    private args;
    constructor(target: Expression, args: Expression[]);
    resolve(context: IResolveContext): string;
}
export declare function propertyAccess(target: Expression, args: Expression[]): IResolvable;
declare class ConditionalExpression extends TFExpression {
    private condition;
    private trueValue;
    private falseValue;
    constructor(condition: Expression, trueValue: Expression, falseValue: Expression);
    resolve(context: IResolveContext): string;
}
/**
 * creates an expression like `true ? 1 : 0`
 */
export declare function conditional(condition: Expression, trueValue: Expression, falseValue: Expression): ConditionalExpression;
export type Operator = "!" | "-" | "*" | "/" | "%" | "+" | ">" | ">=" | "<" | "<=" | "==" | "!=" | "&&" | "||";
/**
 * Base class for Operator Expressions
 * @internal
 */
export declare class OperatorExpression extends TFExpression {
    private operator;
    private left;
    private right?;
    constructor(operator: Operator, left: Expression, right?: any);
    resolve(context: IResolveContext): string;
}
declare class FunctionCall extends TFExpression {
    private name;
    private args;
    constructor(name: string, args: Expression[]);
    resolve(context: IResolveContext): string;
}
export declare function call(name: string, args: Expression[]): IResolvable;
export declare const FOR_EXPRESSION_KEY: IResolvable;
export declare const FOR_EXPRESSION_VALUE: IResolvable;
/**
 * https://developer.hashicorp.com/terraform/language/expressions/for
 */
declare class ForExpression extends TFExpression {
    private input;
    private valueExpression;
    private keyExpression?;
    constructor(input: Expression, valueExpression: Expression, keyExpression?: any);
    resolve(context: IResolveContext): string;
}
/**
 * Creates a for expression, used for advanced internal use cases (e.g. in the @cdktf/aws-adapter)
 * It is currently not possible to use this via JSII in other languages. Usually you might want to
 * use TerraformIterator instead.
 *
 * @param input the input for the expression e.g. a list or an object
 * @param valueExpression will express the value e.g. [ for i in &lt;input&gt; : &lt;valueExpression&gt; ]
 * @param keyExpression if set will return an object e.g. { for k,v in &lt;input&gt; : &lt;keyExpression&gt; => &lt;valueExpression&gt;}
 * @returns a for expression
 *
 * The variables for key (for maps) / index (for lists) and value that are valid in the key- and valueExpression
 * can be referenced by using these constants: FOR_EXPRESSION_KEY and FOR_EXPRESSION_VALUE.
 *
 * Currently nesting for expressions is not supported due to this simplification.
 *
 * Currently does not support filtering elements (if clause) or grouping results
 */
export declare function forExpression(input: Expression, valueExpression: Expression, keyExpression?: Expression): IResolvable;
declare class Dependable extends TFExpression {
    private dependable;
    constructor(dependable: ITerraformDependable);
    resolve(context: IResolveContext): string;
}
export declare function dependable(dependable: ITerraformDependable): string;
export type Expression = Reference | FunctionCall | PropertyAccess | ConditionalExpression | OperatorExpression | Dependable | ForExpression | string | string[] | number | boolean | IResolvable | {
    [key: string]: any;
} | null | any | any[];
export {};
