import { IFragmentConcatenator, IResolvable } from "../resolvable";
import { TokenizedStringFragments } from "../string-fragments";
type LookupFunction = (id: string) => IResolvable | undefined;
export declare const BEGIN_STRING_TOKEN_MARKER = "${TfToken[";
export declare const BEGIN_LIST_TOKEN_MARKER = "#{TfToken[";
export declare const BEGIN_MAP_TOKEN_MARKER = "&{TfToken[";
export declare const END_TOKEN_MARKER = "]}";
export declare const VALID_KEY_CHARS = "a-zA-Z0-9:._-";
/**
 * A string with markers in it that can be resolved to external values
 */
export declare class TokenString {
    private readonly str;
    private readonly re;
    private readonly regexMatchIndex;
    private readonly includeEscapeSequences;
    private readonly warnIfEscapeSequences;
    /**
     * Returns a `TokenString` for this string.
     *
     * @param s The string to tokenize
     * @param includeEscapeSequences Whether to include escape sequences in the tokenization
     * @param warnIfEscapeSequences Whether to warn if escape sequences are found
     */
    static forString(s: string, includeEscapeSequences?: boolean, warnIfEscapeSequences?: boolean): TokenString;
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s: string): TokenString;
    /**
     * Returns a `TokenString` for this string that handles encoded numbers
     */
    static forNumbers(s: string): TokenString;
    /**
     * Returns a `TokenString` for this string that handles encoded maps
     */
    static forMapToken(s: string): TokenString;
    constructor(str: string, re: RegExp, regexMatchIndex?: number, includeEscapeSequences?: boolean, warnIfEscapeSequences?: boolean);
    private testForEscapeTokens;
    private nextEscapeToken;
    private tokenizeNext;
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup: LookupFunction): TokenizedStringFragments;
    /**
     * Indicates if this string includes tokens.
     */
    test(): boolean;
}
/**
 * Quote a string for use in a regex
 */
export declare function regexQuote(s: string): string;
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
export declare class NullConcat implements IFragmentConcatenator {
    join(_left: any | undefined, _right: any | undefined): any;
}
/**
 * Returns true if argument contains a string list token
 */
export declare function containsStringListTokenElement(xs: any[]): boolean;
/**
 * Returns true if argument contains a number list token
 */
export declare function containsNumberListTokenElement(xs: any[]): boolean;
/**
 * Returns true if argument contains a string map token
 */
export declare function containsMapToken(xs: {
    [key: string]: any;
}): boolean;
/**
 * Returns true if argument is a complex element
 */
export declare function isComplexElement(xs: any): boolean;
/**
 * Returns true if list contains a complex element
 */
export declare function containsComplexElement(xs: any): boolean;
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
export declare function unresolved(obj: any): boolean;
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
export declare function createTokenDouble(x: number, list: boolean): number;
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
export declare function extractTokenDouble(encoded: number, list: boolean): number | undefined;
export {};
