"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaggedCloudWorkspaces = exports.NamedCloudWorkspace = exports.CloudWorkspace = exports.CloudBackend = exports.getHostNameType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const util_1 = require("../util");
const remote_backend_1 = require("./remote-backend");
const terraform_backend_1 = require("../terraform-backend");
const validations_1 = require("../validations");
/**
 * checks whether the given hostname belongs to tfc or (else) to tfe
 * If no hostname is given, it will return tfc, as that's the default in backends.
 * @param hostname e.g. app.terraform.io, app.terraform.io:80, tfe.myorg.org
 * @returns "tfc" or "tfe"
 */
function getHostNameType(hostname) {
    if (!hostname)
        return "tfc"; // default is tfc when not passing a hostname to backends
    return hostname.startsWith("app.terraform.io") ? "tfc" : "tfe";
}
exports.getHostNameType = getHostNameType;
/**
 * The Cloud Backend synthesizes a {@link https://developer.hashicorp.com/terraform/cli/cloud/settings#the-cloud-block cloud block}.
 * The cloud block is a nested block within the top-level terraform settings block.
 * It specifies which Terraform Cloud workspaces to use for the current working directory.
 * The cloud block only affects Terraform CLI's behavior.
 * When Terraform Cloud uses a configuration that contains a cloud block - for example, when a workspace is configured to use a VCS provider directly - it ignores the block and behaves according to its own workspace settings.
 */
class CloudBackend extends terraform_backend_1.TerraformBackend {
    constructor(scope, props) {
        super(scope, "backend", "cloud");
        this.props = props;
        this.node.addValidation(new validations_1.ValidateBinaryVersion("terraform", ">=1.1", `${util_1.terraformBinaryName} version`, `The cloud block is only supported for Terraform >=1.1. Please upgrade your Terraform version.`));
    }
    /**
     * Adds this resource to the terraform JSON output.
     */
    toTerraform() {
        return {
            terraform: {
                cloud: (0, util_1.deepMerge)(this.synthesizeAttributes(), this.rawOverrides),
            },
        };
    }
    toMetadata() {
        const cloud = getHostNameType(this.props.hostname);
        return { ...super.toMetadata(), cloud };
    }
    synthesizeAttributes() {
        return (0, util_1.keysToSnakeCase)({
            ...this.props,
            workspaces: this.props.workspaces.toTerraform(),
        });
    }
    getRemoteStateDataSource(scope, name, _fromStack) {
        // The cloud "backend" does not have a coresponding remote state data source since it's meant to also
        // work with multiple workspaces through tags.
        if ("name" in this.props.workspaces) {
            return new remote_backend_1.DataTerraformRemoteState(scope, name, {
                organization: this.props.organization,
                workspaces: { name: this.props.workspaces.name },
                token: this.props.token,
                hostname: this.props.hostname,
            });
        }
        throw new Error("The Cloud backend only supports cross-stack references when the workspace is defined by name instead of by tags.");
    }
}
_a = JSII_RTTI_SYMBOL_1;
CloudBackend[_a] = { fqn: "cdktf.CloudBackend", version: "0.17.0-pre.18" };
exports.CloudBackend = CloudBackend;
/**
 * A cloud workspace can either be a single named workspace, or a list of tagged workspaces.
 */
class CloudWorkspace {
}
_b = JSII_RTTI_SYMBOL_1;
CloudWorkspace[_b] = { fqn: "cdktf.CloudWorkspace", version: "0.17.0-pre.18" };
exports.CloudWorkspace = CloudWorkspace;
/**
 * The name of a single Terraform Cloud workspace.
 * You will only be able to use the workspace specified in the configuration with this working directory, and cannot manage workspaces from the CLI (e.g. terraform workspace select or terraform workspace new).
 */
class NamedCloudWorkspace extends CloudWorkspace {
    constructor(name) {
        super();
        this.name = name;
    }
    toTerraform() {
        return {
            name: this.name,
        };
    }
}
_c = JSII_RTTI_SYMBOL_1;
NamedCloudWorkspace[_c] = { fqn: "cdktf.NamedCloudWorkspace", version: "0.17.0-pre.18" };
exports.NamedCloudWorkspace = NamedCloudWorkspace;
/**
 *  A set of Terraform Cloud workspace tags. You will be able to use this working directory with any workspaces that have all of the specified tags, and can use the terraform workspace commands to switch between them or create new workspaces. New workspaces will automatically have the specified tags. This option conflicts with name.
 */
class TaggedCloudWorkspaces extends CloudWorkspace {
    constructor(tags) {
        super();
        this.tags = tags;
    }
    toTerraform() {
        return {
            tags: this.tags,
        };
    }
}
_d = JSII_RTTI_SYMBOL_1;
TaggedCloudWorkspaces[_d] = { fqn: "cdktf.TaggedCloudWorkspaces", version: "0.17.0-pre.18" };
exports.TaggedCloudWorkspaces = TaggedCloudWorkspaces;
//# sourceMappingURL=data:application/json;base64,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