"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.terraformFunction = exports.asAny = exports.asBoolean = exports.asStringMap = exports.asList = exports.asNumber = exports.asString = exports.variadic = exports.listOf = exports.numericValue = exports.stringValue = exports.mapValue = exports.anyValue = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const token_1 = require("../tokens/token");
const tfExpression_1 = require("../tfExpression");
const encoding_1 = require("../tokens/private/encoding");
/**
 * Determines if given str has unescaped double quotes
 * @param str String to test
 * @returns Boolean
 */
function hasUnescapedDoubleQuotes(str) {
    return /(^|[^\\])([\\]{2})*"/.test(str);
}
// Validators
// eslint-disable-next-line jsdoc/require-jsdoc
function anyValue(value) {
    return { value };
}
exports.anyValue = anyValue;
// eslint-disable-next-line jsdoc/require-jsdoc
function mapValue(value) {
    return { value };
}
exports.mapValue = mapValue;
// eslint-disable-next-line jsdoc/require-jsdoc
function stringValue(value) {
    if (typeof value !== "string" && !token_1.Tokenization.isResolvable(value)) {
        throw new Error(`'${value}' is not a valid string nor a token`);
    }
    if (typeof value === "string" && hasUnescapedDoubleQuotes(value)) {
        throw new Error(`'${value}' can not be used as value directly since it has unescaped double quotes in it. To safely use the value please use Fn.rawString on your string.`);
    }
    return { value };
}
exports.stringValue = stringValue;
// eslint-disable-next-line jsdoc/require-jsdoc
function numericValue(value) {
    if (typeof value !== "number" && !token_1.Tokenization.isResolvable(value)) {
        throw new Error(`${value} is not a valid number nor a token`);
    }
    return { value };
}
exports.numericValue = numericValue;
// eslint-disable-next-line jsdoc/require-jsdoc
function listOf(type) {
    return (value) => {
        if (token_1.Tokenization.isResolvable(value)) {
            return { value };
        }
        if (!Array.isArray(value)) {
            //   throw new Error(`${value} is not a valid list`);
            return { value };
        }
        return {
            value: value
                .filter((item) => item !== undefined && item !== null)
                .map((item, i) => {
                if (token_1.Tokenization.isResolvable(item)) {
                    return item;
                }
                if (encoding_1.TokenString.forListToken(item).test()) {
                    return item;
                }
                if ((0, encoding_1.extractTokenDouble)(item, true) !== undefined) {
                    return item;
                }
                if (encoding_1.TokenString.forMapToken(item).test()) {
                    return item;
                }
                if (typeof item === "string") {
                    const tokenList = token_1.Tokenization.reverseString(item);
                    const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
                    if (numberOfTokens === 1 && tokenList.literals.length === 0) {
                        return item;
                    }
                }
                try {
                    type(item);
                    return typeof item === "string" ? `"${item}"` : item;
                }
                catch (error) {
                    throw new Error(`Element in list ${value} at position ${i} is not of the right type: ${error}`);
                }
            }),
        };
    };
}
exports.listOf = listOf;
// eslint-disable-next-line jsdoc/require-jsdoc
function variadic(type) {
    return (value) => {
        // we use the list validator and set variadic to true in order to have it expanded in the args passed to the TF function
        return { value: listOf(type)(value).value, variadic: true };
    };
}
exports.variadic = variadic;
// Tokenization
// eslint-disable-next-line jsdoc/require-jsdoc
function asString(value) {
    return token_1.Token.asString(value);
}
exports.asString = asString;
// eslint-disable-next-line jsdoc/require-jsdoc
function asNumber(value) {
    return token_1.Token.asNumber(value);
}
exports.asNumber = asNumber;
// eslint-disable-next-line jsdoc/require-jsdoc
function asList(value) {
    return token_1.Token.asList(value);
}
exports.asList = asList;
// eslint-disable-next-line jsdoc/require-jsdoc
function asStringMap(value) {
    return token_1.Token.asStringMap(value);
}
exports.asStringMap = asStringMap;
// eslint-disable-next-line jsdoc/require-jsdoc
function asBoolean(value) {
    return value; // Booleans can not be represented as a token
}
exports.asBoolean = asBoolean;
// eslint-disable-next-line jsdoc/require-jsdoc
function asAny(value) {
    // Ordinarily casting to any can cause issues, but
    // in this case it makes using functions a bit easier in TS
    // and doesn't really harm other languages.
    // Jsii has issues when returning the value directly,
    // so wrap as a string.
    return asString(value);
}
exports.asAny = asAny;
// eslint-disable-next-line jsdoc/require-jsdoc
function terraformFunction(name, argValidators) {
    return function (...args) {
        if (args.length !== argValidators.length) {
            throw new Error(`${name} takes ${argValidators.length} arguments, but ${args.length} were provided`);
        }
        return (0, tfExpression_1.call)(name, 
        // We use flatMap now, since listOf() wraps everything in an extra array but variadic() does not
        // This enables us to have variadic pass multiple args into the call() from a single array
        args.reduce((carry, arg, i) => {
            try {
                const val = argValidators[i](arg);
                if (val.variadic)
                    return [...carry, ...val.value];
                else
                    return [...carry, val.value];
            }
            catch (error) {
                throw new Error(`Argument ${i} of ${name} failed the validation: ${error}`);
            }
        }, []));
    };
}
exports.terraformFunction = terraformFunction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaGVscGVycy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImhlbHBlcnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsK0JBQStCO0FBQy9CLG1DQUFtQztBQUNuQywyQ0FBc0Q7QUFDdEQsa0RBQXVDO0FBRXZDLHlEQUE2RTtBQU83RTs7OztHQUlHO0FBQ0gsU0FBUyx3QkFBd0IsQ0FBQyxHQUFXO0lBQzNDLE9BQU8sc0JBQXNCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBQzFDLENBQUM7QUFFRCxhQUFhO0FBQ2IsK0NBQStDO0FBQy9DLFNBQWdCLFFBQVEsQ0FBSSxLQUFRO0lBQ2xDLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQztBQUNuQixDQUFDO0FBRkQsNEJBRUM7QUFFRCwrQ0FBK0M7QUFDL0MsU0FBZ0IsUUFBUSxDQUFJLEtBQVE7SUFDbEMsT0FBTyxFQUFFLEtBQUssRUFBRSxDQUFDO0FBQ25CLENBQUM7QUFGRCw0QkFFQztBQUVELCtDQUErQztBQUMvQyxTQUFnQixXQUFXLENBQW1CLEtBQVE7SUFDcEQsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksQ0FBQyxvQkFBWSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLElBQUksS0FBSyxxQ0FBcUMsQ0FBQyxDQUFDO0tBQ2pFO0lBRUQsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksd0JBQXdCLENBQUMsS0FBSyxDQUFDLEVBQUU7UUFDaEUsTUFBTSxJQUFJLEtBQUssQ0FDYixJQUFJLEtBQUssaUpBQWlKLENBQzNKLENBQUM7S0FDSDtJQUVELE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQztBQUNuQixDQUFDO0FBWkQsa0NBWUM7QUFFRCwrQ0FBK0M7QUFDL0MsU0FBZ0IsWUFBWSxDQUFJLEtBQVE7SUFDdEMsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksQ0FBQyxvQkFBWSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLEdBQUcsS0FBSyxvQ0FBb0MsQ0FBQyxDQUFDO0tBQy9EO0lBQ0QsT0FBTyxFQUFFLEtBQUssRUFBRSxDQUFDO0FBQ25CLENBQUM7QUFMRCxvQ0FLQztBQUVELCtDQUErQztBQUMvQyxTQUFnQixNQUFNLENBQUksSUFBeUI7SUFDakQsT0FBTyxDQUFDLEtBQVUsRUFBRSxFQUFFO1FBQ3BCLElBQUksb0JBQVksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDcEMsT0FBTyxFQUFFLEtBQUssRUFBRSxDQUFDO1NBQ2xCO1FBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDekIscURBQXFEO1lBQ3JELE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQztTQUNsQjtRQUVELE9BQU87WUFDTCxLQUFLLEVBQUUsS0FBSztpQkFDVCxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksS0FBSyxTQUFTLElBQUksSUFBSSxLQUFLLElBQUksQ0FBQztpQkFDckQsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxFQUFFO2dCQUNmLElBQUksb0JBQVksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEVBQUU7b0JBQ25DLE9BQU8sSUFBSSxDQUFDO2lCQUNiO2dCQUVELElBQUksc0JBQVcsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7b0JBQ3pDLE9BQU8sSUFBSSxDQUFDO2lCQUNiO2dCQUVELElBQUksSUFBQSw2QkFBa0IsRUFBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFO29CQUNoRCxPQUFPLElBQUksQ0FBQztpQkFDYjtnQkFFRCxJQUFJLHNCQUFXLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO29CQUN4QyxPQUFPLElBQUksQ0FBQztpQkFDYjtnQkFFRCxJQUFJLE9BQU8sSUFBSSxLQUFLLFFBQVEsRUFBRTtvQkFDNUIsTUFBTSxTQUFTLEdBQUcsb0JBQVksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7b0JBQ25ELE1BQU0sY0FBYyxHQUNsQixTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQztvQkFDdkQsSUFBSSxjQUFjLEtBQUssQ0FBQyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTt3QkFDM0QsT0FBTyxJQUFJLENBQUM7cUJBQ2I7aUJBQ0Y7Z0JBRUQsSUFBSTtvQkFDRixJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7b0JBQ1gsT0FBTyxPQUFPLElBQUksS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztpQkFDdEQ7Z0JBQUMsT0FBTyxLQUFLLEVBQUU7b0JBQ2QsTUFBTSxJQUFJLEtBQUssQ0FDYixtQkFBbUIsS0FBSyxnQkFBZ0IsQ0FBQyw4QkFBOEIsS0FBSyxFQUFFLENBQy9FLENBQUM7aUJBQ0g7WUFDSCxDQUFDLENBQUM7U0FDTCxDQUFDO0lBQ0osQ0FBQyxDQUFDO0FBQ0osQ0FBQztBQW5ERCx3QkFtREM7QUFFRCwrQ0FBK0M7QUFDL0MsU0FBZ0IsUUFBUSxDQUFJLElBQXlCO0lBQ25ELE9BQU8sQ0FBQyxLQUFVLEVBQUUsRUFBRTtRQUNwQix3SEFBd0g7UUFDeEgsT0FBTyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQztJQUM5RCxDQUFDLENBQUM7QUFDSixDQUFDO0FBTEQsNEJBS0M7QUFFRCxlQUFlO0FBQ2YsK0NBQStDO0FBQy9DLFNBQWdCLFFBQVEsQ0FBQyxLQUFrQjtJQUN6QyxPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDL0IsQ0FBQztBQUZELDRCQUVDO0FBRUQsK0NBQStDO0FBQy9DLFNBQWdCLFFBQVEsQ0FBQyxLQUFrQjtJQUN6QyxPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDL0IsQ0FBQztBQUZELDRCQUVDO0FBRUQsK0NBQStDO0FBQy9DLFNBQWdCLE1BQU0sQ0FBQyxLQUFrQjtJQUN2QyxPQUFPLGFBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDN0IsQ0FBQztBQUZELHdCQUVDO0FBRUQsK0NBQStDO0FBQy9DLFNBQWdCLFdBQVcsQ0FBQyxLQUFrQjtJQUM1QyxPQUFPLGFBQUssQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDbEMsQ0FBQztBQUZELGtDQUVDO0FBRUQsK0NBQStDO0FBQy9DLFNBQWdCLFNBQVMsQ0FBQyxLQUFrQjtJQUMxQyxPQUFPLEtBQUssQ0FBQyxDQUFDLDZDQUE2QztBQUM3RCxDQUFDO0FBRkQsOEJBRUM7QUFFRCwrQ0FBK0M7QUFDL0MsU0FBZ0IsS0FBSyxDQUFDLEtBQWtCO0lBQ3RDLGtEQUFrRDtJQUNsRCwyREFBMkQ7SUFDM0QsMkNBQTJDO0lBQzNDLHFEQUFxRDtJQUNyRCx1QkFBdUI7SUFDdkIsT0FBTyxRQUFRLENBQUMsS0FBSyxDQUFRLENBQUM7QUFDaEMsQ0FBQztBQVBELHNCQU9DO0FBRUQsK0NBQStDO0FBQy9DLFNBQWdCLGlCQUFpQixDQUMvQixJQUFZLEVBQ1osYUFBc0M7SUFFdEMsT0FBTyxVQUFVLEdBQUcsSUFBVztRQUM3QixJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssYUFBYSxDQUFDLE1BQU0sRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUNiLEdBQUcsSUFBSSxVQUFVLGFBQWEsQ0FBQyxNQUFNLG1CQUFtQixJQUFJLENBQUMsTUFBTSxnQkFBZ0IsQ0FDcEYsQ0FBQztTQUNIO1FBQ0QsT0FBTyxJQUFBLG1CQUFJLEVBQ1QsSUFBSTtRQUNKLGdHQUFnRztRQUNoRywwRkFBMEY7UUFDMUYsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDNUIsSUFBSTtnQkFDRixNQUFNLEdBQUcsR0FBRyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ2xDLElBQUksR0FBRyxDQUFDLFFBQVE7b0JBQUUsT0FBTyxDQUFDLEdBQUcsS0FBSyxFQUFFLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDOztvQkFDN0MsT0FBTyxDQUFDLEdBQUcsS0FBSyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUNuQztZQUFDLE9BQU8sS0FBSyxFQUFFO2dCQUNkLE1BQU0sSUFBSSxLQUFLLENBQ2IsWUFBWSxDQUFDLE9BQU8sSUFBSSwyQkFBMkIsS0FBSyxFQUFFLENBQzNELENBQUM7YUFDSDtRQUNILENBQUMsRUFBRSxFQUFFLENBQUMsQ0FDUCxDQUFDO0lBQ0osQ0FBQyxDQUFDO0FBQ0osQ0FBQztBQTNCRCw4Q0EyQkMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgKGMpIEhhc2hpQ29ycCwgSW5jXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTVBMLTIuMFxuaW1wb3J0IHsgVG9rZW5pemF0aW9uLCBUb2tlbiB9IGZyb20gXCIuLi90b2tlbnMvdG9rZW5cIjtcbmltcG9ydCB7IGNhbGwgfSBmcm9tIFwiLi4vdGZFeHByZXNzaW9uXCI7XG5pbXBvcnQgeyBJUmVzb2x2YWJsZSB9IGZyb20gXCIuLi90b2tlbnMvcmVzb2x2YWJsZVwiO1xuaW1wb3J0IHsgVG9rZW5TdHJpbmcsIGV4dHJhY3RUb2tlbkRvdWJsZSB9IGZyb20gXCIuLi90b2tlbnMvcHJpdmF0ZS9lbmNvZGluZ1wiO1xuXG50eXBlIFRGVmFsdWU8VD4gPSB7IHZhcmlhZGljPzogYm9vbGVhbjsgdmFsdWU6IFQgfTtcbnR5cGUgVEZWYWx1ZVZhbGlkYXRvcjxUPiA9ICh2YWx1ZTogVCkgPT4gVEZWYWx1ZTxUPjtcblxudHlwZSBFeGVjdXRhYmxlVGZGdW5jdGlvbiA9ICguLi5hcmdzOiBhbnlbXSkgPT4gSVJlc29sdmFibGU7XG5cbi8qKlxuICogRGV0ZXJtaW5lcyBpZiBnaXZlbiBzdHIgaGFzIHVuZXNjYXBlZCBkb3VibGUgcXVvdGVzXG4gKiBAcGFyYW0gc3RyIFN0cmluZyB0byB0ZXN0XG4gKiBAcmV0dXJucyBCb29sZWFuXG4gKi9cbmZ1bmN0aW9uIGhhc1VuZXNjYXBlZERvdWJsZVF1b3RlcyhzdHI6IHN0cmluZykge1xuICByZXR1cm4gLyhefFteXFxcXF0pKFtcXFxcXXsyfSkqXCIvLnRlc3Qoc3RyKTtcbn1cblxuLy8gVmFsaWRhdG9yc1xuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiBhbnlWYWx1ZTxUPih2YWx1ZTogVCk6IFRGVmFsdWU8VD4ge1xuICByZXR1cm4geyB2YWx1ZSB9O1xufVxuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZXhwb3J0IGZ1bmN0aW9uIG1hcFZhbHVlPFQ+KHZhbHVlOiBUKTogVEZWYWx1ZTxUPiB7XG4gIHJldHVybiB7IHZhbHVlIH07XG59XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBqc2RvYy9yZXF1aXJlLWpzZG9jXG5leHBvcnQgZnVuY3Rpb24gc3RyaW5nVmFsdWU8VCBleHRlbmRzIHN0cmluZz4odmFsdWU6IFQpOiBURlZhbHVlPFQ+IHtcbiAgaWYgKHR5cGVvZiB2YWx1ZSAhPT0gXCJzdHJpbmdcIiAmJiAhVG9rZW5pemF0aW9uLmlzUmVzb2x2YWJsZSh2YWx1ZSkpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYCcke3ZhbHVlfScgaXMgbm90IGEgdmFsaWQgc3RyaW5nIG5vciBhIHRva2VuYCk7XG4gIH1cblxuICBpZiAodHlwZW9mIHZhbHVlID09PSBcInN0cmluZ1wiICYmIGhhc1VuZXNjYXBlZERvdWJsZVF1b3Rlcyh2YWx1ZSkpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBgJyR7dmFsdWV9JyBjYW4gbm90IGJlIHVzZWQgYXMgdmFsdWUgZGlyZWN0bHkgc2luY2UgaXQgaGFzIHVuZXNjYXBlZCBkb3VibGUgcXVvdGVzIGluIGl0LiBUbyBzYWZlbHkgdXNlIHRoZSB2YWx1ZSBwbGVhc2UgdXNlIEZuLnJhd1N0cmluZyBvbiB5b3VyIHN0cmluZy5gXG4gICAgKTtcbiAgfVxuXG4gIHJldHVybiB7IHZhbHVlIH07XG59XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBqc2RvYy9yZXF1aXJlLWpzZG9jXG5leHBvcnQgZnVuY3Rpb24gbnVtZXJpY1ZhbHVlPFQ+KHZhbHVlOiBUKTogVEZWYWx1ZTxUPiB7XG4gIGlmICh0eXBlb2YgdmFsdWUgIT09IFwibnVtYmVyXCIgJiYgIVRva2VuaXphdGlvbi5pc1Jlc29sdmFibGUodmFsdWUpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGAke3ZhbHVlfSBpcyBub3QgYSB2YWxpZCBudW1iZXIgbm9yIGEgdG9rZW5gKTtcbiAgfVxuICByZXR1cm4geyB2YWx1ZSB9O1xufVxuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZXhwb3J0IGZ1bmN0aW9uIGxpc3RPZjxUPih0eXBlOiBURlZhbHVlVmFsaWRhdG9yPFQ+KTogVEZWYWx1ZVZhbGlkYXRvcjxUW10+IHtcbiAgcmV0dXJuICh2YWx1ZTogYW55KSA9PiB7XG4gICAgaWYgKFRva2VuaXphdGlvbi5pc1Jlc29sdmFibGUodmFsdWUpKSB7XG4gICAgICByZXR1cm4geyB2YWx1ZSB9O1xuICAgIH1cblxuICAgIGlmICghQXJyYXkuaXNBcnJheSh2YWx1ZSkpIHtcbiAgICAgIC8vICAgdGhyb3cgbmV3IEVycm9yKGAke3ZhbHVlfSBpcyBub3QgYSB2YWxpZCBsaXN0YCk7XG4gICAgICByZXR1cm4geyB2YWx1ZSB9O1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICB2YWx1ZTogdmFsdWVcbiAgICAgICAgLmZpbHRlcigoaXRlbSkgPT4gaXRlbSAhPT0gdW5kZWZpbmVkICYmIGl0ZW0gIT09IG51bGwpXG4gICAgICAgIC5tYXAoKGl0ZW0sIGkpID0+IHtcbiAgICAgICAgICBpZiAoVG9rZW5pemF0aW9uLmlzUmVzb2x2YWJsZShpdGVtKSkge1xuICAgICAgICAgICAgcmV0dXJuIGl0ZW07XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKFRva2VuU3RyaW5nLmZvckxpc3RUb2tlbihpdGVtKS50ZXN0KCkpIHtcbiAgICAgICAgICAgIHJldHVybiBpdGVtO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmIChleHRyYWN0VG9rZW5Eb3VibGUoaXRlbSwgdHJ1ZSkgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgcmV0dXJuIGl0ZW07XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKFRva2VuU3RyaW5nLmZvck1hcFRva2VuKGl0ZW0pLnRlc3QoKSkge1xuICAgICAgICAgICAgcmV0dXJuIGl0ZW07XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKHR5cGVvZiBpdGVtID09PSBcInN0cmluZ1wiKSB7XG4gICAgICAgICAgICBjb25zdCB0b2tlbkxpc3QgPSBUb2tlbml6YXRpb24ucmV2ZXJzZVN0cmluZyhpdGVtKTtcbiAgICAgICAgICAgIGNvbnN0IG51bWJlck9mVG9rZW5zID1cbiAgICAgICAgICAgICAgdG9rZW5MaXN0LnRva2Vucy5sZW5ndGggKyB0b2tlbkxpc3QuaW50cmluc2ljLmxlbmd0aDtcbiAgICAgICAgICAgIGlmIChudW1iZXJPZlRva2VucyA9PT0gMSAmJiB0b2tlbkxpc3QubGl0ZXJhbHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICAgIHJldHVybiBpdGVtO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cblxuICAgICAgICAgIHRyeSB7XG4gICAgICAgICAgICB0eXBlKGl0ZW0pO1xuICAgICAgICAgICAgcmV0dXJuIHR5cGVvZiBpdGVtID09PSBcInN0cmluZ1wiID8gYFwiJHtpdGVtfVwiYCA6IGl0ZW07XG4gICAgICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgICAgYEVsZW1lbnQgaW4gbGlzdCAke3ZhbHVlfSBhdCBwb3NpdGlvbiAke2l9IGlzIG5vdCBvZiB0aGUgcmlnaHQgdHlwZTogJHtlcnJvcn1gXG4gICAgICAgICAgICApO1xuICAgICAgICAgIH1cbiAgICAgICAgfSksXG4gICAgfTtcbiAgfTtcbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiB2YXJpYWRpYzxUPih0eXBlOiBURlZhbHVlVmFsaWRhdG9yPFQ+KTogVEZWYWx1ZVZhbGlkYXRvcjxUW10+IHtcbiAgcmV0dXJuICh2YWx1ZTogYW55KSA9PiB7XG4gICAgLy8gd2UgdXNlIHRoZSBsaXN0IHZhbGlkYXRvciBhbmQgc2V0IHZhcmlhZGljIHRvIHRydWUgaW4gb3JkZXIgdG8gaGF2ZSBpdCBleHBhbmRlZCBpbiB0aGUgYXJncyBwYXNzZWQgdG8gdGhlIFRGIGZ1bmN0aW9uXG4gICAgcmV0dXJuIHsgdmFsdWU6IGxpc3RPZih0eXBlKSh2YWx1ZSkudmFsdWUsIHZhcmlhZGljOiB0cnVlIH07XG4gIH07XG59XG5cbi8vIFRva2VuaXphdGlvblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiBhc1N0cmluZyh2YWx1ZTogSVJlc29sdmFibGUpIHtcbiAgcmV0dXJuIFRva2VuLmFzU3RyaW5nKHZhbHVlKTtcbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiBhc051bWJlcih2YWx1ZTogSVJlc29sdmFibGUpIHtcbiAgcmV0dXJuIFRva2VuLmFzTnVtYmVyKHZhbHVlKTtcbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiBhc0xpc3QodmFsdWU6IElSZXNvbHZhYmxlKSB7XG4gIHJldHVybiBUb2tlbi5hc0xpc3QodmFsdWUpO1xufVxuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZXhwb3J0IGZ1bmN0aW9uIGFzU3RyaW5nTWFwKHZhbHVlOiBJUmVzb2x2YWJsZSkge1xuICByZXR1cm4gVG9rZW4uYXNTdHJpbmdNYXAodmFsdWUpO1xufVxuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZXhwb3J0IGZ1bmN0aW9uIGFzQm9vbGVhbih2YWx1ZTogSVJlc29sdmFibGUpIHtcbiAgcmV0dXJuIHZhbHVlOyAvLyBCb29sZWFucyBjYW4gbm90IGJlIHJlcHJlc2VudGVkIGFzIGEgdG9rZW5cbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiBhc0FueSh2YWx1ZTogSVJlc29sdmFibGUpIHtcbiAgLy8gT3JkaW5hcmlseSBjYXN0aW5nIHRvIGFueSBjYW4gY2F1c2UgaXNzdWVzLCBidXRcbiAgLy8gaW4gdGhpcyBjYXNlIGl0IG1ha2VzIHVzaW5nIGZ1bmN0aW9ucyBhIGJpdCBlYXNpZXIgaW4gVFNcbiAgLy8gYW5kIGRvZXNuJ3QgcmVhbGx5IGhhcm0gb3RoZXIgbGFuZ3VhZ2VzLlxuICAvLyBKc2lpIGhhcyBpc3N1ZXMgd2hlbiByZXR1cm5pbmcgdGhlIHZhbHVlIGRpcmVjdGx5LFxuICAvLyBzbyB3cmFwIGFzIGEgc3RyaW5nLlxuICByZXR1cm4gYXNTdHJpbmcodmFsdWUpIGFzIGFueTtcbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBmdW5jdGlvbiB0ZXJyYWZvcm1GdW5jdGlvbihcbiAgbmFtZTogc3RyaW5nLFxuICBhcmdWYWxpZGF0b3JzOiBURlZhbHVlVmFsaWRhdG9yPGFueT5bXVxuKTogRXhlY3V0YWJsZVRmRnVuY3Rpb24ge1xuICByZXR1cm4gZnVuY3Rpb24gKC4uLmFyZ3M6IGFueVtdKSB7XG4gICAgaWYgKGFyZ3MubGVuZ3RoICE9PSBhcmdWYWxpZGF0b3JzLmxlbmd0aCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgJHtuYW1lfSB0YWtlcyAke2FyZ1ZhbGlkYXRvcnMubGVuZ3RofSBhcmd1bWVudHMsIGJ1dCAke2FyZ3MubGVuZ3RofSB3ZXJlIHByb3ZpZGVkYFxuICAgICAgKTtcbiAgICB9XG4gICAgcmV0dXJuIGNhbGwoXG4gICAgICBuYW1lLFxuICAgICAgLy8gV2UgdXNlIGZsYXRNYXAgbm93LCBzaW5jZSBsaXN0T2YoKSB3cmFwcyBldmVyeXRoaW5nIGluIGFuIGV4dHJhIGFycmF5IGJ1dCB2YXJpYWRpYygpIGRvZXMgbm90XG4gICAgICAvLyBUaGlzIGVuYWJsZXMgdXMgdG8gaGF2ZSB2YXJpYWRpYyBwYXNzIG11bHRpcGxlIGFyZ3MgaW50byB0aGUgY2FsbCgpIGZyb20gYSBzaW5nbGUgYXJyYXlcbiAgICAgIGFyZ3MucmVkdWNlKChjYXJyeSwgYXJnLCBpKSA9PiB7XG4gICAgICAgIHRyeSB7XG4gICAgICAgICAgY29uc3QgdmFsID0gYXJnVmFsaWRhdG9yc1tpXShhcmcpO1xuICAgICAgICAgIGlmICh2YWwudmFyaWFkaWMpIHJldHVybiBbLi4uY2FycnksIC4uLnZhbC52YWx1ZV07XG4gICAgICAgICAgZWxzZSByZXR1cm4gWy4uLmNhcnJ5LCB2YWwudmFsdWVdO1xuICAgICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgIGBBcmd1bWVudCAke2l9IG9mICR7bmFtZX0gZmFpbGVkIHRoZSB2YWxpZGF0aW9uOiAke2Vycm9yfWBcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICB9LCBbXSlcbiAgICApO1xuICB9O1xufVxuIl19