"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findFileAboveCwd = exports.hashPath = exports.archiveSync = exports.copySync = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const crypto = require("crypto");
const HASH_LEN = 32;
// Full implementation at https://github.com/jprichardson/node-fs-extra/blob/master/lib/copy/copy-sync.js
/**
 * Copy a file or directory. The directory can have contents and subfolders.
 * @param {string} src
 * @param {string} dest
 */
function copySync(src, dest) {
    /**
     * Copies file if present otherwise walks subfolder
     * @param {string} p
     */
    function copyItem(p) {
        const sourcePath = path.resolve(src, p);
        const stat = fs.statSync(sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(sourcePath, path.resolve(dest, p));
        }
        if (stat.isDirectory()) {
            walkSubfolder(p);
        }
    }
    /**
     * Copies contents of subfolder
     * @param {string} p
     */
    function walkSubfolder(p) {
        const sourceDir = path.resolve(src, p);
        fs.mkdirSync(path.resolve(dest, p), { recursive: true });
        fs.readdirSync(sourceDir).forEach((item) => copyItem(path.join(p, item)));
    }
    walkSubfolder(".");
}
exports.copySync = copySync;
/**
 * Zips contents at src and places zip archive at dest
 * @param {string} src
 * @param {string} dest
 */
function archiveSync(src, dest) {
    const projectRoot = path.resolve(__dirname, "..", "..");
    const zipSyncPath = path.resolve(projectRoot, "bin", "zipSync.js");
    (0, child_process_1.execSync)(`node ${zipSyncPath} ${src} ${dest}`);
}
exports.archiveSync = archiveSync;
// eslint-disable-next-line jsdoc/require-jsdoc
function hashPath(src) {
    const hash = crypto.createHash("md5");
    // eslint-disable-next-line jsdoc/require-jsdoc
    function hashRecursion(p) {
        const stat = fs.statSync(p);
        if (stat.isFile()) {
            hash.update(fs.readFileSync(p));
        }
        else if (stat.isDirectory()) {
            fs.readdirSync(p).forEach((filename) => hashRecursion(path.resolve(p, filename)));
        }
    }
    hashRecursion(src);
    return hash.digest("hex").slice(0, HASH_LEN).toUpperCase();
}
exports.hashPath = hashPath;
// eslint-disable-next-line jsdoc/require-jsdoc
function findFileAboveCwd(file, rootPath = process.cwd()) {
    const fullPath = path.resolve(rootPath, file);
    if (fs.existsSync(fullPath)) {
        return fullPath;
    }
    const parentDir = path.resolve(rootPath, "..");
    if (fs.existsSync(parentDir) && parentDir !== rootPath) {
        return findFileAboveCwd(file, parentDir);
    }
    return null;
}
exports.findFileAboveCwd = findFileAboveCwd;
//# sourceMappingURL=data:application/json;base64,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