"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const UNIQUE_SEP = "_";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components, allowSepChars) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error("Unable to calculate a unique id for an empty set of components");
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeDisallowedCharacters(components[0], allowSepChars);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map((s) => removeDisallowedCharacters(s, allowSepChars))
        .join(UNIQUE_SEP)
        .slice(0, MAX_HUMAN_LEN);
    return human + UNIQUE_SEP + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
// eslint-disable-next-line jsdoc/require-jsdoc
function removeDisallowedCharacters(s, allowSepChars) {
    if (allowSepChars) {
        return removeNonAlphanumericSep(s);
    }
    else {
        return removeNonAlphanumeric(s);
    }
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, "");
}
// eslint-disable-next-line jsdoc/require-jsdoc
function removeNonAlphanumericSep(s) {
    return s.replace(/[^A-Za-z0-9_-]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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