"use strict";
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// ----------------------------------------------------------------------
// PROPERTY MAPPERS
//
// These are used while converting generated classes/property bags to Terraform JSON objects
//
// We use identity mappers for the primitive types. These don't do anything but are there to make the code
// generation work out nicely (so the code generator doesn't need to emit different code for primitive
Object.defineProperty(exports, "__esModule", { value: true });
exports.isComplexElement = exports.canInspect = exports.hashMapper = exports.listMapper = exports.numberToTerraform = exports.anyToTerraform = exports.booleanToTerraform = exports.stringToTerraform = void 0;
const encoding_1 = require("./tokens/private/encoding");
Object.defineProperty(exports, "isComplexElement", { enumerable: true, get: function () { return encoding_1.isComplexElement; } });
const tokens_1 = require("./tokens");
const terraform_dynamic_block_1 = require("./terraform-dynamic-block");
const terraform_dynamic_expression_1 = require("./terraform-dynamic-expression");
// eslint-disable-next-line jsdoc/require-jsdoc
function identity(x) {
    return x;
}
exports.stringToTerraform = identity;
exports.booleanToTerraform = identity;
exports.anyToTerraform = identity;
exports.numberToTerraform = identity;
/**
 * @param isBlockType blocks and lists of objects need to be handled different in Terraform
 *                    but are represented exactly the same in generated bindings
 *                    currently this is used to resolve TerraformDynamicBlocks if not within
 *                    a block type (will be resolved to a for expression for those instead of
 *                    a dynamic block)
 */
function listMapper(elementMapper, isBlockType) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        // replace dynamic expressions for block types so they can be detected and replaced properly by processDynamicAttributes()
        // which also relocates them to dynamic.attributeName (something we can't do when resolving a value, put it somewhere else)
        // if isBlockType is false, the TerraformDynamicExpression that is present will be resolved (it implements IResolvable) to a
        // for expression directly (which can be used e.g. within Terraform functions or for attributes that are not of a block type)
        if (terraform_dynamic_expression_1.TerraformDynamicExpression.isTerraformDynamicExpression(x) &&
            isBlockType) {
            return terraform_dynamic_block_1.TerraformDynamicBlock.fromDynamicExpression(x);
        }
        if (!Array.isArray(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
exports.listMapper = listMapper;
// eslint-disable-next-line jsdoc/require-jsdoc
function hashMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        // Check if it's a token object
        if ((0, encoding_1.containsComplexElement)(x)) {
            return x;
        }
        if (tokens_1.Tokenization.isResolvable(x)) {
            return x;
        }
        // We can't treat strings as hashes (likely a token or a misconfiguration)
        if (typeof x === "string") {
            return x;
        }
        if ((0, encoding_1.containsMapToken)(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
exports.hashMapper = hashMapper;
/**
 * Return whether this object can be validated at all
 *
 * True unless it's undefined
 */
function canInspect(x) {
    // Note: using weak equality on purpose, we also want to catch undefined
    return x != null;
}
exports.canInspect = canInspect;
//# sourceMappingURL=data:application/json;base64,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