"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomSynthesis = exports.invokeAspects = exports.StackSynthesizer = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const path = require("path");
const fs = require("fs");
const stringify = require("json-stable-stringify");
const annotations_1 = require("../annotations");
const constructs_1 = require("constructs");
const aspect_1 = require("../aspect");
// eslint-disable-next-line jsdoc/require-jsdoc
class StackSynthesizer {
    /**
     * @param stack the stack to synthesize
     * @param continueOnErrorAnnotations if set to true, `synthesize()` will not throw an error
     * if it encounters an error Annotation. However, it will still add all errors to the manifest
     * file. The cdktf-cli uses this flag to print the errors itself.
     */
    constructor(stack, continueOnErrorAnnotations = false) {
        this.stack = stack;
        this.continueOnErrorAnnotations = continueOnErrorAnnotations;
    }
    synthesize(session) {
        invokeAspects(this.stack);
        if (!session.skipValidation) {
            this.stack.runAllValidations();
        }
        const manifest = session.manifest;
        const stackManifest = manifest.forStack(this.stack);
        const workingDirectory = path.join(session.outdir, stackManifest.workingDirectory);
        if (!fs.existsSync(workingDirectory))
            fs.mkdirSync(workingDirectory);
        // call custom synthesis on child nodes (leafs first)
        this.stack.node
            .findAll(constructs_1.ConstructOrder.POSTORDER)
            .forEach((node) => getCustomSynthesis(node)?.onSynthesize(session));
        // collect Annotations into Manifest
        const annotations = this.stack.node
            .findAll()
            .map((node) => ({
            node,
            metadatas: node.node.metadata.filter(isAnnotationMetadata),
        }))
            .map(({ node, metadatas }) => metadatas.map((metadata) => ({
            constructPath: node.node.path,
            level: metadata.type,
            message: metadata.data,
            stacktrace: metadata.trace,
        })))
            .reduce((list, metadatas) => [...list, ...metadatas], []); // Array.flat()
        // it is readonly but this is the place where we are allowed to write to it
        stackManifest.annotations = annotations;
        // abort if one or more error annotations have been encountered
        if (!this.continueOnErrorAnnotations &&
            annotations.some(isErrorAnnotation)) {
            throw new Error(`Encountered Annotations with level "ERROR":\n${annotations
                .filter(isErrorAnnotation)
                .map((a) => `[${a.constructPath}] ${a.message}`)
                .join("\n")}`);
        }
        const tfConfig = this.stack.toTerraform();
        fs.writeFileSync(path.join(session.outdir, stackManifest.synthesizedStackPath), stringify(tfConfig, { space: 2 }));
    }
}
exports.StackSynthesizer = StackSynthesizer;
/**
 * Invoke aspects on the given construct tree.
 *
 * originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/@aws-cdk/core/lib/private/synthesis.ts#L99-L137
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    // eslint-disable-next-line jsdoc/require-jsdoc
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...(inheritedAspects ?? []), ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning("We detected an Aspect was added via another Aspect, and will not be applied");
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            recurse(child, allAspectsHere);
        }
    }
}
exports.invokeAspects = invokeAspects;
const annotationMetadataEntryTypes = [
    annotations_1.AnnotationMetadataEntryType.INFO,
    annotations_1.AnnotationMetadataEntryType.WARN,
    annotations_1.AnnotationMetadataEntryType.ERROR,
];
// eslint-disable-next-line jsdoc/require-jsdoc
function isAnnotationMetadata(metadata) {
    return annotationMetadataEntryTypes.includes(metadata.type);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function isErrorAnnotation(annotation) {
    return annotation.level === annotations_1.AnnotationMetadataEntryType.ERROR;
}
// originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/%40aws-cdk/core/lib/private/synthesis.ts#L52
const CUSTOM_SYNTHESIS_SYM = Symbol.for("cdktf/customSynthesis");
// eslint-disable-next-line jsdoc/require-jsdoc
function addCustomSynthesis(construct, synthesis) {
    Object.defineProperty(construct, CUSTOM_SYNTHESIS_SYM, {
        value: synthesis,
        enumerable: false,
    });
}
exports.addCustomSynthesis = addCustomSynthesis;
// eslint-disable-next-line jsdoc/require-jsdoc
function getCustomSynthesis(construct) {
    return construct[CUSTOM_SYNTHESIS_SYM];
}
//# sourceMappingURL=data:application/json;base64,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