"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformAsset = exports.AssetType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const constructs_1 = require("constructs");
const fs = require("fs");
const path = require("path");
const fs_1 = require("./private/fs");
const synthesizer_1 = require("./synthesize/synthesizer");
const terraform_stack_1 = require("./terraform-stack");
var AssetType;
(function (AssetType) {
    AssetType[AssetType["FILE"] = 0] = "FILE";
    AssetType[AssetType["DIRECTORY"] = 1] = "DIRECTORY";
    AssetType[AssetType["ARCHIVE"] = 2] = "ARCHIVE";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
const ARCHIVE_NAME = "archive.zip";
const ASSETS_DIRECTORY = "assets";
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformAsset extends constructs_1.Construct {
    /**
     * A Terraform Asset takes a file or directory outside of the CDK for Terraform context and moves it into it.
     * Assets copy referenced files into the stacks context for further usage in other resources.
     * @param scope
     * @param id
     * @param config
     */
    constructor(scope, id, config) {
        super(scope, id);
        this.stack = terraform_stack_1.TerraformStack.of(this);
        if (path.isAbsolute(config.path)) {
            this.sourcePath = config.path;
        }
        else {
            const cdktfJsonPath = scope.node.tryGetContext("cdktfJsonPath") ??
                (0, fs_1.findFileAboveCwd)("cdktf.json");
            if (cdktfJsonPath) {
                // Relative paths are always considered to be relative to cdktf.json, but operations are performed relative to process.cwd
                const absolutePath = path.resolve(path.dirname(cdktfJsonPath), config.path);
                this.sourcePath = path.relative(process.cwd(), absolutePath);
            }
            else {
                throw new Error(`TerraformAsset ${id} was created with a relative path '${config.path}', but no cdktf.json file was found to base it on.`);
            }
        }
        const stat = fs.statSync(this.sourcePath);
        const inferredType = stat.isFile() ? AssetType.FILE : AssetType.DIRECTORY;
        this.type = config.type ?? inferredType;
        this.assetHash = config.assetHash || (0, fs_1.hashPath)(this.sourcePath);
        if (stat.isFile() && this.type !== AssetType.FILE) {
            throw new Error(`TerraformAsset ${id} expects path to be a directory, a file was passed: '${config.path}'`);
        }
        if (!stat.isFile() && this.type === AssetType.FILE) {
            throw new Error(`TerraformAsset ${id} expects path to be a file, a directory was passed: '${config.path}'`);
        }
        (0, synthesizer_1.addCustomSynthesis)(this, {
            onSynthesize: this._onSynthesize.bind(this),
        });
    }
    get namedFolder() {
        return path.posix.join(ASSETS_DIRECTORY, this.stack.getLogicalId(this.node));
    }
    /**
     * The path relative to the root of the terraform directory in posix format
     * Use this property to reference the asset
     */
    get path() {
        return path.posix.join(this.namedFolder, // readable name
        this.assetHash, // hash depending on content so that path changes if content changes
        this.type === AssetType.DIRECTORY ? "" : this.fileName);
    }
    /**
     * Name of the asset
     */
    get fileName() {
        switch (this.type) {
            case AssetType.ARCHIVE:
                return ARCHIVE_NAME;
            default:
                return path.basename(this.sourcePath);
        }
    }
    _onSynthesize(session) {
        const stackManifest = session.manifest.forStack(this.stack);
        const basePath = path.join(session.manifest.outdir, stackManifest.synthesizedStackPath, "..");
        // Cleanup existing assets
        const previousVersionsFolder = path.join(basePath, this.namedFolder);
        if (fs.existsSync(previousVersionsFolder)) {
            fs.rmSync(previousVersionsFolder, { recursive: true });
        }
        const targetPath = path.join(basePath, this.path);
        if (this.type === AssetType.DIRECTORY) {
            fs.mkdirSync(targetPath, { recursive: true });
        }
        else {
            fs.mkdirSync(path.dirname(targetPath), { recursive: true });
        }
        switch (this.type) {
            case AssetType.FILE:
                fs.copyFileSync(this.sourcePath, targetPath);
                break;
            case AssetType.DIRECTORY:
                (0, fs_1.copySync)(this.sourcePath, targetPath);
                break;
            case AssetType.ARCHIVE:
                (0, fs_1.archiveSync)(this.sourcePath, targetPath);
                break;
            default:
                throw new Error(`Asset type ${this.type} is not implemented`);
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
TerraformAsset[_a] = { fqn: "cdktf.TerraformAsset", version: "0.17.0-pre.18" };
exports.TerraformAsset = TerraformAsset;
//# sourceMappingURL=data:application/json;base64,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