"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformDynamicBlock = void 0;
const tokens_1 = require("./tokens");
const stack_trace_1 = require("./tokens/private/stack-trace");
const DYNAMIC_BLOCK_SYMBOL = Symbol.for("cdktf/TerraformDynamicBlock");
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformDynamicBlock {
    // labels: TODO: support labels, but they seem to be quite rare (issue: https://github.com/hashicorp/terraform-cdk/issues/1939)
    constructor(args) {
        this.iterator = "each"; // name of temporary variable used in iteration, needed in the future when allowing nesting iterators
        Object.defineProperty(this, DYNAMIC_BLOCK_SYMBOL, { value: true });
        this.creationStack = (0, stack_trace_1.captureStackTrace)();
        this.forEach = args.forEach;
        this.content = args.content;
    }
    static fromDynamicExpression(expr) {
        return new TerraformDynamicBlock({
            forEach: expr.iterator,
            content: expr.content,
        });
    }
    resolve() {
        throw new Error(`Tried to directly resolve a TerraformDynamicBlock which is not supported.
Dynamic blocks are only supported on block attributes of resources, data sources and providers.`);
    }
    toTerraformDynamicBlockJson() {
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                context.iteratorContext = "DYNAMIC_BLOCK"; // resolve nested Lazy values (within this.content) as dynamic block expressions
                return {
                    for_each: this.forEach._getForEachExpression(),
                    iterator: this.iterator,
                    content: context.resolve(this.content),
                };
            },
        });
    }
    toString() {
        return tokens_1.Token.asString(this);
    }
    /**
     * Turn this Dynamic Block into JSON
     *
     * Called automatically when JSON.stringify() is called on this.
     */
    toJSON() {
        return "<unresolved-terraform-dynamic-block>";
    }
    static isTerraformDynamicBlock(x) {
        return x !== null && typeof x === "object" && DYNAMIC_BLOCK_SYMBOL in x;
    }
}
exports.TerraformDynamicBlock = TerraformDynamicBlock;
//# sourceMappingURL=data:application/json;base64,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