"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformDynamicExpression = void 0;
const _1 = require(".");
const tokens_1 = require("./tokens");
const stack_trace_1 = require("./tokens/private/stack-trace");
const DYNAMIC_EXPRESSION_SYMBOL = Symbol.for("cdktf/TerraformDynamicExpression");
/**
 * A TerraformDynamicExpression is returned by TerraformIterator.dynamic
 * and can be passed to list attributes of e.g. TerraformResources
 * As it resolves to a for expression, it can be used directly on attributes of
 * a list(object) type (or similar) and in Terraform functions that take a list.
 * When used in attributes that are of a block type which need to be modeled as
 * a dynamic block, the listMapper for that attribute will change this to a
 * TerraformDynamicBlock, which in turn will be properly handled by the
 * processDynamicAttributes() utility function.
 */
class TerraformDynamicExpression {
    constructor(args) {
        Object.defineProperty(this, DYNAMIC_EXPRESSION_SYMBOL, { value: true });
        this.creationStack = (0, stack_trace_1.captureStackTrace)();
        this.iterator = args.iterator;
        this.content = args.content;
    }
    /**
     * resolves this as a Terraform for expression.
     * This allows using iterator.dynamic() on attributes that are not blocks but
     * lists of objects (which is not visible to the user when working with provider
     * mappings as it is abstracted away)
     */
    resolve() {
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                // resolve nested Lazy values (within this.content) for a for expression
                context.iteratorContext = "FOR_EXPRESSION";
                // returns [ for key, value in forEachExpr : content ]
                // context.resolve is required for the iteratorContext to be correctly passed
                // to Lazy values within this.content
                return context.resolve((0, _1.forExpression)(this.iterator._getForEachExpression(), this.content));
            },
        });
    }
    toString() {
        return tokens_1.Token.asString(this);
    }
    /**
     * Turn this Dynamic Expression into JSON
     *
     * Called automatically when JSON.stringify() is called on this.
     */
    toJSON() {
        return "<unresolved-terraform-dynamic-expression>";
    }
    static isTerraformDynamicExpression(x) {
        return (x !== null && typeof x === "object" && DYNAMIC_EXPRESSION_SYMBOL in x);
    }
}
exports.TerraformDynamicExpression = TerraformDynamicExpression;
//# sourceMappingURL=data:application/json;base64,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