"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MapTerraformIterator = exports.ListTerraformIterator = exports.TerraformIterator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const terraform_dynamic_expression_1 = require("./terraform-dynamic-expression");
const terraform_functions_1 = require("./terraform-functions");
const tfExpression_1 = require("./tfExpression");
const tokens_1 = require("./tokens");
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformIterator {
    /**
     * Creates a new iterator from a list
     */
    static fromList(list) {
        // TODO: this could return different iterators depending on the type of the list
        // for example it could return a NumberListIterator whose iterator.key would be a number
        return new ListTerraformIterator(list);
    }
    /**
     * Creates a new iterator from a map
     */
    static fromMap(map) {
        return new MapTerraformIterator(map);
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a string
     */
    getString(attribute) {
        return tokens_1.Token.asString((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a number
     */
    getNumber(attribute) {
        return tokens_1.Token.asNumber((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a boolean
     */
    getBoolean(attribute) {
        return tokens_1.Token.asAny((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as any
     */
    getAny(attribute) {
        return tokens_1.Token.asAny((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a (string) list
     */
    getList(attribute) {
        return tokens_1.Token.asList((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a number list
     */
    getNumberList(attribute) {
        return tokens_1.Token.asNumberList((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map
     */
    getMap(attribute) {
        return tokens_1.Token.asAnyMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of strings
     */
    getStringMap(attribute) {
        return tokens_1.Token.asStringMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of numbers
     */
    getNumberMap(attribute) {
        return tokens_1.Token.asNumberMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of booleans
     */
    getBooleanMap(attribute) {
        return tokens_1.Token.asBooleanMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of any
     */
    getAnyMap(attribute) {
        return tokens_1.Token.asAnyMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @internal
     */
    _getValue() {
        // uses a Lazy value to be able to access the current TerraformStack and pass it to ref()
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                switch (context.iteratorContext) {
                    case "DYNAMIC_BLOCK":
                        return (0, tfExpression_1.ref)("each.value");
                    case "FOR_EXPRESSION":
                        return tfExpression_1.FOR_EXPRESSION_VALUE;
                    default:
                        // same as dynamic block, as this is the case when a iterator is passed to the root level of e.g. a resource
                        return (0, tfExpression_1.ref)("each.value");
                }
            },
        }, { displayHint: "<iterator value>" });
    }
    /**
     * @internal
     */
    _getKey() {
        // uses a Lazy value to be able to access the current TerraformStack and pass it to ref()
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                switch (context.iteratorContext) {
                    case "DYNAMIC_BLOCK":
                        return (0, tfExpression_1.ref)("each.key");
                    case "FOR_EXPRESSION":
                        return tfExpression_1.FOR_EXPRESSION_KEY;
                    default:
                        // same as dynamic block, as this is the case when a iterator is passed to the root level of e.g. a resource
                        return (0, tfExpression_1.ref)("each.key");
                }
            },
        }, { displayHint: "<iterator key>" });
    }
    dynamic(attributes) {
        return tokens_1.Token.asAny(new terraform_dynamic_expression_1.TerraformDynamicExpression({
            iterator: this,
            content: attributes,
        }));
    }
}
_a = JSII_RTTI_SYMBOL_1;
TerraformIterator[_a] = { fqn: "cdktf.TerraformIterator", version: "0.17.0-pre.18" };
exports.TerraformIterator = TerraformIterator;
// eslint-disable-next-line jsdoc/require-jsdoc
class ListTerraformIterator extends TerraformIterator {
    constructor(list) {
        super();
        this.list = list;
    }
    /**
     * Returns the currenty entry in the list or set that is being iterated over.
     * For lists this is the same as `iterator.value`. If you need the index,
     * use count using the escape hatch:
     * https://developer.hashicorp.com/terraform/cdktf/concepts/resources#escape-hatch
     */
    get key() {
        return this._getKey();
    }
    /**
     * Returns the value of the current item iterated over.
     */
    get value() {
        return this._getValue();
    }
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression() {
        // needs to be wrapped in a set as Terraform only allows sets in for_each
        return terraform_functions_1.Fn.toset(this.list);
    }
}
_b = JSII_RTTI_SYMBOL_1;
ListTerraformIterator[_b] = { fqn: "cdktf.ListTerraformIterator", version: "0.17.0-pre.18" };
exports.ListTerraformIterator = ListTerraformIterator;
// eslint-disable-next-line jsdoc/require-jsdoc
class MapTerraformIterator extends TerraformIterator {
    constructor(map) {
        super();
        this.map = map;
    }
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression() {
        // explicit wrapping to circumvent "Found an encoded map token in a scalar string context." error
        return tokens_1.Token.asString(this.map);
    }
    /**
     * Returns the key of the current entry in the map that is being iterated over.
     */
    get key() {
        return tokens_1.Token.asString(this._getKey());
    }
    /**
     * Returns the value of the current item iterated over.
     */
    get value() {
        return this._getValue();
    }
}
_c = JSII_RTTI_SYMBOL_1;
MapTerraformIterator[_c] = { fqn: "cdktf.MapTerraformIterator", version: "0.17.0-pre.18" };
exports.MapTerraformIterator = MapTerraformIterator;
//# sourceMappingURL=data:application/json;base64,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