"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const constructs_1 = require("constructs");
const _tokens_1 = require("./_tokens");
const terraform_element_1 = require("./terraform-element");
const util_1 = require("./util");
const terraform_provider_1 = require("./terraform-provider");
const local_backend_1 = require("./backends/local-backend");
const tfExpression_1 = require("./tfExpression");
const terraform_output_1 = require("./terraform-output");
const features_1 = require("./features");
const unique_1 = require("./private/unique");
const synthesizer_1 = require("./synthesize/synthesizer");
const STACK_SYMBOL = Symbol.for("cdktf/TerraformStack");
const validations_1 = require("./validations");
const app_1 = require("./app");
const terraform_backend_1 = require("./terraform-backend");
// eslint-disable-next-line jsdoc/require-jsdoc
function throwIfIdIsGlobCharacter(str) {
    const err = (char) => `Can not create Terraform stack with id "${str}". It contains a glob character: "${char}"`;
    ["*", "?", "[", "]", "{", "}", "!"].forEach((char) => {
        if (str.includes(char)) {
            throw new Error(err(char));
        }
    });
}
// eslint-disable-next-line jsdoc/require-jsdoc
function throwIfIdContainsWhitespace(str) {
    if (/\s/.test(str)) {
        throw new Error(`Can not create TerraformStack with id "${str}". It contains a whitespace character.`);
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformStack extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.rawOverrides = {};
        this.crossStackOutputs = {};
        this.crossStackDataSources = {};
        this.dependencies = [];
        throwIfIdIsGlobCharacter(id);
        throwIfIdContainsWhitespace(id);
        this.cdktfVersion = this.node.tryGetContext("cdktfVersion");
        this.synthesizer = new synthesizer_1.StackSynthesizer(this, process.env.CDKTF_CONTINUE_SYNTH_ON_ERROR_ANNOTATIONS !== undefined);
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this.node.addValidation(new validations_1.ValidateProviderPresence(this));
    }
    static isStack(x) {
        return x !== null && typeof x === "object" && STACK_SYMBOL in x;
    }
    static of(construct) {
        return _lookup(construct);
        // eslint-disable-next-line jsdoc/require-jsdoc
        function _lookup(c) {
            if (TerraformStack.isStack(c)) {
                return c;
            }
            const node = c.node;
            if (!node.scope) {
                let hint = "";
                if (construct.node.scope === c &&
                    app_1.App.isApp(c) &&
                    terraform_backend_1.TerraformBackend.isBackend(construct)) {
                    // the scope of the originally passed construct equals the construct c
                    // which has no scope (i.e. has no parent construct) and c is an App
                    // and our construct is a Backend
                    hint = `. You seem to have passed your root App as scope to a TerraformBackend construct. Pass a stack as scope to your backend instead.`;
                }
                throw new Error(`No stack could be identified for the construct at path '${construct.node.path}'${hint}`);
            }
            return _lookup(node.scope);
        }
    }
    findAll(predicate) {
        const items = [];
        const visit = async (node) => {
            if (predicate(node)) {
                items.push(node);
            }
            for (const child of node.node.children) {
                visit(child);
            }
        };
        visit(this);
        return items;
    }
    prepareStack() {
        // Ensure we have a backend configured
        this.ensureBackendExists();
        // A preparing resolve run might add new resources to the stack, e.g. for cross stack references.
        terraformElements(this).forEach((e) => (0, _tokens_1.resolve)(this, e.toTerraform(), true));
    }
    addOverride(path, value) {
        const parts = path.split(".");
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null &&
                typeof curr[key] === "object" &&
                !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    getLogicalId(tfElement) {
        // wrap the allocation for future renaming support
        return this.allocateLogicalId(tfElement);
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * @param tfElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(tfElement) {
        const node = terraform_element_1.TerraformElement.isTerraformElement(tfElement)
            ? tfElement.node
            : tfElement;
        const stack = terraform_element_1.TerraformElement.isTerraformElement(tfElement)
            ? tfElement.cdktfStack
            : this;
        let stackIndex;
        if (node.tryGetContext(features_1.EXCLUDE_STACK_ID_FROM_LOGICAL_IDS)) {
            stackIndex = node.scopes.indexOf(stack);
        }
        else {
            stackIndex = 0;
        }
        const components = node.scopes.slice(stackIndex + 1).map((c) => c.node.id);
        return components.length > 0
            ? (0, unique_1.makeUniqueId)(components, node.tryGetContext(features_1.ALLOW_SEP_CHARS_IN_LOGICAL_IDS))
            : "";
    }
    allProviders() {
        return this.findAll(terraform_provider_1.TerraformProvider.isTerraformProvider);
    }
    ensureBackendExists() {
        const backends = this.findAll(terraform_backend_1.TerraformBackend.isBackend);
        return backends[0] || new local_backend_1.LocalBackend(this, {});
    }
    toTerraform() {
        const tf = {};
        const metadata = {
            version: this.cdktfVersion,
            stackName: this.node.id,
            backend: "local",
            cloud: undefined,
            ...(Object.keys(this.rawOverrides).length > 0
                ? { overrides: { stack: Object.keys(this.rawOverrides) } }
                : {}),
        };
        const elements = terraformElements(this);
        const metadatas = elements.map((e) => (0, _tokens_1.resolve)(this, e.toMetadata()));
        for (const meta of metadatas) {
            (0, util_1.deepMerge)(metadata, meta);
        }
        const outputs = elements.reduce((carry, item) => {
            if (!terraform_output_1.TerraformOutput.isTerraformOutput(item)) {
                return carry;
            }
            (0, util_1.deepMerge)(carry, item.node.path.split("/").reduceRight((innerCarry, part) => {
                if (Object.keys(innerCarry).length === 0) {
                    return { [part]: item.friendlyUniqueId };
                }
                return { [part]: innerCarry };
            }, {}));
            return carry;
        }, {});
        tf["//"] = { metadata, outputs };
        const fragments = elements.map((e) => (0, _tokens_1.resolve)(this, e.toTerraform()));
        for (const fragment of fragments) {
            (0, util_1.deepMerge)(tf, fragment);
        }
        (0, util_1.deepMerge)(tf, this.rawOverrides);
        return (0, _tokens_1.resolve)(this, tf);
    }
    registerOutgoingCrossStackReference(identifier) {
        if (this.crossStackOutputs[identifier]) {
            return this.crossStackOutputs[identifier];
        }
        const output = new terraform_output_1.TerraformOutput(this, `cross-stack-output-${identifier}`, {
            value: (0, tfExpression_1.ref)(identifier, this),
            sensitive: true,
        });
        this.crossStackOutputs[identifier] = output;
        return output;
    }
    registerIncomingCrossStackReference(fromStack) {
        if (this.crossStackDataSources[String(fromStack)]) {
            return this.crossStackDataSources[String(fromStack)];
        }
        const originBackend = fromStack.ensureBackendExists();
        const originPath = fromStack.node.path;
        const remoteState = originBackend.getRemoteStateDataSource(this, `cross-stack-reference-input-${originPath}`, originPath);
        this.crossStackDataSources[originPath] = remoteState;
        return remoteState;
    }
    // Check here for loops in the dependency graph
    dependsOn(stack) {
        return (this.dependencies.includes(stack) ||
            this.dependencies.some((d) => d.dependsOn(stack)));
    }
    addDependency(dependency) {
        if (dependency.dependsOn(this)) {
            throw new Error(`Can not add dependency ${dependency} to ${this} since it would result in a loop`);
        }
        if (this.dependencies.includes(dependency)) {
            return;
        }
        this.dependencies.push(dependency);
    }
    /**
     * Run all validations on the stack.
     */
    runAllValidations() {
        const errors = this.node
            .findAll()
            .map((node) => node.node.validate().map((error) => ({ message: error, source: node })))
            .reduce((prev, curr) => [...prev, ...curr], []);
        if (errors.length > 0) {
            const errorList = errors
                .map((e) => `[${e.source.node.path}] ${e.message}`)
                .join("\n  ");
            throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
TerraformStack[_a] = { fqn: "cdktf.TerraformStack", version: "0.17.0-pre.18" };
exports.TerraformStack = TerraformStack;
// eslint-disable-next-line jsdoc/require-jsdoc
function terraformElements(node, into = []) {
    if (terraform_element_1.TerraformElement.isTerraformElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (TerraformStack.isStack(child)) {
            continue;
        }
        terraformElements(child, into);
    }
    return into;
}
//# sourceMappingURL=data:application/json;base64,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