"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const index_1 = require("../index");
const resource_1 = require("../../../test/helper/resource");
const matchers_1 = require("../matchers");
const data_source_1 = require("../../../test/helper/data-source");
const terraform_stack_1 = require("../../terraform-stack");
const provider_1 = require("../../../test/helper/provider");
const fs = require("fs");
const path = require("path");
function corruptSynthesizedStack(stackPath) {
    const manifest = JSON.parse(fs.readFileSync(path.resolve(stackPath, "manifest.json"), "utf8"));
    const { synthesizedStackPath } = Object.values(manifest.stacks)[0];
    const absoluteStackPath = path.resolve(stackPath, synthesizedStackPath);
    fs.writeFileSync(absoluteStackPath, fs.readFileSync(absoluteStackPath, "utf8") +
        "Weird text that makes the JSON file invalid", "utf8");
}
describe("deepEqualIgnoringObjectCasing", () => {
    it("should compare simple items", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(1, 1)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)("1", "1")).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(true, true)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(false, false)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(null, null)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(undefined, undefined)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 3])).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(1, 2)).toBe(false);
    });
    it("should compare arrays", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 3])).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 4])).toBe(false);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 3, 4])).toBe(false);
    });
    it("should compare objects", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { c: 3 }, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 3, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(false);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { c: 3 }, b: 2 }, { a: 3, b: 2 })).toBe(false);
    });
    it("should ignore case when comparing object keys", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 1, fooBar: 2 }, { a: 1, foo_bar: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { fooBar: 2 } }, { a: { foo_bar: 2 } })).toBe(true);
    });
    it("should ignore keys not present in expectation", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { fooBar: 2 } }, { a: { foo_bar: 2 }, b: 3 })).toBe(true);
    });
});
describe("matchers", () => {
    const toHaveResourceWithProperties = (0, matchers_1.getToHaveResourceWithProperties)();
    describe("toHaveResourceWithProperties", () => {
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test", tags: { foo: "bar" } });
            });
        });
        it("should pass with no properties", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeTruthy();
            expect(res.message).toMatchInlineSnapshot(`
        "Expected no test_resource with properties {} to be present in synthesized stack.
        Found 1 test_resource resources instead:
        [
          {
            "name": "test",
            "tags": {
              "foo": "bar"
            }
          }
        ]"
      `);
        });
        it("should fail with wrong resouce", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, data_source_1.TestDataSource);
            expect(res.pass).toBeFalsy();
            expect(res.message).toMatchInlineSnapshot(`
        "Expected test_data_source with properties {} to be present in synthesized stack.
        Found no test_data_source resources instead"
      `);
        });
        it("should work on nested elements", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource, {
                tags: { foo: "bar" },
            });
            expect(res.pass).toBeTruthy();
        });
        it("should fail if no item is found as asserted", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.DockerImage);
            expect(res.pass).toBeFalsy();
        });
    });
    describe("toHaveDataSourceWithProperties", () => {
        const toHaveDataSourceWithProperties = (0, matchers_1.getToHaveDataSourceWithProperties)();
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test" });
                new data_source_1.TestDataSource(scope, "test-data-source", { name: "data" });
            });
        });
        it("should not find resources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeFalsy();
        });
        it("should find data sources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, data_source_1.TestDataSource, { name: "data" });
            expect(res.pass).toBeTruthy();
        });
    });
    describe("toHaveProviderWithProperties", () => {
        const toHaveProviderWithProperties = (0, matchers_1.getToHaveProviderWithProperties)();
        let synthesizedStack;
        let synthesizedStackNoProvider;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new provider_1.DockerProvider(scope, "test-provider", {
                    alias: "test-alias",
                    ssh_opts: ["-o", "StrictHostKeyChecking=no"],
                });
                new resource_1.DockerImage(scope, "test-image", { name: "test" });
            });
            synthesizedStackNoProvider = index_1.Testing.synthScope((scope) => {
                new resource_1.DockerImage(scope, "test", { name: "test" });
            });
        });
        it("should find provider", () => {
            const res = toHaveProviderWithProperties(synthesizedStack, provider_1.DockerProvider, { alias: "test-alias", ssh_opts: ["-o", "StrictHostKeyChecking=no"] });
            expect(res.pass).toBeTruthy();
        });
        it("should not find provider", () => {
            const res = toHaveProviderWithProperties(synthesizedStackNoProvider, provider_1.DockerProvider, {});
            expect(res.pass).toBeFalsy();
        });
        it("should not find resources", () => {
            const res = toHaveProviderWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeFalsy();
        });
        it("should not find data sources", () => {
            const res = toHaveProviderWithProperties(synthesizedStack, data_source_1.TestDataSource);
            expect(res.pass).toBeFalsy();
        });
    });
    describe("toBeValidTerraform", () => {
        it("fails if anything but a path is passed", () => {
            const res = (0, matchers_1.toBeValidTerraform)("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = (0, matchers_1.toBeValidTerraform)(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject not to be a valid terraform stack"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = (0, matchers_1.toBeValidTerraform)(result);
            expect(res.pass).toBeFalsy();
            expect(res.message).toEqual(expect.stringContaining("Expected subject to be a valid terraform stack"));
            expect(res.message).toEqual(expect.stringContaining("There are some problems with the configuration, described below."));
            expect(res.message).toEqual(expect.stringContaining("Expected subject to be a valid terraform stack"));
        });
    });
    describe("toPlanSuccessfully", () => {
        it("fails if anything but a path is passed", () => {
            const res = (0, matchers_1.toPlanSuccessfully)("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = (0, matchers_1.toPlanSuccessfully)(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject not to plan successfully"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = (0, matchers_1.toPlanSuccessfully)(result);
            expect(res.pass).toBeFalsy();
            expect(res.message).toEqual(expect.stringContaining("Expected subject to plan successfully"));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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