"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Testing = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const fs = require("fs");
const path = require("path");
const os = require("os");
const stringify = require("json-stable-stringify");
const lib_1 = require("../../lib");
const manifest_1 = require("../manifest");
const features_1 = require("../features");
const jest_1 = require("./adapters/jest");
const synthesizer_1 = require("../synthesize/synthesizer");
const matchers_1 = require("./matchers");
const DefaultTestingAppConfig = {
    stackTraces: false,
    stubVersion: true,
    enableFutureFlags: true,
    fakeCdktfJsonPath: false,
};
/**
 * Testing utilities for cdktf applications.
 */
class Testing {
    /**
     * Returns an app for testing with the following properties:
     * - Output directory is a temp dir.
     */
    static app(options = {}) {
        const appConfig = { ...DefaultTestingAppConfig, ...options };
        if (!appConfig.outdir) {
            appConfig.outdir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf.outdir."));
        }
        const app = new lib_1.App({
            outdir: appConfig.outdir,
            stackTraces: appConfig.stackTraces,
        });
        if (appConfig.stubVersion) {
            this.stubVersion(app);
        }
        if (appConfig.enableFutureFlags) {
            this.enableFutureFlags(app);
        }
        if (appConfig.fakeCdktfJsonPath) {
            this.fakeCdktfJsonPath(app);
        }
        return app;
    }
    static stubVersion(app) {
        app.node.setContext("cdktfVersion", "stubbed");
        app.manifest.version = "stubbed";
        return app;
    }
    static fakeCdktfJsonPath(app) {
        app.node.setContext("cdktfJsonPath", `${process.cwd()}/cdktf.json`);
        return app;
    }
    static enableFutureFlags(app) {
        const node = app.node;
        Object.entries(features_1.FUTURE_FLAGS).forEach(([key, value]) => node.setContext(key, value));
        return app;
    }
    static synthScope(fn) {
        const stack = new lib_1.TerraformStack(Testing.app(), "stack");
        fn(stack);
        return Testing.synth(stack);
    }
    /**
     * Returns the Terraform synthesized JSON.
     */
    static synth(stack, runValidations = false) {
        (0, synthesizer_1.invokeAspects)(stack);
        if (runValidations) {
            stack.runAllValidations();
        }
        const tfConfig = stack.toTerraform();
        // eslint-disable-next-line jsdoc/require-jsdoc
        function removeMetadata(item) {
            if (item !== null && typeof item === "object") {
                if (Array.isArray(item)) {
                    return item.map(removeMetadata);
                }
                const cleanedItem = Object.entries(item)
                    // order alphabetically
                    .sort(([a], [b]) => a.localeCompare(b))
                    .reduce((acc, [key, value]) => ({ ...acc, [key]: removeMetadata(value) }), {});
                // Remove metadata
                delete cleanedItem["//"];
                return cleanedItem;
            }
            return item;
        }
        const cleaned = removeMetadata(tfConfig);
        return stringify(cleaned, { space: 2 });
    }
    static fullSynth(stack) {
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf.outdir."));
        const manifest = new manifest_1.Manifest("stubbed", outdir);
        stack.synthesizer.synthesize({
            outdir,
            manifest,
        });
        manifest.writeToFile();
        return outdir;
    }
    static renderConstructTree(construct) {
        return render(construct, 0, false);
        // eslint-disable-next-line jsdoc/require-jsdoc
        function render(construct, level, isLast) {
            let prefix = "";
            if (level > 0) {
                const spaces = " ".repeat((level - 1) * 4);
                const symbol = isLast ? "└" : "├";
                prefix = `${spaces}${symbol}── `;
            }
            const name = lib_1.App.isApp(construct)
                ? "App"
                : `${construct.node.id} (${construct.constructor.name})`;
            return `${prefix}${name}\n${construct.node.children
                .map((child, idx, arr) => {
                const isLast = idx === arr.length - 1;
                return render(child, level + 1, isLast);
            })
                .join("")}`;
        }
    }
    static toHaveDataSourceWithProperties(received, resourceType, properties = {}) {
        return (0, matchers_1.getToHaveDataSourceWithProperties)()(received, { tfResourceType: resourceType }, properties).pass;
    }
    static toHaveDataSource(received, resourceType) {
        return (0, matchers_1.getToHaveDataSourceWithProperties)()(received, { tfResourceType: resourceType }, {}).pass;
    }
    static toHaveResourceWithProperties(received, resourceType, properties = {}) {
        return (0, matchers_1.getToHaveResourceWithProperties)()(received, { tfResourceType: resourceType }, properties).pass;
    }
    static toHaveResource(received, resourceType) {
        return (0, matchers_1.getToHaveResourceWithProperties)()(received, { tfResourceType: resourceType }, {}).pass;
    }
    static toHaveProviderWithProperties(received, resourceType, properties = {}) {
        return (0, matchers_1.getToHaveProviderWithProperties)()(received, { tfResourceType: resourceType }, properties).pass;
    }
    static toHaveProvider(received, resourceType) {
        return (0, matchers_1.getToHaveProviderWithProperties)()(received, { tfResourceType: resourceType }, {}).pass;
    }
    static toBeValidTerraform(received) {
        return (0, matchers_1.toBeValidTerraform)(received).pass;
    }
    static setupJest() {
        (0, jest_1.setupJest)();
    }
}
_a = JSII_RTTI_SYMBOL_1;
Testing[_a] = { fqn: "cdktf.Testing", version: "0.17.0-pre.18" };
exports.Testing = Testing;
//# sourceMappingURL=data:application/json;base64,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