"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.toPlanSuccessfully = exports.toBeValidTerraform = exports.getToHaveProviderWithProperties = exports.getToHaveResourceWithProperties = exports.getToHaveDataSourceWithProperties = exports.asymetricDeepEqualIgnoringObjectCasing = exports.returnMatcherToJest = exports.AssertionReturn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const fs = require("fs");
const path = require("path");
const child_process_1 = require("child_process");
const util_1 = require("../util");
/**
 * Class representing the contents of a return by an assertion
 */
class AssertionReturn {
    /**
     * Create an AssertionReturn
     * @param message - String message containing information about the result of the assertion
     * @param pass - Boolean pass denoting the success of the assertion
     */
    constructor(message, pass) {
        this.message = message;
        this.pass = pass;
    }
}
_a = JSII_RTTI_SYMBOL_1;
AssertionReturn[_a] = { fqn: "cdktf.testingMatchers.AssertionReturn", version: "0.17.0-pre.18" };
exports.AssertionReturn = AssertionReturn;
/**
 * Reformats the contents of the base testing matcher return type AssertionReturn into type useable by jest
 * @param toReturn
 * @returns {MatcherReturnJest}
 */
function returnMatcherToJest(toReturn) {
    return {
        message: () => toReturn.message,
        pass: toReturn.pass,
    };
}
exports.returnMatcherToJest = returnMatcherToJest;
/**
 * Compares expected and received. All expected properties are matched and considered equal even if
 * there are more properties in the received object than in the expected object in which case it will still return true.
 * @param expected
 * @param received
 * @returns {boolean}
 */
function asymetricDeepEqualIgnoringObjectCasing(expected, received) {
    switch (typeof expected) {
        case "object":
            if (Array.isArray(expected)) {
                return (Array.isArray(received) &&
                    expected.length === received.length &&
                    expected.every((item, index) => asymetricDeepEqualIgnoringObjectCasing(item, received[index]) // recursively compare arrays
                    ));
            }
            if (expected === null && received === null) {
                return true;
            }
            if (expected === undefined && received === undefined) {
                return true;
            }
            if (expected === null || received === null) {
                return false;
            }
            // recursively compare objects and allow snake case as well as camel case
            return Object.keys(expected).every((key) => {
                if (received[key] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[key]);
                }
                if (received[(0, util_1.snakeCase)(key)] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[(0, util_1.snakeCase)(key)]);
                }
                return false;
            });
        default:
            return expected === received;
    }
}
exports.asymetricDeepEqualIgnoringObjectCasing = asymetricDeepEqualIgnoringObjectCasing;
const defaultPassEvaluation = (items, assertedProperties) => {
    return Object.values(items).some((item) => asymetricDeepEqualIgnoringObjectCasing(assertedProperties, item));
};
// eslint-disable-next-line jsdoc/require-jsdoc
function isAsymmetric(obj) {
    return !!obj && typeof obj === "object" && "asymmetricMatch" in obj;
}
// You can use expect.Anything(), expect.ObjectContaining, etc in jest, this makes it nicer to read
// when we print error messages
// eslint-disable-next-line jsdoc/require-jsdoc
function jestAsymetricMatcherStringifyReplacer(_key, value) {
    return isAsymmetric(value) ? `expect.${value.toString()}` : value;
}
// eslint-disable-next-line jsdoc/require-jsdoc
function getAssertElementWithProperties(
// We have the evaluation function configurable so we can make use of the specific testing frameworks capabilities
// This makes the resulting tests more native to the testing framework
customPassEvaluation) {
    const passEvaluation = customPassEvaluation || defaultPassEvaluation;
    return function getAssertElementWithProperties(type, received, itemType, properties = {}) {
        let stack;
        try {
            stack = JSON.parse(received);
        }
        catch (e) {
            throw new Error(`invalid JSON string passed: ${received}`);
        }
        const items = Object.values(Object.entries(stack[type] || {}) // for all data/resource entries
            .find(
        // find the object with a matching name
        ([type, _values]) => type === itemType.tfResourceType)?.[1] || {} // get all items of that type (encoded as a record of name -> config)
        ) || []; // get a list of all configs of that type
        const pass = passEvaluation(items, properties);
        if (pass) {
            return new AssertionReturn(`Expected no ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesized stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`, pass);
        }
        else {
            return new AssertionReturn(`Expected ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesized stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`, pass);
        }
    };
}
/**
 * Returns the function toHaveDataSourceWithProperties using the evaluation properties of customPassEvaluation
 * @param customPassEvaluation
 * @returns {getToHaveDataSourceWithProperties~toHaveDataSourceWithProperties}
 */
function getToHaveDataSourceWithProperties(customPassEvaluation) {
    /**
     * Evaluates the received stack to have the data source resourceType containing specified properties
     * @param received
     * @param resourceType
     * @param properties
     * @returns {AssertionReturn}
     */
    return function toHaveDataSourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("data", received, resourceType, properties);
    };
}
exports.getToHaveDataSourceWithProperties = getToHaveDataSourceWithProperties;
/**
 * Returns the function toHaveResourceWithProperties using the evaluation properties of customPassEvaluation
 * @param customPassEvaluation
 * @returns
 */
function getToHaveResourceWithProperties(customPassEvaluation) {
    /**
     * Evaluates the received stack to have the resource resourceType containing specified properties
     * @param received
     * @param resourceType
     * @param properties
     * @returns {AssertionReturn}
     */
    return function toHaveResourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("resource", received, resourceType, properties);
    };
}
exports.getToHaveResourceWithProperties = getToHaveResourceWithProperties;
/**
 * A helper util to verify wether an Error was caused by the Nodejs `process.spawn` API.
 *
 * @param   {Error}   err The Error object to verify
 * @returns {Boolean}     A bool indicating wether the input Error is containing process.spawn output.
 */
const isExecSpawnError = (err) => "output" in err &&
    Array.isArray(err.output) &&
    err.output.some((buf) => Buffer.isBuffer(buf));
/**
 * A helper util to append `process.spawn` output to assertion messages to improve developer expirience.
 *
 * @param   {String} message The message to optionally append process output to.
 * @param   {Error}  err     The error from which the `process.spawn` output should be retreived from.
 * @returns {String}         The finalized assertion message decorated with the `process.spawn` output.
 */
const withProcessOutput = (message, err) => {
    let output = "";
    if (isExecSpawnError(err)) {
        output =
            err.output
                ?.map((buffer) => buffer?.toString("utf8"))
                .filter(Boolean)
                .join("\n") ?? "";
    }
    const appendix = output.length ? `. Output: ${output}` : "";
    return `${message}: ${err}${appendix}.`;
};
/**
 * Returns the function toHaveProviderWithProperties using the evaluation properties of customPassEvaluation
 * @param customPassEvaluation
 * @returns {getToHaveProviderWithProperties~toHaveProviderWithProperties}
 */
function getToHaveProviderWithProperties(customPassEvaluation) {
    /**
     * Evaluates the received stack to have the provider resourceType containing specified properties
     * @param received
     * @param resourceType
     * @param properties
     * @returns {AssertionReturn}
     */
    return function toHaveProviderWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("provider", received, resourceType, properties);
    };
}
exports.getToHaveProviderWithProperties = getToHaveProviderWithProperties;
/**
 * Evaluates the validity of the received stack
 * @param received
 * @returns {AssertionReturn}
 */
function toBeValidTerraform(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw new Error("Path is not a directory");
        }
    }
    catch (e) {
        return new AssertionReturn(`Expected subject to be a terraform directory: ${e}`, false);
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([name, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "pipe",
            };
            (0, child_process_1.execSync)(`${util_1.terraformBinaryName} init`, opts);
            const out = (0, child_process_1.execSync)(`${util_1.terraformBinaryName} validate -json`, opts);
            const result = JSON.parse(out.toString());
            if (!result.valid) {
                throw new Error(`Found ${result.error_count} Errors in stack ${name}: ${result.diagnostics.join("\n")}`);
            }
        });
        return new AssertionReturn(`Expected subject not to be a valid terraform stack`, true);
    }
    catch (e) {
        return new AssertionReturn(withProcessOutput(`Expected subject to be a valid terraform stack`, e), false);
    }
}
exports.toBeValidTerraform = toBeValidTerraform;
/**
 * Evaluates the ability for the received stack to successfully plan
 * @param received
 * @returns {AssertionReturn}
 */
function toPlanSuccessfully(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw new Error("Path is not a directory");
        }
    }
    catch (e) {
        return new AssertionReturn(`Expected subject to be a terraform directory: ${e}`, false);
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "ignore",
            };
            (0, child_process_1.execSync)(`${util_1.terraformBinaryName} init`, opts);
            // Throws on a non-zero exit code
            (0, child_process_1.execSync)(`${util_1.terraformBinaryName} plan -input=false -lock=false `, opts);
        });
        return new AssertionReturn(`Expected subject not to plan successfully`, true);
    }
    catch (e) {
        return new AssertionReturn(withProcessOutput(`Expected subject to plan successfully`, e), false);
    }
}
exports.toPlanSuccessfully = toPlanSuccessfully;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWF0Y2hlcnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtYXRjaGVycy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLCtCQUErQjtBQUMvQixtQ0FBbUM7QUFDbkMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QixpREFBMkQ7QUFDM0Qsa0NBQXlEO0FBYXpEOztHQUVHO0FBQ0gsTUFBYSxlQUFlO0lBQzFCOzs7O09BSUc7SUFDSCxZQUE0QixPQUFlLEVBQWtCLElBQWE7UUFBOUMsWUFBTyxHQUFQLE9BQU8sQ0FBUTtRQUFrQixTQUFJLEdBQUosSUFBSSxDQUFTO0lBQUcsQ0FBQzs7OztBQU5uRSwwQ0FBZTtBQVc1Qjs7OztHQUlHO0FBQ0gsU0FBZ0IsbUJBQW1CLENBQ2pDLFFBQXlCO0lBRXpCLE9BQU87UUFDTCxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU87UUFDL0IsSUFBSSxFQUFFLFFBQVEsQ0FBQyxJQUFJO0tBQ3BCLENBQUM7QUFDSixDQUFDO0FBUEQsa0RBT0M7QUFFRDs7Ozs7O0dBTUc7QUFDSCxTQUFnQixzQ0FBc0MsQ0FDcEQsUUFBaUIsRUFDakIsUUFBaUI7SUFFakIsUUFBUSxPQUFPLFFBQVEsRUFBRTtRQUN2QixLQUFLLFFBQVE7WUFDWCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUU7Z0JBQzNCLE9BQU8sQ0FDTCxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztvQkFDdkIsUUFBUSxDQUFDLE1BQU0sS0FBSyxRQUFRLENBQUMsTUFBTTtvQkFDbkMsUUFBUSxDQUFDLEtBQUssQ0FDWixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsRUFBRSxDQUNkLHNDQUFzQyxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyw2QkFBNkI7cUJBQzlGLENBQ0YsQ0FBQzthQUNIO1lBQ0QsSUFBSSxRQUFRLEtBQUssSUFBSSxJQUFJLFFBQVEsS0FBSyxJQUFJLEVBQUU7Z0JBQzFDLE9BQU8sSUFBSSxDQUFDO2FBQ2I7WUFDRCxJQUFJLFFBQVEsS0FBSyxTQUFTLElBQUksUUFBUSxLQUFLLFNBQVMsRUFBRTtnQkFDcEQsT0FBTyxJQUFJLENBQUM7YUFDYjtZQUNELElBQUksUUFBUSxLQUFLLElBQUksSUFBSSxRQUFRLEtBQUssSUFBSSxFQUFFO2dCQUMxQyxPQUFPLEtBQUssQ0FBQzthQUNkO1lBRUQseUVBQXlFO1lBQ3pFLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFtQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7Z0JBQ3BFLElBQUssUUFBZ0IsQ0FBQyxHQUFHLENBQUMsS0FBSyxTQUFTLEVBQUU7b0JBQ3hDLE9BQU8sc0NBQXNDLENBQzFDLFFBQWdCLENBQUMsR0FBRyxDQUFDLEVBQ3JCLFFBQWdCLENBQUMsR0FBRyxDQUFDLENBQ3ZCLENBQUM7aUJBQ0g7Z0JBRUQsSUFBSyxRQUFnQixDQUFDLElBQUEsZ0JBQVMsRUFBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLFNBQVMsRUFBRTtvQkFDbkQsT0FBTyxzQ0FBc0MsQ0FDMUMsUUFBZ0IsQ0FBQyxHQUFHLENBQUMsRUFDckIsUUFBZ0IsQ0FBQyxJQUFBLGdCQUFTLEVBQUMsR0FBRyxDQUFDLENBQUMsQ0FDbEMsQ0FBQztpQkFDSDtnQkFFRCxPQUFPLEtBQUssQ0FBQztZQUNmLENBQUMsQ0FBQyxDQUFDO1FBQ0w7WUFDRSxPQUFPLFFBQVEsS0FBSyxRQUFRLENBQUM7S0FDaEM7QUFDSCxDQUFDO0FBL0NELHdGQStDQztBQUNELE1BQU0scUJBQXFCLEdBQUcsQ0FDNUIsS0FBVSxFQUNWLGtCQUF1QyxFQUN2QyxFQUFFO0lBQ0YsT0FBTyxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQVMsRUFBRSxFQUFFLENBQzdDLHNDQUFzQyxDQUFDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxDQUNqRSxDQUFDO0FBQ0osQ0FBQyxDQUFDO0FBRUYsK0NBQStDO0FBQy9DLFNBQVMsWUFBWSxDQUFDLEdBQVE7SUFDNUIsT0FBTyxDQUFDLENBQUMsR0FBRyxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxpQkFBaUIsSUFBSSxHQUFHLENBQUM7QUFDdEUsQ0FBQztBQUNELG1HQUFtRztBQUNuRywrQkFBK0I7QUFDL0IsK0NBQStDO0FBQy9DLFNBQVMscUNBQXFDLENBQUMsSUFBWSxFQUFFLEtBQVU7SUFDckUsT0FBTyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsS0FBSyxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQztBQUNwRSxDQUFDO0FBQ0QsK0NBQStDO0FBQy9DLFNBQVMsOEJBQThCO0FBQ3JDLGtIQUFrSDtBQUNsSCxzRUFBc0U7QUFDdEUsb0JBR1k7SUFFWixNQUFNLGNBQWMsR0FBRyxvQkFBb0IsSUFBSSxxQkFBcUIsQ0FBQztJQUNyRSxPQUFPLFNBQVMsOEJBQThCLENBQzVDLElBQTRCLEVBQzVCLFFBQWdCLEVBQ2hCLFFBQThCLEVBQzlCLGFBQWtDLEVBQUU7UUFFcEMsSUFBSSxLQUF1QixDQUFDO1FBQzVCLElBQUk7WUFDRixLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQXFCLENBQUM7U0FDbEQ7UUFBQyxPQUFPLENBQUMsRUFBRTtZQUNWLE1BQU0sSUFBSSxLQUFLLENBQUMsK0JBQStCLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDNUQ7UUFFRCxNQUFNLEtBQUssR0FDVCxNQUFNLENBQUMsTUFBTSxDQUNYLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLGdDQUFnQzthQUMvRCxJQUFJO1FBQ0gsdUNBQXVDO1FBQ3ZDLENBQUMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLEVBQUUsRUFBRSxDQUFDLElBQUksS0FBSyxRQUFRLENBQUMsY0FBYyxDQUN0RCxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLHFFQUFxRTtTQUNyRixJQUFJLEVBQUUsQ0FBQyxDQUFDLHlDQUF5QztRQUNwRCxNQUFNLElBQUksR0FBRyxjQUFjLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLElBQUksSUFBSSxFQUFFO1lBQ1IsT0FBTyxJQUFJLGVBQWUsQ0FDeEIsZUFDRSxRQUFRLENBQUMsY0FDWCxvQkFBb0IsSUFBSSxDQUFDLFNBQVMsQ0FDaEMsVUFBVSxFQUNWLHFDQUFxQyxDQUN0QztRQUNELEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQ3RDLFFBQVEsQ0FBQyxjQUNYLHFCQUNFLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUM5RCxFQUFFLEVBQ0YsSUFBSSxDQUNMLENBQUM7U0FDSDthQUFNO1lBQ0wsT0FBTyxJQUFJLGVBQWUsQ0FDeEIsWUFBWSxRQUFRLENBQUMsY0FBYyxvQkFBb0IsSUFBSSxDQUFDLFNBQVMsQ0FDbkUsVUFBVSxFQUNWLHFDQUFxQyxDQUN0QztRQUNELEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQ3RDLFFBQVEsQ0FBQyxjQUNYLHFCQUNFLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUM5RCxFQUFFLEVBQ0YsSUFBSSxDQUNMLENBQUM7U0FDSDtJQUNILENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBZ0IsaUNBQWlDLENBQy9DLG9CQUdZO0lBRVo7Ozs7OztPQU1HO0lBQ0gsT0FBTyxTQUFTLDhCQUE4QixDQUM1QyxRQUFnQixFQUNoQixZQUFrQyxFQUNsQyxhQUFrQyxFQUFFO1FBRXBDLE9BQU8sOEJBQThCLENBQUMsb0JBQW9CLENBQUMsQ0FDekQsTUFBTSxFQUNOLFFBQVEsRUFDUixZQUFZLEVBQ1osVUFBVSxDQUNYLENBQUM7SUFDSixDQUFDLENBQUM7QUFDSixDQUFDO0FBekJELDhFQXlCQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQiwrQkFBK0IsQ0FDN0Msb0JBR1k7SUFFWjs7Ozs7O09BTUc7SUFDSCxPQUFPLFNBQVMsNEJBQTRCLENBQzFDLFFBQWdCLEVBQ2hCLFlBQWtDLEVBQ2xDLGFBQWtDLEVBQUU7UUFFcEMsT0FBTyw4QkFBOEIsQ0FBQyxvQkFBb0IsQ0FBQyxDQUN6RCxVQUFVLEVBQ1YsUUFBUSxFQUNSLFlBQVksRUFDWixVQUFVLENBQ1gsQ0FBQztJQUNKLENBQUMsQ0FBQztBQUNKLENBQUM7QUF6QkQsMEVBeUJDO0FBRUQ7Ozs7O0dBS0c7QUFDSCxNQUFNLGdCQUFnQixHQUFHLENBQUMsR0FBUSxFQUF3QyxFQUFFLENBQzFFLFFBQVEsSUFBSSxHQUFHO0lBQ2YsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDO0lBQ3pCLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBUSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7QUFFdEQ7Ozs7OztHQU1HO0FBQ0gsTUFBTSxpQkFBaUIsR0FBRyxDQUFDLE9BQWUsRUFBRSxHQUFZLEVBQUUsRUFBRTtJQUMxRCxJQUFJLE1BQU0sR0FBRyxFQUFFLENBQUM7SUFFaEIsSUFBSSxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUN6QixNQUFNO1lBQ0osR0FBRyxDQUFDLE1BQU07Z0JBQ1IsRUFBRSxHQUFHLENBQUMsQ0FBQyxNQUFjLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUM7aUJBQ2xELE1BQU0sQ0FBQyxPQUFPLENBQUM7aUJBQ2YsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztLQUN2QjtJQUVELE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGFBQWEsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUU1RCxPQUFPLEdBQUcsT0FBTyxLQUFLLEdBQUcsR0FBRyxRQUFRLEdBQUcsQ0FBQztBQUMxQyxDQUFDLENBQUM7QUFFRjs7OztHQUlHO0FBQ0gsU0FBZ0IsK0JBQStCLENBQzdDLG9CQUdZO0lBRVo7Ozs7OztPQU1HO0lBQ0gsT0FBTyxTQUFTLDRCQUE0QixDQUMxQyxRQUFnQixFQUNoQixZQUFrQyxFQUNsQyxhQUFrQyxFQUFFO1FBRXBDLE9BQU8sOEJBQThCLENBQUMsb0JBQW9CLENBQUMsQ0FDekQsVUFBVSxFQUNWLFFBQVEsRUFDUixZQUFZLEVBQ1osVUFBVSxDQUNYLENBQUM7SUFDSixDQUFDLENBQUM7QUFDSixDQUFDO0FBekJELDBFQXlCQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixrQkFBa0IsQ0FBQyxRQUFnQjtJQUNqRCxJQUFJO1FBQ0YsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1NBQzVDO0tBQ0Y7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE9BQU8sSUFBSSxlQUFlLENBQ3hCLGlEQUFpRCxDQUFDLEVBQUUsRUFDcEQsS0FBSyxDQUNOLENBQUM7S0FDSDtJQUVELElBQUk7UUFDRixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUN6QixFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLGVBQWUsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUNqRSxDQUFDO1FBRUYsTUFBTSxNQUFNLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFL0MsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEdBQUc7Z0JBQ1gsR0FBRyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFHLEtBQWEsQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDNUQsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO2dCQUNoQixLQUFLLEVBQUUsTUFBTTthQUNQLENBQUM7WUFDVCxJQUFBLHdCQUFRLEVBQUMsR0FBRywwQkFBbUIsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQzlDLE1BQU0sR0FBRyxHQUFHLElBQUEsd0JBQVEsRUFBQyxHQUFHLDBCQUFtQixpQkFBaUIsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUVwRSxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1lBQzFDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFO2dCQUNqQixNQUFNLElBQUksS0FBSyxDQUNiLFNBQ0UsTUFBTSxDQUFDLFdBQ1Qsb0JBQW9CLElBQUksS0FBSyxNQUFNLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUM3RCxDQUFDO2FBQ0g7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxlQUFlLENBQ3hCLG9EQUFvRCxFQUNwRCxJQUFJLENBQ0wsQ0FBQztLQUNIO0lBQUMsT0FBTyxDQUFDLEVBQUU7UUFDVixPQUFPLElBQUksZUFBZSxDQUN4QixpQkFBaUIsQ0FBQyxnREFBZ0QsRUFBRSxDQUFDLENBQUMsRUFDdEUsS0FBSyxDQUNOLENBQUM7S0FDSDtBQUNILENBQUM7QUEvQ0QsZ0RBK0NDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQWdCLGtCQUFrQixDQUFDLFFBQWdCO0lBQ2pELElBQUk7UUFDRixJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLHlCQUF5QixDQUFDLENBQUM7U0FDNUM7S0FDRjtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsT0FBTyxJQUFJLGVBQWUsQ0FDeEIsaURBQWlELENBQUMsRUFBRSxFQUNwRCxLQUFLLENBQ04sQ0FBQztLQUNIO0lBRUQsSUFBSTtRQUNGLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQ3pCLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsZUFBZSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQ2pFLENBQUM7UUFFRixNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUvQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEdBQUc7Z0JBQ1gsR0FBRyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFHLEtBQWEsQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDNUQsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO2dCQUNoQixLQUFLLEVBQUUsUUFBUTthQUNULENBQUM7WUFDVCxJQUFBLHdCQUFRLEVBQUMsR0FBRywwQkFBbUIsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDO1lBRTlDLGlDQUFpQztZQUNqQyxJQUFBLHdCQUFRLEVBQUMsR0FBRywwQkFBbUIsaUNBQWlDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDMUUsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLElBQUksZUFBZSxDQUN4QiwyQ0FBMkMsRUFDM0MsSUFBSSxDQUNMLENBQUM7S0FDSDtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsT0FBTyxJQUFJLGVBQWUsQ0FDeEIsaUJBQWlCLENBQUMsdUNBQXVDLEVBQUUsQ0FBQyxDQUFDLEVBQzdELEtBQUssQ0FDTixDQUFDO0tBQ0g7QUFDSCxDQUFDO0FBekNELGdEQXlDQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmNcbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNUEwtMi4wXG5pbXBvcnQgKiBhcyBmcyBmcm9tIFwiZnNcIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IGV4ZWNTeW5jLCBTcGF3blN5bmNSZXR1cm5zIH0gZnJvbSBcImNoaWxkX3Byb2Nlc3NcIjtcbmltcG9ydCB7IHNuYWtlQ2FzZSwgdGVycmFmb3JtQmluYXJ5TmFtZSB9IGZyb20gXCIuLi91dGlsXCI7XG5cbi8vIFRlcnJhZm9ybUNvbnN0cnVjdG9yIGlzIGNsYXNzIHdpdGggdGhlIHN0YXRpYyBwcm9wZXJ0eSAndGZSZXNvdXJjZVR5cGUnXG5leHBvcnQgaW50ZXJmYWNlIFRlcnJhZm9ybUNvbnN0cnVjdG9yIHtcbiAgcmVhZG9ubHkgdGZSZXNvdXJjZVR5cGU6IHN0cmluZztcbn1cblxuZXhwb3J0IHR5cGUgU3ludGhlc2l6ZWRTdGFjayA9IHtcbiAgcmVzb3VyY2U6IFJlY29yZDxzdHJpbmcsIGFueT47XG4gIGRhdGE6IFJlY29yZDxzdHJpbmcsIGFueT47XG4gIHByb3ZpZGVyOiBSZWNvcmQ8c3RyaW5nLCBhbnk+O1xufTtcblxuLyoqXG4gKiBDbGFzcyByZXByZXNlbnRpbmcgdGhlIGNvbnRlbnRzIG9mIGEgcmV0dXJuIGJ5IGFuIGFzc2VydGlvblxuICovXG5leHBvcnQgY2xhc3MgQXNzZXJ0aW9uUmV0dXJuIHtcbiAgLyoqXG4gICAqIENyZWF0ZSBhbiBBc3NlcnRpb25SZXR1cm5cbiAgICogQHBhcmFtIG1lc3NhZ2UgLSBTdHJpbmcgbWVzc2FnZSBjb250YWluaW5nIGluZm9ybWF0aW9uIGFib3V0IHRoZSByZXN1bHQgb2YgdGhlIGFzc2VydGlvblxuICAgKiBAcGFyYW0gcGFzcyAtIEJvb2xlYW4gcGFzcyBkZW5vdGluZyB0aGUgc3VjY2VzcyBvZiB0aGUgYXNzZXJ0aW9uXG4gICAqL1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgbWVzc2FnZTogc3RyaW5nLCBwdWJsaWMgcmVhZG9ubHkgcGFzczogYm9vbGVhbikge31cbn1cblxuZXhwb3J0IHR5cGUgTWF0Y2hlclJldHVybkplc3QgPSB7IG1lc3NhZ2U6ICgpID0+IHN0cmluZzsgcGFzczogYm9vbGVhbiB9O1xuXG4vKipcbiAqIFJlZm9ybWF0cyB0aGUgY29udGVudHMgb2YgdGhlIGJhc2UgdGVzdGluZyBtYXRjaGVyIHJldHVybiB0eXBlIEFzc2VydGlvblJldHVybiBpbnRvIHR5cGUgdXNlYWJsZSBieSBqZXN0XG4gKiBAcGFyYW0gdG9SZXR1cm5cbiAqIEByZXR1cm5zIHtNYXRjaGVyUmV0dXJuSmVzdH1cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJldHVybk1hdGNoZXJUb0plc3QoXG4gIHRvUmV0dXJuOiBBc3NlcnRpb25SZXR1cm5cbik6IE1hdGNoZXJSZXR1cm5KZXN0IHtcbiAgcmV0dXJuIHtcbiAgICBtZXNzYWdlOiAoKSA9PiB0b1JldHVybi5tZXNzYWdlLFxuICAgIHBhc3M6IHRvUmV0dXJuLnBhc3MsXG4gIH07XG59XG5cbi8qKlxuICogQ29tcGFyZXMgZXhwZWN0ZWQgYW5kIHJlY2VpdmVkLiBBbGwgZXhwZWN0ZWQgcHJvcGVydGllcyBhcmUgbWF0Y2hlZCBhbmQgY29uc2lkZXJlZCBlcXVhbCBldmVuIGlmXG4gKiB0aGVyZSBhcmUgbW9yZSBwcm9wZXJ0aWVzIGluIHRoZSByZWNlaXZlZCBvYmplY3QgdGhhbiBpbiB0aGUgZXhwZWN0ZWQgb2JqZWN0IGluIHdoaWNoIGNhc2UgaXQgd2lsbCBzdGlsbCByZXR1cm4gdHJ1ZS5cbiAqIEBwYXJhbSBleHBlY3RlZFxuICogQHBhcmFtIHJlY2VpdmVkXG4gKiBAcmV0dXJucyB7Ym9vbGVhbn1cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGFzeW1ldHJpY0RlZXBFcXVhbElnbm9yaW5nT2JqZWN0Q2FzaW5nKFxuICBleHBlY3RlZDogdW5rbm93bixcbiAgcmVjZWl2ZWQ6IHVua25vd25cbik6IGJvb2xlYW4ge1xuICBzd2l0Y2ggKHR5cGVvZiBleHBlY3RlZCkge1xuICAgIGNhc2UgXCJvYmplY3RcIjpcbiAgICAgIGlmIChBcnJheS5pc0FycmF5KGV4cGVjdGVkKSkge1xuICAgICAgICByZXR1cm4gKFxuICAgICAgICAgIEFycmF5LmlzQXJyYXkocmVjZWl2ZWQpICYmXG4gICAgICAgICAgZXhwZWN0ZWQubGVuZ3RoID09PSByZWNlaXZlZC5sZW5ndGggJiZcbiAgICAgICAgICBleHBlY3RlZC5ldmVyeShcbiAgICAgICAgICAgIChpdGVtLCBpbmRleCkgPT5cbiAgICAgICAgICAgICAgYXN5bWV0cmljRGVlcEVxdWFsSWdub3JpbmdPYmplY3RDYXNpbmcoaXRlbSwgcmVjZWl2ZWRbaW5kZXhdKSAvLyByZWN1cnNpdmVseSBjb21wYXJlIGFycmF5c1xuICAgICAgICAgIClcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICAgIGlmIChleHBlY3RlZCA9PT0gbnVsbCAmJiByZWNlaXZlZCA9PT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgIH1cbiAgICAgIGlmIChleHBlY3RlZCA9PT0gdW5kZWZpbmVkICYmIHJlY2VpdmVkID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9XG4gICAgICBpZiAoZXhwZWN0ZWQgPT09IG51bGwgfHwgcmVjZWl2ZWQgPT09IG51bGwpIHtcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuXG4gICAgICAvLyByZWN1cnNpdmVseSBjb21wYXJlIG9iamVjdHMgYW5kIGFsbG93IHNuYWtlIGNhc2UgYXMgd2VsbCBhcyBjYW1lbCBjYXNlXG4gICAgICByZXR1cm4gT2JqZWN0LmtleXMoZXhwZWN0ZWQgYXMgUmVjb3JkPHN0cmluZywgdW5rbm93bj4pLmV2ZXJ5KChrZXkpID0+IHtcbiAgICAgICAgaWYgKChyZWNlaXZlZCBhcyBhbnkpW2tleV0gIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHJldHVybiBhc3ltZXRyaWNEZWVwRXF1YWxJZ25vcmluZ09iamVjdENhc2luZyhcbiAgICAgICAgICAgIChleHBlY3RlZCBhcyBhbnkpW2tleV0sXG4gICAgICAgICAgICAocmVjZWl2ZWQgYXMgYW55KVtrZXldXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICgocmVjZWl2ZWQgYXMgYW55KVtzbmFrZUNhc2Uoa2V5KV0gIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHJldHVybiBhc3ltZXRyaWNEZWVwRXF1YWxJZ25vcmluZ09iamVjdENhc2luZyhcbiAgICAgICAgICAgIChleHBlY3RlZCBhcyBhbnkpW2tleV0sXG4gICAgICAgICAgICAocmVjZWl2ZWQgYXMgYW55KVtzbmFrZUNhc2Uoa2V5KV1cbiAgICAgICAgICApO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfSk7XG4gICAgZGVmYXVsdDpcbiAgICAgIHJldHVybiBleHBlY3RlZCA9PT0gcmVjZWl2ZWQ7XG4gIH1cbn1cbmNvbnN0IGRlZmF1bHRQYXNzRXZhbHVhdGlvbiA9IChcbiAgaXRlbXM6IGFueSxcbiAgYXNzZXJ0ZWRQcm9wZXJ0aWVzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+XG4pID0+IHtcbiAgcmV0dXJuIE9iamVjdC52YWx1ZXMoaXRlbXMpLnNvbWUoKGl0ZW06IGFueSkgPT5cbiAgICBhc3ltZXRyaWNEZWVwRXF1YWxJZ25vcmluZ09iamVjdENhc2luZyhhc3NlcnRlZFByb3BlcnRpZXMsIGl0ZW0pXG4gICk7XG59O1xuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZnVuY3Rpb24gaXNBc3ltbWV0cmljKG9iajogYW55KSB7XG4gIHJldHVybiAhIW9iaiAmJiB0eXBlb2Ygb2JqID09PSBcIm9iamVjdFwiICYmIFwiYXN5bW1ldHJpY01hdGNoXCIgaW4gb2JqO1xufVxuLy8gWW91IGNhbiB1c2UgZXhwZWN0LkFueXRoaW5nKCksIGV4cGVjdC5PYmplY3RDb250YWluaW5nLCBldGMgaW4gamVzdCwgdGhpcyBtYWtlcyBpdCBuaWNlciB0byByZWFkXG4vLyB3aGVuIHdlIHByaW50IGVycm9yIG1lc3NhZ2VzXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZnVuY3Rpb24gamVzdEFzeW1ldHJpY01hdGNoZXJTdHJpbmdpZnlSZXBsYWNlcihfa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpIHtcbiAgcmV0dXJuIGlzQXN5bW1ldHJpYyh2YWx1ZSkgPyBgZXhwZWN0LiR7dmFsdWUudG9TdHJpbmcoKX1gIDogdmFsdWU7XG59XG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZnVuY3Rpb24gZ2V0QXNzZXJ0RWxlbWVudFdpdGhQcm9wZXJ0aWVzKFxuICAvLyBXZSBoYXZlIHRoZSBldmFsdWF0aW9uIGZ1bmN0aW9uIGNvbmZpZ3VyYWJsZSBzbyB3ZSBjYW4gbWFrZSB1c2Ugb2YgdGhlIHNwZWNpZmljIHRlc3RpbmcgZnJhbWV3b3JrcyBjYXBhYmlsaXRpZXNcbiAgLy8gVGhpcyBtYWtlcyB0aGUgcmVzdWx0aW5nIHRlc3RzIG1vcmUgbmF0aXZlIHRvIHRoZSB0ZXN0aW5nIGZyYW1ld29ya1xuICBjdXN0b21QYXNzRXZhbHVhdGlvbj86IChcbiAgICBpdGVtczogYW55W10sIC8vIGNvbmZpZ3VyYXRpb25zIG9mIHRoZSByZXF1ZXN0ZWQgdHlwZVxuICAgIGFzc2VydGVkUHJvcGVydGllczogUmVjb3JkPHN0cmluZywgYW55PlxuICApID0+IGJvb2xlYW5cbikge1xuICBjb25zdCBwYXNzRXZhbHVhdGlvbiA9IGN1c3RvbVBhc3NFdmFsdWF0aW9uIHx8IGRlZmF1bHRQYXNzRXZhbHVhdGlvbjtcbiAgcmV0dXJuIGZ1bmN0aW9uIGdldEFzc2VydEVsZW1lbnRXaXRoUHJvcGVydGllcyhcbiAgICB0eXBlOiBrZXlvZiBTeW50aGVzaXplZFN0YWNrLFxuICAgIHJlY2VpdmVkOiBzdHJpbmcsXG4gICAgaXRlbVR5cGU6IFRlcnJhZm9ybUNvbnN0cnVjdG9yLFxuICAgIHByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fVxuICApOiBBc3NlcnRpb25SZXR1cm4ge1xuICAgIGxldCBzdGFjazogU3ludGhlc2l6ZWRTdGFjaztcbiAgICB0cnkge1xuICAgICAgc3RhY2sgPSBKU09OLnBhcnNlKHJlY2VpdmVkKSBhcyBTeW50aGVzaXplZFN0YWNrO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgaW52YWxpZCBKU09OIHN0cmluZyBwYXNzZWQ6ICR7cmVjZWl2ZWR9YCk7XG4gICAgfVxuXG4gICAgY29uc3QgaXRlbXMgPVxuICAgICAgT2JqZWN0LnZhbHVlcyhcbiAgICAgICAgT2JqZWN0LmVudHJpZXMoc3RhY2tbdHlwZV0gfHwge30pIC8vIGZvciBhbGwgZGF0YS9yZXNvdXJjZSBlbnRyaWVzXG4gICAgICAgICAgLmZpbmQoXG4gICAgICAgICAgICAvLyBmaW5kIHRoZSBvYmplY3Qgd2l0aCBhIG1hdGNoaW5nIG5hbWVcbiAgICAgICAgICAgIChbdHlwZSwgX3ZhbHVlc10pID0+IHR5cGUgPT09IGl0ZW1UeXBlLnRmUmVzb3VyY2VUeXBlXG4gICAgICAgICAgKT8uWzFdIHx8IHt9IC8vIGdldCBhbGwgaXRlbXMgb2YgdGhhdCB0eXBlIChlbmNvZGVkIGFzIGEgcmVjb3JkIG9mIG5hbWUgLT4gY29uZmlnKVxuICAgICAgKSB8fCBbXTsgLy8gZ2V0IGEgbGlzdCBvZiBhbGwgY29uZmlncyBvZiB0aGF0IHR5cGVcbiAgICBjb25zdCBwYXNzID0gcGFzc0V2YWx1YXRpb24oaXRlbXMsIHByb3BlcnRpZXMpO1xuICAgIGlmIChwYXNzKSB7XG4gICAgICByZXR1cm4gbmV3IEFzc2VydGlvblJldHVybihcbiAgICAgICAgYEV4cGVjdGVkIG5vICR7XG4gICAgICAgICAgaXRlbVR5cGUudGZSZXNvdXJjZVR5cGVcbiAgICAgICAgfSB3aXRoIHByb3BlcnRpZXMgJHtKU09OLnN0cmluZ2lmeShcbiAgICAgICAgICBwcm9wZXJ0aWVzLFxuICAgICAgICAgIGplc3RBc3ltZXRyaWNNYXRjaGVyU3RyaW5naWZ5UmVwbGFjZXJcbiAgICAgICAgKX0gdG8gYmUgcHJlc2VudCBpbiBzeW50aGVzaXplZCBzdGFjay5cbkZvdW5kICR7aXRlbXMubGVuZ3RoID09PSAwID8gXCJub1wiIDogaXRlbXMubGVuZ3RofSAke1xuICAgICAgICAgIGl0ZW1UeXBlLnRmUmVzb3VyY2VUeXBlXG4gICAgICAgIH0gcmVzb3VyY2VzIGluc3RlYWQke1xuICAgICAgICAgIGl0ZW1zLmxlbmd0aCA+IDAgPyBcIjpcXG5cIiArIEpTT04uc3RyaW5naWZ5KGl0ZW1zLCBudWxsLCAyKSA6IFwiXCJcbiAgICAgICAgfWAsXG4gICAgICAgIHBhc3NcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBuZXcgQXNzZXJ0aW9uUmV0dXJuKFxuICAgICAgICBgRXhwZWN0ZWQgJHtpdGVtVHlwZS50ZlJlc291cmNlVHlwZX0gd2l0aCBwcm9wZXJ0aWVzICR7SlNPTi5zdHJpbmdpZnkoXG4gICAgICAgICAgcHJvcGVydGllcyxcbiAgICAgICAgICBqZXN0QXN5bWV0cmljTWF0Y2hlclN0cmluZ2lmeVJlcGxhY2VyXG4gICAgICAgICl9IHRvIGJlIHByZXNlbnQgaW4gc3ludGhlc2l6ZWQgc3RhY2suXG5Gb3VuZCAke2l0ZW1zLmxlbmd0aCA9PT0gMCA/IFwibm9cIiA6IGl0ZW1zLmxlbmd0aH0gJHtcbiAgICAgICAgICBpdGVtVHlwZS50ZlJlc291cmNlVHlwZVxuICAgICAgICB9IHJlc291cmNlcyBpbnN0ZWFkJHtcbiAgICAgICAgICBpdGVtcy5sZW5ndGggPiAwID8gXCI6XFxuXCIgKyBKU09OLnN0cmluZ2lmeShpdGVtcywgbnVsbCwgMikgOiBcIlwiXG4gICAgICAgIH1gLFxuICAgICAgICBwYXNzXG4gICAgICApO1xuICAgIH1cbiAgfTtcbn1cblxuLyoqXG4gKiBSZXR1cm5zIHRoZSBmdW5jdGlvbiB0b0hhdmVEYXRhU291cmNlV2l0aFByb3BlcnRpZXMgdXNpbmcgdGhlIGV2YWx1YXRpb24gcHJvcGVydGllcyBvZiBjdXN0b21QYXNzRXZhbHVhdGlvblxuICogQHBhcmFtIGN1c3RvbVBhc3NFdmFsdWF0aW9uXG4gKiBAcmV0dXJucyB7Z2V0VG9IYXZlRGF0YVNvdXJjZVdpdGhQcm9wZXJ0aWVzfnRvSGF2ZURhdGFTb3VyY2VXaXRoUHJvcGVydGllc31cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldFRvSGF2ZURhdGFTb3VyY2VXaXRoUHJvcGVydGllcyhcbiAgY3VzdG9tUGFzc0V2YWx1YXRpb24/OiAoXG4gICAgaXRlbXM6IGFueSxcbiAgICBhc3NlcnRlZFByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT5cbiAgKSA9PiBib29sZWFuXG4pIHtcbiAgLyoqXG4gICAqIEV2YWx1YXRlcyB0aGUgcmVjZWl2ZWQgc3RhY2sgdG8gaGF2ZSB0aGUgZGF0YSBzb3VyY2UgcmVzb3VyY2VUeXBlIGNvbnRhaW5pbmcgc3BlY2lmaWVkIHByb3BlcnRpZXNcbiAgICogQHBhcmFtIHJlY2VpdmVkXG4gICAqIEBwYXJhbSByZXNvdXJjZVR5cGVcbiAgICogQHBhcmFtIHByb3BlcnRpZXNcbiAgICogQHJldHVybnMge0Fzc2VydGlvblJldHVybn1cbiAgICovXG4gIHJldHVybiBmdW5jdGlvbiB0b0hhdmVEYXRhU291cmNlV2l0aFByb3BlcnRpZXMoXG4gICAgcmVjZWl2ZWQ6IHN0cmluZyxcbiAgICByZXNvdXJjZVR5cGU6IFRlcnJhZm9ybUNvbnN0cnVjdG9yLFxuICAgIHByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fVxuICApOiBBc3NlcnRpb25SZXR1cm4ge1xuICAgIHJldHVybiBnZXRBc3NlcnRFbGVtZW50V2l0aFByb3BlcnRpZXMoY3VzdG9tUGFzc0V2YWx1YXRpb24pKFxuICAgICAgXCJkYXRhXCIsXG4gICAgICByZWNlaXZlZCxcbiAgICAgIHJlc291cmNlVHlwZSxcbiAgICAgIHByb3BlcnRpZXNcbiAgICApO1xuICB9O1xufVxuXG4vKipcbiAqIFJldHVybnMgdGhlIGZ1bmN0aW9uIHRvSGF2ZVJlc291cmNlV2l0aFByb3BlcnRpZXMgdXNpbmcgdGhlIGV2YWx1YXRpb24gcHJvcGVydGllcyBvZiBjdXN0b21QYXNzRXZhbHVhdGlvblxuICogQHBhcmFtIGN1c3RvbVBhc3NFdmFsdWF0aW9uXG4gKiBAcmV0dXJuc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0VG9IYXZlUmVzb3VyY2VXaXRoUHJvcGVydGllcyhcbiAgY3VzdG9tUGFzc0V2YWx1YXRpb24/OiAoXG4gICAgaXRlbXM6IGFueSxcbiAgICBhc3NlcnRlZFByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT5cbiAgKSA9PiBib29sZWFuXG4pIHtcbiAgLyoqXG4gICAqIEV2YWx1YXRlcyB0aGUgcmVjZWl2ZWQgc3RhY2sgdG8gaGF2ZSB0aGUgcmVzb3VyY2UgcmVzb3VyY2VUeXBlIGNvbnRhaW5pbmcgc3BlY2lmaWVkIHByb3BlcnRpZXNcbiAgICogQHBhcmFtIHJlY2VpdmVkXG4gICAqIEBwYXJhbSByZXNvdXJjZVR5cGVcbiAgICogQHBhcmFtIHByb3BlcnRpZXNcbiAgICogQHJldHVybnMge0Fzc2VydGlvblJldHVybn1cbiAgICovXG4gIHJldHVybiBmdW5jdGlvbiB0b0hhdmVSZXNvdXJjZVdpdGhQcm9wZXJ0aWVzKFxuICAgIHJlY2VpdmVkOiBzdHJpbmcsXG4gICAgcmVzb3VyY2VUeXBlOiBUZXJyYWZvcm1Db25zdHJ1Y3RvcixcbiAgICBwcm9wZXJ0aWVzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge31cbiAgKTogQXNzZXJ0aW9uUmV0dXJuIHtcbiAgICByZXR1cm4gZ2V0QXNzZXJ0RWxlbWVudFdpdGhQcm9wZXJ0aWVzKGN1c3RvbVBhc3NFdmFsdWF0aW9uKShcbiAgICAgIFwicmVzb3VyY2VcIixcbiAgICAgIHJlY2VpdmVkLFxuICAgICAgcmVzb3VyY2VUeXBlLFxuICAgICAgcHJvcGVydGllc1xuICAgICk7XG4gIH07XG59XG5cbi8qKlxuICogQSBoZWxwZXIgdXRpbCB0byB2ZXJpZnkgd2V0aGVyIGFuIEVycm9yIHdhcyBjYXVzZWQgYnkgdGhlIE5vZGVqcyBgcHJvY2Vzcy5zcGF3bmAgQVBJLlxuICpcbiAqIEBwYXJhbSAgIHtFcnJvcn0gICBlcnIgVGhlIEVycm9yIG9iamVjdCB0byB2ZXJpZnlcbiAqIEByZXR1cm5zIHtCb29sZWFufSAgICAgQSBib29sIGluZGljYXRpbmcgd2V0aGVyIHRoZSBpbnB1dCBFcnJvciBpcyBjb250YWluaW5nIHByb2Nlc3Muc3Bhd24gb3V0cHV0LlxuICovXG5jb25zdCBpc0V4ZWNTcGF3bkVycm9yID0gKGVycjogYW55KTogZXJyIGlzIEVycm9yICYgU3Bhd25TeW5jUmV0dXJuczxhbnk+ID0+XG4gIFwib3V0cHV0XCIgaW4gZXJyICYmXG4gIEFycmF5LmlzQXJyYXkoZXJyLm91dHB1dCkgJiZcbiAgZXJyLm91dHB1dC5zb21lKChidWY6IGFueSkgPT4gQnVmZmVyLmlzQnVmZmVyKGJ1ZikpO1xuXG4vKipcbiAqIEEgaGVscGVyIHV0aWwgdG8gYXBwZW5kIGBwcm9jZXNzLnNwYXduYCBvdXRwdXQgdG8gYXNzZXJ0aW9uIG1lc3NhZ2VzIHRvIGltcHJvdmUgZGV2ZWxvcGVyIGV4cGlyaWVuY2UuXG4gKlxuICogQHBhcmFtICAge1N0cmluZ30gbWVzc2FnZSBUaGUgbWVzc2FnZSB0byBvcHRpb25hbGx5IGFwcGVuZCBwcm9jZXNzIG91dHB1dCB0by5cbiAqIEBwYXJhbSAgIHtFcnJvcn0gIGVyciAgICAgVGhlIGVycm9yIGZyb20gd2hpY2ggdGhlIGBwcm9jZXNzLnNwYXduYCBvdXRwdXQgc2hvdWxkIGJlIHJldHJlaXZlZCBmcm9tLlxuICogQHJldHVybnMge1N0cmluZ30gICAgICAgICBUaGUgZmluYWxpemVkIGFzc2VydGlvbiBtZXNzYWdlIGRlY29yYXRlZCB3aXRoIHRoZSBgcHJvY2Vzcy5zcGF3bmAgb3V0cHV0LlxuICovXG5jb25zdCB3aXRoUHJvY2Vzc091dHB1dCA9IChtZXNzYWdlOiBzdHJpbmcsIGVycjogdW5rbm93bikgPT4ge1xuICBsZXQgb3V0cHV0ID0gXCJcIjtcblxuICBpZiAoaXNFeGVjU3Bhd25FcnJvcihlcnIpKSB7XG4gICAgb3V0cHV0ID1cbiAgICAgIGVyci5vdXRwdXRcbiAgICAgICAgPy5tYXAoKGJ1ZmZlcjogQnVmZmVyKSA9PiBidWZmZXI/LnRvU3RyaW5nKFwidXRmOFwiKSlcbiAgICAgICAgLmZpbHRlcihCb29sZWFuKVxuICAgICAgICAuam9pbihcIlxcblwiKSA/PyBcIlwiO1xuICB9XG5cbiAgY29uc3QgYXBwZW5kaXggPSBvdXRwdXQubGVuZ3RoID8gYC4gT3V0cHV0OiAke291dHB1dH1gIDogXCJcIjtcblxuICByZXR1cm4gYCR7bWVzc2FnZX06ICR7ZXJyfSR7YXBwZW5kaXh9LmA7XG59O1xuXG4vKipcbiAqIFJldHVybnMgdGhlIGZ1bmN0aW9uIHRvSGF2ZVByb3ZpZGVyV2l0aFByb3BlcnRpZXMgdXNpbmcgdGhlIGV2YWx1YXRpb24gcHJvcGVydGllcyBvZiBjdXN0b21QYXNzRXZhbHVhdGlvblxuICogQHBhcmFtIGN1c3RvbVBhc3NFdmFsdWF0aW9uXG4gKiBAcmV0dXJucyB7Z2V0VG9IYXZlUHJvdmlkZXJXaXRoUHJvcGVydGllc350b0hhdmVQcm92aWRlcldpdGhQcm9wZXJ0aWVzfVxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0VG9IYXZlUHJvdmlkZXJXaXRoUHJvcGVydGllcyhcbiAgY3VzdG9tUGFzc0V2YWx1YXRpb24/OiAoXG4gICAgaXRlbXM6IGFueSxcbiAgICBhc3NlcnRlZFByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT5cbiAgKSA9PiBib29sZWFuXG4pIHtcbiAgLyoqXG4gICAqIEV2YWx1YXRlcyB0aGUgcmVjZWl2ZWQgc3RhY2sgdG8gaGF2ZSB0aGUgcHJvdmlkZXIgcmVzb3VyY2VUeXBlIGNvbnRhaW5pbmcgc3BlY2lmaWVkIHByb3BlcnRpZXNcbiAgICogQHBhcmFtIHJlY2VpdmVkXG4gICAqIEBwYXJhbSByZXNvdXJjZVR5cGVcbiAgICogQHBhcmFtIHByb3BlcnRpZXNcbiAgICogQHJldHVybnMge0Fzc2VydGlvblJldHVybn1cbiAgICovXG4gIHJldHVybiBmdW5jdGlvbiB0b0hhdmVQcm92aWRlcldpdGhQcm9wZXJ0aWVzKFxuICAgIHJlY2VpdmVkOiBzdHJpbmcsXG4gICAgcmVzb3VyY2VUeXBlOiBUZXJyYWZvcm1Db25zdHJ1Y3RvcixcbiAgICBwcm9wZXJ0aWVzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge31cbiAgKTogQXNzZXJ0aW9uUmV0dXJuIHtcbiAgICByZXR1cm4gZ2V0QXNzZXJ0RWxlbWVudFdpdGhQcm9wZXJ0aWVzKGN1c3RvbVBhc3NFdmFsdWF0aW9uKShcbiAgICAgIFwicHJvdmlkZXJcIixcbiAgICAgIHJlY2VpdmVkLFxuICAgICAgcmVzb3VyY2VUeXBlLFxuICAgICAgcHJvcGVydGllc1xuICAgICk7XG4gIH07XG59XG5cbi8qKlxuICogRXZhbHVhdGVzIHRoZSB2YWxpZGl0eSBvZiB0aGUgcmVjZWl2ZWQgc3RhY2tcbiAqIEBwYXJhbSByZWNlaXZlZFxuICogQHJldHVybnMge0Fzc2VydGlvblJldHVybn1cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHRvQmVWYWxpZFRlcnJhZm9ybShyZWNlaXZlZDogc3RyaW5nKTogQXNzZXJ0aW9uUmV0dXJuIHtcbiAgdHJ5IHtcbiAgICBpZiAoIWZzLnN0YXRTeW5jKHJlY2VpdmVkKS5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJQYXRoIGlzIG5vdCBhIGRpcmVjdG9yeVwiKTtcbiAgICB9XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICByZXR1cm4gbmV3IEFzc2VydGlvblJldHVybihcbiAgICAgIGBFeHBlY3RlZCBzdWJqZWN0IHRvIGJlIGEgdGVycmFmb3JtIGRpcmVjdG9yeTogJHtlfWAsXG4gICAgICBmYWxzZVxuICAgICk7XG4gIH1cblxuICB0cnkge1xuICAgIGNvbnN0IG1hbmlmZXN0ID0gSlNPTi5wYXJzZShcbiAgICAgIGZzLnJlYWRGaWxlU3luYyhwYXRoLnJlc29sdmUocmVjZWl2ZWQsIFwibWFuaWZlc3QuanNvblwiKSwgXCJ1dGY4XCIpXG4gICAgKTtcblxuICAgIGNvbnN0IHN0YWNrcyA9IE9iamVjdC5lbnRyaWVzKG1hbmlmZXN0LnN0YWNrcyk7XG5cbiAgICBzdGFja3MuZm9yRWFjaCgoW25hbWUsIHN0YWNrXSkgPT4ge1xuICAgICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgICAgY3dkOiBwYXRoLnJlc29sdmUocmVjZWl2ZWQsIChzdGFjayBhcyBhbnkpLndvcmtpbmdEaXJlY3RvcnkpLFxuICAgICAgICBlbnY6IHByb2Nlc3MuZW52LFxuICAgICAgICBzdGRpbzogXCJwaXBlXCIsXG4gICAgICB9IGFzIGFueTtcbiAgICAgIGV4ZWNTeW5jKGAke3RlcnJhZm9ybUJpbmFyeU5hbWV9IGluaXRgLCBvcHRzKTtcbiAgICAgIGNvbnN0IG91dCA9IGV4ZWNTeW5jKGAke3RlcnJhZm9ybUJpbmFyeU5hbWV9IHZhbGlkYXRlIC1qc29uYCwgb3B0cyk7XG5cbiAgICAgIGNvbnN0IHJlc3VsdCA9IEpTT04ucGFyc2Uob3V0LnRvU3RyaW5nKCkpO1xuICAgICAgaWYgKCFyZXN1bHQudmFsaWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBGb3VuZCAke1xuICAgICAgICAgICAgcmVzdWx0LmVycm9yX2NvdW50XG4gICAgICAgICAgfSBFcnJvcnMgaW4gc3RhY2sgJHtuYW1lfTogJHtyZXN1bHQuZGlhZ25vc3RpY3Muam9pbihcIlxcblwiKX1gXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfSk7XG4gICAgcmV0dXJuIG5ldyBBc3NlcnRpb25SZXR1cm4oXG4gICAgICBgRXhwZWN0ZWQgc3ViamVjdCBub3QgdG8gYmUgYSB2YWxpZCB0ZXJyYWZvcm0gc3RhY2tgLFxuICAgICAgdHJ1ZVxuICAgICk7XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICByZXR1cm4gbmV3IEFzc2VydGlvblJldHVybihcbiAgICAgIHdpdGhQcm9jZXNzT3V0cHV0KGBFeHBlY3RlZCBzdWJqZWN0IHRvIGJlIGEgdmFsaWQgdGVycmFmb3JtIHN0YWNrYCwgZSksXG4gICAgICBmYWxzZVxuICAgICk7XG4gIH1cbn1cblxuLyoqXG4gKiBFdmFsdWF0ZXMgdGhlIGFiaWxpdHkgZm9yIHRoZSByZWNlaXZlZCBzdGFjayB0byBzdWNjZXNzZnVsbHkgcGxhblxuICogQHBhcmFtIHJlY2VpdmVkXG4gKiBAcmV0dXJucyB7QXNzZXJ0aW9uUmV0dXJufVxuICovXG5leHBvcnQgZnVuY3Rpb24gdG9QbGFuU3VjY2Vzc2Z1bGx5KHJlY2VpdmVkOiBzdHJpbmcpOiBBc3NlcnRpb25SZXR1cm4ge1xuICB0cnkge1xuICAgIGlmICghZnMuc3RhdFN5bmMocmVjZWl2ZWQpLmlzRGlyZWN0b3J5KCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIlBhdGggaXMgbm90IGEgZGlyZWN0b3J5XCIpO1xuICAgIH1cbiAgfSBjYXRjaCAoZSkge1xuICAgIHJldHVybiBuZXcgQXNzZXJ0aW9uUmV0dXJuKFxuICAgICAgYEV4cGVjdGVkIHN1YmplY3QgdG8gYmUgYSB0ZXJyYWZvcm0gZGlyZWN0b3J5OiAke2V9YCxcbiAgICAgIGZhbHNlXG4gICAgKTtcbiAgfVxuXG4gIHRyeSB7XG4gICAgY29uc3QgbWFuaWZlc3QgPSBKU09OLnBhcnNlKFxuICAgICAgZnMucmVhZEZpbGVTeW5jKHBhdGgucmVzb2x2ZShyZWNlaXZlZCwgXCJtYW5pZmVzdC5qc29uXCIpLCBcInV0ZjhcIilcbiAgICApO1xuXG4gICAgY29uc3Qgc3RhY2tzID0gT2JqZWN0LmVudHJpZXMobWFuaWZlc3Quc3RhY2tzKTtcblxuICAgIHN0YWNrcy5mb3JFYWNoKChbLCBzdGFja10pID0+IHtcbiAgICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICAgIGN3ZDogcGF0aC5yZXNvbHZlKHJlY2VpdmVkLCAoc3RhY2sgYXMgYW55KS53b3JraW5nRGlyZWN0b3J5KSxcbiAgICAgICAgZW52OiBwcm9jZXNzLmVudixcbiAgICAgICAgc3RkaW86IFwiaWdub3JlXCIsXG4gICAgICB9IGFzIGFueTtcbiAgICAgIGV4ZWNTeW5jKGAke3RlcnJhZm9ybUJpbmFyeU5hbWV9IGluaXRgLCBvcHRzKTtcblxuICAgICAgLy8gVGhyb3dzIG9uIGEgbm9uLXplcm8gZXhpdCBjb2RlXG4gICAgICBleGVjU3luYyhgJHt0ZXJyYWZvcm1CaW5hcnlOYW1lfSBwbGFuIC1pbnB1dD1mYWxzZSAtbG9jaz1mYWxzZSBgLCBvcHRzKTtcbiAgICB9KTtcblxuICAgIHJldHVybiBuZXcgQXNzZXJ0aW9uUmV0dXJuKFxuICAgICAgYEV4cGVjdGVkIHN1YmplY3Qgbm90IHRvIHBsYW4gc3VjY2Vzc2Z1bGx5YCxcbiAgICAgIHRydWVcbiAgICApO1xuICB9IGNhdGNoIChlKSB7XG4gICAgcmV0dXJuIG5ldyBBc3NlcnRpb25SZXR1cm4oXG4gICAgICB3aXRoUHJvY2Vzc091dHB1dChgRXhwZWN0ZWQgc3ViamVjdCB0byBwbGFuIHN1Y2Nlc3NmdWxseWAsIGUpLFxuICAgICAgZmFsc2VcbiAgICApO1xuICB9XG59XG4iXX0=