"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.dependable = exports.forExpression = exports.FOR_EXPRESSION_VALUE = exports.FOR_EXPRESSION_KEY = exports.call = exports.OperatorExpression = exports.conditional = exports.propertyAccess = exports.insideTfExpression = exports.ref = exports.rawString = void 0;
const intrinsic_1 = require("./tokens/private/intrinsic");
const token_1 = require("./tokens/token");
const app_1 = require("./app");
const terraform_stack_1 = require("./terraform-stack");
const constructs_1 = require("constructs");
const TERRAFORM_IDENTIFIER_REGEX = /^[_a-zA-Z][_a-zA-Z0-9]*$/;
// eslint-disable-next-line jsdoc/require-jsdoc
class TFExpression extends intrinsic_1.Intrinsic {
    resolveArg(context, arg) {
        const resolvedArg = context.resolve(arg);
        if (token_1.Tokenization.isResolvable(arg)) {
            return resolvedArg;
        }
        if (typeof arg === "string") {
            return this.resolveString(arg, resolvedArg);
        }
        if (Array.isArray(resolvedArg)) {
            return `[${resolvedArg
                .map((_, index) => this.resolveArg(context, arg[index]))
                .join(", ")}]`;
        }
        if (typeof resolvedArg === "object" && resolvedArg !== null) {
            return `{${Object.keys(resolvedArg)
                .map((key) => `"${key}" = ${this.resolveArg(context, arg[key])}`)
                .join(", ")}}`;
        }
        return resolvedArg;
    }
    /**
     * Escape string removes characters from the string that are not allowed in Terraform or JSON
     * It must only be used on non-token values
     */
    escapeString(str) {
        return str
            .replace(/\n/g, "\\n") // escape newlines
            .replace(/\${/g, "$$${"); // escape ${ to $${
    }
    resolveString(str, resolvedArg) {
        const tokenList = token_1.Tokenization.reverseString(str);
        const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
        // String literal
        if (numberOfTokens === 0) {
            return resolvedArg !== `"` &&
                resolvedArg.startsWith('"') &&
                resolvedArg.endsWith('"')
                ? this.escapeString(resolvedArg)
                : `"${this.escapeString(resolvedArg)}"`;
        }
        // Only a token reference
        if (tokenList.literals.length === 0 &&
            tokenList.escapes.length === 0 &&
            numberOfTokens === 1) {
            return resolvedArg;
        }
        // String literal + token reference combination
        const joinResult = tokenList.join({
            join: (left, right) => {
                const leftTokens = token_1.Tokenization.reverse(left);
                const leftTokenList = token_1.Tokenization.reverseString(left);
                const rightTokens = token_1.Tokenization.reverse(right);
                const leftTokenCount = leftTokenList.intrinsic.length + leftTokenList.tokens.length;
                // if left is mixed, needs to be left alone (because it's a result of a previous join iteration)
                let leftValue = left;
                // if left is a string literal, then we need to escape it
                if (leftTokenList.literals.length === 1 && leftTokenCount === 0) {
                    leftValue = this.escapeString(left);
                }
                // if left is only a token, needs to be wrapped as terraform expression
                if (leftTokenList.literals.length === 0 && leftTokenCount === 1) {
                    leftValue = `\${${leftTokens[0]}}`;
                }
                const rightValue = rightTokens.length === 0
                    ? this.escapeString(right)
                    : `\${${rightTokens[0]}}`;
                return `${leftValue}${rightValue}`;
            },
        });
        return joinResult !== `"` &&
            joinResult.startsWith('"') &&
            joinResult.endsWith('"')
            ? joinResult
            : `"${joinResult}"`;
    }
}
// A string that represents an input value to be escaped
// eslint-disable-next-line jsdoc/require-jsdoc
class RawString extends TFExpression {
    constructor(str) {
        super(str);
        this.str = str;
    }
    resolve(context) {
        const qts = context.suppressBraces ? `"` : ``;
        return `${qts}${this.escapeString(this.str).replace(/\"/g, '\\"')}${qts}`; // eslint-disable-line no-useless-escape
    }
    toString() {
        return this.str;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function rawString(str) {
    return new RawString(str);
}
exports.rawString = rawString;
// eslint-disable-next-line jsdoc/require-jsdoc
class Reference extends TFExpression {
    constructor(identifier, originStack) {
        super(identifier);
        this.identifier = identifier;
        this.originStack = originStack;
        /**
         * A single reference could be used in multiple stacks,
         * e.g. if we expose the ref directly or as token on the stack.
         * We need to store the identifier for each stack,
         * so that the resolved identifier string matches the stack it's resolved in.
         */
        this.crossStackIdentifier = {};
    }
    resolve(context) {
        // We check for cross stack references on preparation, setting a new identifier
        const resolutionStack = terraform_stack_1.TerraformStack.of(context.scope);
        const stackName = resolutionStack.toString();
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        if (context.preparing) {
            // Cross stack reference
            if (this.originStack && this.originStack !== resolutionStack) {
                const app = app_1.App.of(this.originStack);
                const csr = app.crossStackReference(this.originStack, resolutionStack, this.identifier);
                this.crossStackIdentifier[stackName] = csr;
            }
        }
        // If this is a cross stack reference we will resolve to a reference within this stack.
        if (this.crossStackIdentifier[stackName]) {
            return suppressBraces
                ? this.crossStackIdentifier[stackName]
                : `\${${this.crossStackIdentifier[stackName]}}`;
        }
        return suppressBraces ? this.identifier : `\${${this.identifier}}`;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function ref(identifier, stack) {
    return new Reference(identifier, stack);
}
exports.ref = ref;
/**
 * marks the argument as being used in a terraform expression
 * @deprecated
 */
function insideTfExpression(arg) {
    return arg;
}
exports.insideTfExpression = insideTfExpression;
// eslint-disable-next-line jsdoc/require-jsdoc
class PropertyAccess extends TFExpression {
    constructor(target, args) {
        super({ target, args });
        this.target = target;
        this.args = args;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const serializedArgs = this.args
            .map((arg) => {
            if (arg === `*`) {
                return "[*]";
            }
            const a = this.resolveArg(context, arg);
            const isPlainString = typeof arg === "string" && TERRAFORM_IDENTIFIER_REGEX.test(arg);
            const isPlainNumber = typeof arg === "number" && !token_1.Token.isUnresolved(arg);
            if (isPlainString || isPlainNumber) {
                return `.${arg}`;
            }
            return `[${a}]`;
        }) // property access
            .join("");
        const targetExpr = constructs_1.Construct.isConstruct(this.target) && "fqn" in this.target
            ? this.target.fqn
            : this.resolveArg(context, this.target);
        const expr = `${targetExpr}${serializedArgs}`;
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function propertyAccess(target, args) {
    return new PropertyAccess(target, args);
}
exports.propertyAccess = propertyAccess;
// eslint-disable-next-line jsdoc/require-jsdoc
class ConditionalExpression extends TFExpression {
    constructor(condition, trueValue, falseValue) {
        super({ condition, trueValue, falseValue });
        this.condition = condition;
        this.trueValue = trueValue;
        this.falseValue = falseValue;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const condition = this.resolveArg(context, this.condition);
        const trueValue = this.resolveArg(context, this.trueValue);
        const falseValue = this.resolveArg(context, this.falseValue);
        const expr = `${condition} ? ${trueValue} : ${falseValue}`;
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
/**
 * creates an expression like `true ? 1 : 0`
 */
function conditional(condition, trueValue, falseValue) {
    return new ConditionalExpression(condition, trueValue, falseValue);
}
exports.conditional = conditional;
/**
 * Base class for Operator Expressions
 * @internal
 */
class OperatorExpression extends TFExpression {
    constructor(operator, left, right // optional for ! and - operator
    ) {
        super({ operator, leftValue: left, rightValue: right });
        this.operator = operator;
        this.left = left;
        this.right = right;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const left = this.resolveArg(context, this.left);
        const right = this.right ? this.resolveArg(context, this.right) : undefined;
        let expr = "";
        switch (this.operator) {
            case "!": {
                expr = `!${left}`; // no braces needed as ! has highest precedence
                break;
            }
            case "-": {
                if (right) {
                    // subtraction
                    expr = `(${left} - ${right})`;
                }
                else {
                    // negation
                    expr = `-${left}`; // no braces needed as - has highest precedence
                }
                break;
            }
            default: {
                expr = `(${left} ${this.operator} ${right})`;
            }
        }
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
exports.OperatorExpression = OperatorExpression;
// eslint-disable-next-line jsdoc/require-jsdoc
class FunctionCall extends TFExpression {
    constructor(name, args) {
        super({ name, args });
        this.name = name;
        this.args = args;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        const originalIgnoreEscapes = context.ignoreEscapes;
        const originalWarnEscapes = context.warnEscapes;
        context.suppressBraces = true;
        context.ignoreEscapes = true;
        context.warnEscapes = true;
        const serializedArgs = this.args
            .map((arg) => this.resolveArg(context, arg))
            .join(", ");
        const expr = `${this.name}(${serializedArgs})`;
        context.ignoreEscapes = originalIgnoreEscapes;
        context.warnEscapes = originalWarnEscapes;
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function call(name, args) {
    return new FunctionCall(name, args);
}
exports.call = call;
exports.FOR_EXPRESSION_KEY = ref("key");
exports.FOR_EXPRESSION_VALUE = ref("val");
/**
 * https://developer.hashicorp.com/terraform/language/expressions/for
 */
class ForExpression extends TFExpression {
    constructor(input, valueExpression, keyExpression) {
        super({ input, valueExpression, keyExpression });
        this.input = input;
        this.valueExpression = valueExpression;
        this.keyExpression = keyExpression;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const key = this.resolveArg(context, exports.FOR_EXPRESSION_KEY);
        const value = this.resolveArg(context, exports.FOR_EXPRESSION_VALUE);
        const input = this.resolveArg(context, this.input);
        const valueExpr = this.resolveArg(context, this.valueExpression);
        let expr;
        if (this.keyExpression) {
            const keyExpr = this.resolveArg(context, this.keyExpression);
            expr = `{ for ${key}, ${value} in ${input}: ${keyExpr} => ${valueExpr} }`;
        }
        else {
            expr = `[ for ${key}, ${value} in ${input}: ${valueExpr}]`;
        }
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
/**
 * Creates a for expression, used for advanced internal use cases (e.g. in the @cdktf/aws-adapter)
 * It is currently not possible to use this via JSII in other languages. Usually you might want to
 * use TerraformIterator instead.
 *
 * @param input the input for the expression e.g. a list or an object
 * @param valueExpression will express the value e.g. [ for i in &lt;input&gt; : &lt;valueExpression&gt; ]
 * @param keyExpression if set will return an object e.g. { for k,v in &lt;input&gt; : &lt;keyExpression&gt; => &lt;valueExpression&gt;}
 * @returns a for expression
 *
 * The variables for key (for maps) / index (for lists) and value that are valid in the key- and valueExpression
 * can be referenced by using these constants: FOR_EXPRESSION_KEY and FOR_EXPRESSION_VALUE.
 *
 * Currently nesting for expressions is not supported due to this simplification.
 *
 * Currently does not support filtering elements (if clause) or grouping results
 */
function forExpression(input, valueExpression, keyExpression) {
    return new ForExpression(input, valueExpression, keyExpression);
}
exports.forExpression = forExpression;
// eslint-disable-next-line jsdoc/require-jsdoc
class Dependable extends TFExpression {
    constructor(dependable) {
        super(dependable);
        this.dependable = dependable;
    }
    resolve(context) {
        context.suppressBraces = true;
        return this.dependable.fqn;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function dependable(dependable) {
    return token_1.Token.asString(new Dependable(dependable));
}
exports.dependable = dependable;
//# sourceMappingURL=data:application/json;base64,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