"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractTokenDouble = exports.createTokenDouble = exports.unresolved = exports.containsComplexElement = exports.isComplexElement = exports.containsMapToken = exports.containsNumberListTokenElement = exports.containsStringListTokenElement = exports.NullConcat = exports.regexQuote = exports.TokenString = exports.VALID_KEY_CHARS = exports.END_TOKEN_MARKER = exports.BEGIN_MAP_TOKEN_MARKER = exports.BEGIN_LIST_TOKEN_MARKER = exports.BEGIN_STRING_TOKEN_MARKER = void 0;
const string_fragments_1 = require("../string-fragments");
const token_1 = require("../token");
// Details for encoding and decoding Tokens into native types; should not be exported
exports.BEGIN_STRING_TOKEN_MARKER = "${TfToken[";
exports.BEGIN_LIST_TOKEN_MARKER = "#{TfToken[";
exports.BEGIN_MAP_TOKEN_MARKER = "&{TfToken[";
exports.END_TOKEN_MARKER = "]}";
exports.VALID_KEY_CHARS = "a-zA-Z0-9:._-";
const QUOTED_BEGIN_STRING_TOKEN_MARKER = regexQuote(exports.BEGIN_STRING_TOKEN_MARKER);
const QUOTED_BEGIN_LIST_TOKEN_MARKER = regexQuote(exports.BEGIN_LIST_TOKEN_MARKER);
const QUOTED_BEGIN_MAP_TOKEN_MARKER = regexQuote(exports.BEGIN_MAP_TOKEN_MARKER);
const QUOTED_END_TOKEN_MARKER = regexQuote(exports.END_TOKEN_MARKER);
const STRING_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_STRING_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const LIST_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_LIST_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const MAP_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_MAP_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const NUMBER_TOKEN_REGEX = new RegExp("[-|\\d|.\\+]+(e[-|\\d|.|e|E|\\+]+)", "g");
const ESCAPE_TOKEN_BEGIN_REGEX = /\$\{(?!TfToken\[)/g;
const ESCAPE_TOKEN_END_REGEX = /\}/g;
/**
 * A string with markers in it that can be resolved to external values
 */
class TokenString {
    /**
     * Returns a `TokenString` for this string.
     *
     * @param s The string to tokenize
     * @param includeEscapeSequences Whether to include escape sequences in the tokenization
     * @param warnIfEscapeSequences Whether to warn if escape sequences are found
     */
    static forString(s, includeEscapeSequences = false, warnIfEscapeSequences = false) {
        return new TokenString(s, STRING_TOKEN_REGEX, 1, includeEscapeSequences, warnIfEscapeSequences);
    }
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s) {
        return new TokenString(s, LIST_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded numbers
     */
    static forNumbers(s) {
        return new TokenString(s, NUMBER_TOKEN_REGEX, 0);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded maps
     */
    static forMapToken(s) {
        return new TokenString(s, MAP_TOKEN_REGEX);
    }
    constructor(str, re, regexMatchIndex = 1, includeEscapeSequences = false, warnIfEscapeSequences = false) {
        this.str = str;
        this.re = re;
        this.regexMatchIndex = regexMatchIndex;
        this.includeEscapeSequences = includeEscapeSequences;
        this.warnIfEscapeSequences = warnIfEscapeSequences;
    }
    testForEscapeTokens(startIndex, maxIndex) {
        ESCAPE_TOKEN_BEGIN_REGEX.lastIndex = startIndex;
        let startMatch = ESCAPE_TOKEN_BEGIN_REGEX.exec(this.str);
        if (startMatch && startMatch.index >= maxIndex) {
            startMatch = null;
        }
        return !!startMatch;
    }
    nextEscapeToken(fragments, startIndex, escapeDepth, maxIndex) {
        ESCAPE_TOKEN_BEGIN_REGEX.lastIndex = startIndex;
        ESCAPE_TOKEN_END_REGEX.lastIndex = startIndex;
        let startMatch = ESCAPE_TOKEN_BEGIN_REGEX.exec(this.str);
        let endMatch = ESCAPE_TOKEN_END_REGEX.exec(this.str);
        if (startMatch && startMatch.index >= maxIndex) {
            startMatch = null;
        }
        if (endMatch && endMatch.index >= maxIndex) {
            endMatch = null;
        }
        if (!startMatch && !endMatch) {
            return {
                index: -1,
                escapeDepth: escapeDepth,
            };
        }
        if (startMatch && endMatch) {
            if (startMatch.index > startIndex && startMatch.index > startIndex) {
                const lede = this.str.substring(startIndex, Math.min(startMatch.index, endMatch.index));
                fragments.addLiteral(lede);
            }
            if (startMatch.index < endMatch.index) {
                fragments.addEscape("open");
                return {
                    index: ESCAPE_TOKEN_BEGIN_REGEX.lastIndex,
                    escapeDepth: escapeDepth + 1,
                };
            }
            fragments.addEscape("close");
            return {
                index: ESCAPE_TOKEN_END_REGEX.lastIndex,
                escapeDepth: escapeDepth - 1,
            };
        }
        if (startMatch) {
            if (startMatch.index > startIndex) {
                const lede = this.str.substring(startIndex, startMatch.index);
                fragments.addLiteral(lede);
            }
            fragments.addEscape("open");
            return {
                index: ESCAPE_TOKEN_BEGIN_REGEX.lastIndex,
                escapeDepth: escapeDepth + 1,
            };
        }
        if (endMatch) {
            if (endMatch.index > startIndex) {
                const lede = this.str.substring(startIndex, endMatch.index);
                fragments.addLiteral(lede);
            }
            fragments.addEscape("close");
            return {
                index: ESCAPE_TOKEN_END_REGEX.lastIndex,
                escapeDepth: escapeDepth - 1,
            };
        }
        return {
            index: -1,
            escapeDepth: escapeDepth,
        };
    }
    tokenizeNext(lookup, fragments, startIndex, escapeDepth) {
        this.re.lastIndex = startIndex;
        if (startIndex >= this.str.length) {
            return {
                index: -1,
                escapeDepth,
            };
        }
        const match = this.re.exec(this.str);
        if (!match) {
            if (this.includeEscapeSequences) {
                const next = this.nextEscapeToken(fragments, startIndex, escapeDepth, this.str.length);
                if (next.index === -1) {
                    fragments.addLiteral(this.str.substring(startIndex));
                    return {
                        index: -1,
                        escapeDepth,
                    };
                }
                else {
                    return next;
                }
            }
            fragments.addLiteral(this.str.substring(startIndex));
            return {
                index: -1,
                escapeDepth: escapeDepth,
            };
        }
        if (match.index > startIndex) {
            if (this.includeEscapeSequences) {
                const next = this.nextEscapeToken(fragments, startIndex, escapeDepth, match.index);
                if (next.index === -1) {
                    fragments.addLiteral(this.str.substring(startIndex, match.index));
                    return {
                        index: match.index,
                        escapeDepth: escapeDepth,
                    };
                }
                else {
                    return next;
                }
            }
            const lede = this.str.substring(startIndex, match.index);
            fragments.addLiteral(lede);
            return {
                index: match.index,
                escapeDepth,
            };
        }
        const token = lookup(match[this.regexMatchIndex]);
        if (token) {
            fragments.addToken(token);
        }
        else {
            fragments.addLiteral(this.str.substring(match.index, this.re.lastIndex));
        }
        const nextIndex = this.re.lastIndex;
        return {
            index: nextIndex,
            escapeDepth,
        };
    }
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup) {
        const ret = new string_fragments_1.TokenizedStringFragments();
        let index = 0;
        let escapeDepth = 0;
        if (this.warnIfEscapeSequences &&
            this.testForEscapeTokens(0, this.str.length)) {
            // Only print warning and act as if escape sequences are ignored
            console.warn("You're using escape sequences (${...}) with CDKTF Built-in functions. This is not supported yet, and the output may be incorrect.");
            console.warn(this.str);
        }
        while (index >= 0) {
            const iter = this.tokenizeNext(lookup, ret, index, escapeDepth);
            index = iter.index;
            escapeDepth = iter.escapeDepth;
        }
        return ret;
    }
    /**
     * Indicates if this string includes tokens.
     */
    test() {
        this.re.lastIndex = 0; // Reset
        return this.re.test(this.str);
    }
}
exports.TokenString = TokenString;
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, "\\$&");
}
exports.regexQuote = regexQuote;
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
class NullConcat {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    join(_left, _right) {
        return undefined;
    }
}
exports.NullConcat = NullConcat;
/**
 * Returns true if argument contains a string list token
 */
function containsStringListTokenElement(xs) {
    return xs.some((x) => typeof x === "string" && TokenString.forListToken(x).test());
}
exports.containsStringListTokenElement = containsStringListTokenElement;
/**
 * Returns true if argument contains a number list token
 */
function containsNumberListTokenElement(xs) {
    return xs.some((x) => typeof x === "number" && extractTokenDouble(x, true) !== undefined);
}
exports.containsNumberListTokenElement = containsNumberListTokenElement;
/**
 * Returns true if argument contains a string map token
 */
function containsMapToken(xs) {
    return Object.keys(xs).some((x) => typeof x === "string" && TokenString.forMapToken(x).test());
}
exports.containsMapToken = containsMapToken;
/**
 * Returns true if argument is a complex element
 */
function isComplexElement(xs) {
    return (typeof xs === "object" &&
        xs !== null &&
        "withinArray" in xs &&
        "interpolationAsList" in xs);
}
exports.isComplexElement = isComplexElement;
/**
 * Returns true if list contains a complex element
 */
function containsComplexElement(xs) {
    return xs.length > 0 && isComplexElement(xs[0]);
}
exports.containsComplexElement = containsComplexElement;
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
function unresolved(obj) {
    if (typeof obj === "string") {
        return TokenString.forString(obj).test();
    }
    else if (typeof obj === "number") {
        return extractTokenDouble(obj, false) !== undefined;
    }
    else if (Array.isArray(obj) && obj.length === 1) {
        return ((typeof obj[0] === "string" && TokenString.forListToken(obj[0]).test()) ||
            (typeof obj[0] === "number" &&
                extractTokenDouble(obj[0], true) !== undefined));
    }
    else if (obj != null && containsMapToken(obj)) {
        return true;
    }
    else {
        return token_1.Tokenization.isResolvable(obj);
    }
}
exports.unresolved = unresolved;
/**
 * Bit pattern in the top 16 bits of a double to indicate a Token
 *
 * An IEEE double in LE memory order looks like this (grouped
 * into octets, then grouped into 32-bit words):
 *
 * mmmmmmmm.mmmmmmmm.mmmmmmmm.mmmmmmmm | mmmmmmmm.mmmmmmmm.EEEEmmmm.sEEEEEEE
 *
 * - m: mantissa (52 bits)
 * - E: exponent (11 bits)
 * - s: sign (1 bit)
 *
 * We put the following marker into the top 16 bits (exponent and sign), and
 * use the mantissa part to encode the token index. To save some bit twiddling
 * we use all top 16 bits for the tag. That loses us 4 mantissa bits to store
 * information in but we still have 48, which is going to be plenty for any
 * number of tokens to be created during the lifetime of any CDK application.
 *
 * Can't have all bits set because that makes a NaN, so unset the least
 * significant exponent bit.
 *
 * Currently not supporting BE architectures.
 */
// tslint:disable-next-line:no-bitwise
const DOUBLE_TOKEN_MARKER_BITS = 0xfdff << 16;
// tslint:disable-next-line:no-bitwise
const DOUBLE_LIST_TOKEN_MARKER_BITS = 0xfbff << 16;
/**
 * Highest encodable number
 */
const MAX_ENCODABLE_INTEGER = Math.pow(2, 48) - 1;
/**
 * Get 2^32 as a number, so we can do multiplication and div instead of bit shifting
 *
 * Necessary because in JavaScript, bit operations implicitly convert
 * to int32 and we need them to work on "int64"s.
 *
 * So instead of x >> 32, we do Math.floor(x / 2^32), and vice versa.
 */
const BITS32 = Math.pow(2, 32);
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
function createTokenDouble(x, list) {
    if (Math.floor(x) !== x || x < 0) {
        throw new Error("Can only encode positive integers");
    }
    if (x > MAX_ENCODABLE_INTEGER) {
        throw new Error(`Got an index too large to encode: ${x}`);
    }
    const buf = new ArrayBuffer(8);
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    ints[0] = x & 0x0000ffffffff; // Bottom 32 bits of number
    // This needs an "x >> 32" but that will make it a 32-bit number instead
    // of a 64-bit number.
    ints[1] =
        (shr32(x) & 0xffff) |
            (list ? DOUBLE_LIST_TOKEN_MARKER_BITS : DOUBLE_TOKEN_MARKER_BITS); // Top 16 bits of number and the mask
    // tslint:enable:no-bitwise
    return new Float64Array(buf)[0];
}
exports.createTokenDouble = createTokenDouble;
/**
 * Shift a 64-bit int right 32 bits
 */
function shr32(x) {
    return Math.floor(x / BITS32);
}
/**
 * Shift a 64-bit left 32 bits
 */
function shl32(x) {
    return x * BITS32;
}
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
function extractTokenDouble(encoded, list) {
    const buf = new ArrayBuffer(8);
    new Float64Array(buf)[0] = encoded;
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    if ((ints[1] & 0xffff0000) !==
        (list ? DOUBLE_LIST_TOKEN_MARKER_BITS : DOUBLE_TOKEN_MARKER_BITS)) {
        return undefined;
    }
    // Must use + instead of | here (bitwise operations
    // will force 32-bits integer arithmetic, + will not).
    return ints[0] + shl32(ints[1] & 0xffff);
    // tslint:enable:no-bitwise
}
exports.extractTokenDouble = extractTokenDouble;
//# sourceMappingURL=data:application/json;base64,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