"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RememberingTokenResolver = exports.findTokens = exports.resolve = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
const token_1 = require("../token");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = "/" + prefix.join("/");
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            suppressBraces: options.previousContext?.suppressBraces,
            ignoreEscapes: options.previousContext?.ignoreEscapes,
            warnEscapes: options.previousContext?.warnEscapes,
            iteratorContext: options.previousContext?.iteratorContext,
            registerPostProcessor(pp) {
                postProcessor = pp;
            },
            resolve(x) {
                return resolve(x, {
                    ...options,
                    prefix: newPrefix,
                    previousContext: context,
                });
            },
        };
        return [
            context,
            {
                postProcess(x) {
                    return postProcessor ? postProcessor.postProcess(x, context) : x;
                },
            },
        ];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error("Unable to resolve object tree with circular reference. Path: " + pathName);
    }
    //
    // undefined
    //
    if (typeof obj === "undefined") {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof obj === "function") {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof obj === "string") {
        // If this is a "list element" Token, it should never occur by itself in string context
        if (encoding_1.TokenString.forListToken(obj).test()) {
            throw new Error("Found an encoded list token string in a scalar string context. Use 'Fn.element(list, 0)' (not 'list[0]') to extract elements from token lists");
        }
        if (obj === token_1.Token.STRING_MAP_TOKEN_VALUE ||
            obj === token_1.Token.ANY_MAP_TOKEN_VALUE) {
            throw new Error("Found an encoded map token in a scalar string context. Use 'Fn.lookup(map, key, default)' (not 'map[key]') to extract values from token maps.");
        }
        let str = obj;
        const context = makeContext()[0];
        const tokenStr = encoding_1.TokenString.forString(str, !context.ignoreEscapes, context.warnEscapes);
        if (tokenStr.test()) {
            const fragments = tokenStr.split(tokenMap.lookupToken.bind(tokenMap));
            str = options.resolver.resolveString(fragments, makeContext()[0]);
        }
        // replace concatenated token numbers
        const tokenNumberStr = encoding_1.TokenString.forNumbers(str);
        if (tokenNumberStr.test()) {
            const fragments = tokenNumberStr.split((id) => {
                return token_map_1.TokenMap.instance().lookupNumberToken(parseFloat(id));
            });
            const context = makeContext()[0];
            str = fragments.mapTokens(context).join(new resolvable_1.StringConcat());
        }
        return str;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof obj === "number") {
        if (obj === token_1.Token.NUMBER_MAP_TOKEN_VALUE) {
            throw new Error("Found an encoded map token in a scalar number context. Use 'Fn.lookup(map, key, default)' (not 'map[key]') to extract values from token maps.");
        }
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    // eslint-disable-next-line no-instanceof/no-instanceof
    if (typeof obj !== "object" || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if ((0, encoding_1.containsStringListTokenElement)(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        if ((0, encoding_1.containsNumberListTokenElement)(obj)) {
            return options.resolver.resolveNumberList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter((x) => typeof x !== "undefined");
        return arr;
    }
    // check for tokenized map
    if ((0, encoding_1.containsMapToken)(obj)) {
        return options.resolver.resolveMap(obj, makeContext()[0]);
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if ((0, encoding_1.unresolved)(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error("Trying to resolve() a Construct at " + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof resolvedKey !== "string") {
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof value === "undefined") {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
/**
 * Resolves a number token
 */
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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