"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// copied from https://github.com/aws/constructs/blob/e01e47f78ef1e9b600efcd23ff7705aa8d384017/lib/private/uniqueid.ts
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error("Unable to calculate a unique id for an empty set of components");
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter((c) => (0, encoding_1.unresolved)(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(",")}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeNonAlphanumeric(components[0]);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join("")
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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