"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenizedStringFragments = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 * @experimental
 */
class TokenizedStringFragments {
    constructor() {
        /**
         * Fragments in the tokenized string
         */
        this.fragments = new Array();
    }
    /**
     * Returns the first token.
     */
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === "token") {
            return first.token;
        }
        return undefined;
    }
    /**
     * Returns the first value.
     */
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    /**
     * Returns the number of fragments.
     */
    get length() {
        return this.fragments.length;
    }
    /**
     * Adds a literal fragment
     * @param lit the literal to add
     */
    addLiteral(lit) {
        this.fragments.push({ type: "literal", lit });
    }
    /**
     * Adds a token fragment
     * @param token the token to add
     */
    addToken(token) {
        this.fragments.push({ type: "token", token });
    }
    /**
     * Adds an intrinsic fragment
     * @param value the intrinsic value to add
     */
    addIntrinsic(value) {
        this.fragments.push({ type: "intrinsic", value });
    }
    addEscape(kind) {
        this.fragments.push({ type: "escape", kind });
    }
    concat(other) {
        this.fragments.concat(other.fragments);
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "token") {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Return all literals from this string
     */
    get literals() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "literal") {
                ret.push(f.lit);
            }
        }
        return ret;
    }
    /**
     * Return all intrinsic fragments from this string
     */
    get intrinsic() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "intrinsic") {
                ret.push(f.value);
            }
        }
        return ret;
    }
    /**
     * Return all escape fragments from this string
     */
    get escapes() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "escape") {
                if (f.kind === "open")
                    ret.push("${");
                else
                    ret.push("}");
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(context) {
        const ret = new TokenizedStringFragments();
        const originalSupressBraces = context.suppressBraces;
        for (const f of this.fragments) {
            switch (f.type) {
                case "literal":
                    ret.addLiteral(f.lit);
                    break;
                case "escape":
                    if (context.ignoreEscapes) {
                        ret.addLiteral(f.kind === "open" ? "${" : "}");
                        break;
                    }
                    ret.addEscape(f.kind);
                    if (f.kind === "open") {
                        context.suppressBraces = true;
                    }
                    else {
                        context.suppressBraces = originalSupressBraces;
                    }
                    break;
                case "token":
                    // eslint-disable-next-line no-case-declarations
                    const mapped = context.resolve(f.token);
                    if (token_1.Tokenization.isResolvable(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case "intrinsic":
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
_a = JSII_RTTI_SYMBOL_1;
TokenizedStringFragments[_a] = { fqn: "cdktf.TokenizedStringFragments", version: "0.17.0-pre.18" };
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case "literal":
            return fragment.lit;
        case "escape":
            return fragment.kind === "open" ? "${" : "}";
        case "token":
            return fragment.token.toString();
        case "intrinsic":
            return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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