"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tokenization = exports.Token = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return (0, encoding_1.unresolved)(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === "string") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === "number") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every((x) => typeof x === "string")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible list representation of this token
     */
    static asNumberList(value) {
        if (Array.isArray(value) && value.every((x) => typeof x === "number")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumberList(Token.asAny(value));
    }
    /**
     * Return a reversible map representation of this token
     */
    static asMap(value, mapValue, options = {}) {
        // since the return value is basically an object, just encode always
        return token_map_1.TokenMap.instance().registerMap(Token.asAny(value), mapValue, options.displayHint);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asStringMap(value, options = {}) {
        return this.asMap(value, Token.STRING_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asNumberMap(value, options = {}) {
        return this.asMap(value, Token.NUMBER_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asBooleanMap(value, options = {}) {
        return this.asMap(value, true, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asAnyMap(value, options = {}) {
        return this.asMap(value, Token.ANY_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return Tokenization.isResolvable(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    /**
     * Return a Token containing a `null` value
     *
     * Note: This is different than `undefined`, `nil`, `None` or similar
     * as it will end up in the Terraform config and can be used to explicitly
     * not set an attribute (which is sometimes required by Terraform providers)
     *
     * @returns a Token resolving to `null` as understood by Terraform
     */
    static nullValue() {
        // passing null works in Typescript, this is why we can do this here
        // but users wouldn't be able to call e.g. Token.as_any(None) in Python
        return Token.asAny(null);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Token[_a] = { fqn: "cdktf.Token", version: "0.17.0-pre.18" };
/**
 * String Map token value representation
 */
Token.STRING_MAP_TOKEN_VALUE = "String Map Token Value";
/**
 * Number Map token value representation
 */
Token.NUMBER_MAP_TOKEN_VALUE = -123456789;
/**
 * Any map token representation
 */
Token.ANY_MAP_TOKEN_VALUE = "Any Map Token Value";
exports.Token = Token;
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Reverse any value into Resolvables, if possible
     */
    static reverse(x) {
        if (Tokenization.isResolvable(x)) {
            return [x];
        }
        if (typeof x === "string") {
            const reversedString = Tokenization.reverseString(x);
            return [...reversedString.tokens, ...reversedString.intrinsic];
        }
        if (Array.isArray(x)) {
            const reversedList = Tokenization.reverseList(x);
            if (reversedList) {
                return [reversedList];
            }
            const reversedNumberList = Tokenization.reverseNumberList(x);
            return reversedNumberList ? [reversedNumberList] : [];
        }
        if (typeof x === "number") {
            const reversedNumber = Tokenization.reverseNumber(x);
            return reversedNumber ? [reversedNumber] : [];
        }
        if (typeof x === "object" && x !== null) {
            const reversedMap = Tokenization.reverseMap(x);
            return reversedMap ? [reversedMap] : [];
        }
        return []; // null or undefined cannot be reversed
    }
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseNumberList(l) {
        return token_map_1.TokenMap.instance().lookupNumberList(l);
    }
    /**
     * Un-encode a Tokenized value from a map
     */
    static reverseMap(m) {
        return token_map_1.TokenMap.instance().lookupMap(m);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return (0, resolve_1.resolve)(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: options.preparing !== undefined ? options.preparing : false,
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return (typeof obj === "object" &&
            obj !== null &&
            typeof obj.resolve === "function");
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({
                produce: (context) => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== "number" ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== "number" ? x : `${x}`;
        }
    }
}
_b = JSII_RTTI_SYMBOL_1;
Tokenization[_b] = { fqn: "cdktf.Tokenization", version: "0.17.0-pre.18" };
exports.Tokenization = Tokenization;
//# sourceMappingURL=data:application/json;base64,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