"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processDynamicAttributes = exports.keysToSnakeCase = exports.snakeCase = exports.deepMerge = exports.terraformBinaryName = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const terraform_dynamic_block_1 = require("./terraform-dynamic-block");
const token_1 = require("./tokens/token");
exports.terraformBinaryName = process.env.TERRAFORM_BINARY_NAME || "terraform";
/**
 * Merges `source` into `target`, overriding any existing values.
 * `undefined` will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    if (token_1.Tokenization.isResolvable(target) && sources.length > 0) {
        throw new Error(`Invalid usage. Target (${target.toString()}) can not be a resolvable token when overrides are specified. Please replace the value of the field you are overriding with a static value.`);
    }
    for (const source of sources) {
        if (typeof source !== "object" || typeof target !== "object") {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof value === "object" && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = {};
                }
                // if the value is a resolvable we don't want to recurse into it
                if (token_1.Tokenization.isResolvable(value)) {
                    target[key] = value;
                    continue;
                }
                deepMerge(target[key], value);
            }
            else if (typeof value === "object" &&
                value != null &&
                Array.isArray(value)) {
                if (Array.isArray(target[key])) {
                    target[key] = [...target[key], ...value];
                }
                else {
                    target[key] = value;
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
exports.deepMerge = deepMerge;
/**
 * Transforms a string to snake case
 */
function snakeCase(str) {
    if (!str)
        return "";
    return String(str)
        .replace(/^[^A-Za-z0-9]*|[^A-Za-z0-9]*$/g, "")
        .replace(/([a-z0-9])([A-Z])/g, (_m, a, b) => a + "_" + b.toLowerCase())
        .replace(/[^A-Za-z0-9]+|_+/g, "_")
        .toLowerCase();
}
exports.snakeCase = snakeCase;
/**
 * Transforms all keys in a object to snake case
 */
function keysToSnakeCase(object) {
    if (token_1.Tokenization.isResolvable(object)) {
        return object;
    }
    if (Array.isArray(object)) {
        return object.map((e) => {
            return typeof e === "object" ? keysToSnakeCase(e) : e;
        });
    }
    const keys = Object.keys(object);
    return keys.reduce((newObject, key) => {
        if (key === "tags" || key === "environment") {
            newObject[key] = object[key];
            return newObject;
        }
        let value = object[key];
        if (typeof value === "object") {
            value = keysToSnakeCase(value);
        }
        if (Array.isArray(value)) {
            value = value.map((e) => {
                return typeof e === "object" ? keysToSnakeCase(e) : e;
            });
        }
        newObject[snakeCase(key)] = value;
        return newObject;
    }, {});
}
exports.keysToSnakeCase = keysToSnakeCase;
/**
 * dynamic attributes are located at a different position than normal block attributes
 * This method detects them and moves them from .attributeName to .dynamic.attributeName
 * It also invokes the .toTerraform() method on the dynamic attribute to get the correct
 * Terraform representation
 */
function processDynamicAttributes(attributes) {
    const result = {};
    Object.entries(attributes).forEach(([attributeName, value]) => {
        if (terraform_dynamic_block_1.TerraformDynamicBlock.isTerraformDynamicBlock(value)) {
            if (!result.dynamic) {
                result.dynamic = {};
            }
            result.dynamic[attributeName] = value.toTerraformDynamicBlockJson();
        }
        else {
            const recurse = value !== null &&
                typeof value === "object" &&
                value.constructor === Object; // only descend into plain objects
            result[attributeName] = recurse ? processDynamicAttributes(value) : value;
        }
    });
    return result;
}
exports.processDynamicAttributes = processDynamicAttributes;
//# sourceMappingURL=data:application/json;base64,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