"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidateBinaryVersion = void 0;
const child_process_1 = require("child_process");
const semver = require("semver");
/**
 * A validation that can be applied to a construct that will error if the
 * construct is being used in an environment with a version of a binary lower than the specified version.
 */
class ValidateBinaryVersion {
    constructor(binary, versionConstraint, versionCommand, hint) {
        this.binary = binary;
        this.versionConstraint = versionConstraint;
        this.versionCommand = versionCommand;
        this.hint = hint;
    }
    validate() {
        try {
            const versionOfCommand = (0, child_process_1.execSync)(this.versionCommand, {
                stdio: "pipe",
            }).toString();
            const semverRegex = /\d+\.\d+\.\d+/;
            const version = versionOfCommand.match(semverRegex);
            if (!version) {
                return [
                    `Could not determine version of ${this.binary} (running ${this.versionCommand})`,
                ];
            }
            if (!semver.satisfies(version[0], this.versionConstraint)) {
                return [
                    `${this.binary} version ${version[0]} is lower than the required version ${this.versionConstraint} for this construct. ${this.hint || ""}`,
                ];
            }
            return [];
        }
        catch (e) {
            return [
                `Could not determine version of ${this.binary}, ${this.versionCommand} failed: ${e}`,
            ];
        }
    }
}
exports.ValidateBinaryVersion = ValidateBinaryVersion;
//# sourceMappingURL=data:application/json;base64,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