"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidateProviderPresence = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const constructs_1 = require("constructs");
const terraform_provider_1 = require("../terraform-provider");
const terraform_resource_1 = require("../terraform-resource");
const terraform_data_source_1 = require("../terraform-data-source");
const terraform_stack_1 = require("../terraform-stack");
/**
 * A validation that is added by default, ensuring that all providers
 * used are defined via Constructs.
 */
class ValidateProviderPresence {
    /**
     * Creates a ValidateProviderPresence
     */
    constructor(host) {
        this.host = host;
        /**
         * All provider names found in the construct tree.
         */
        this.providerNames = new Set();
        /**
         * All TerraformProviders found in the construct tree.
         */
        this.foundProviders = [];
    }
    /**
     * Checks the construct tree recursively
     */
    check(node) {
        if (terraform_resource_1.TerraformResource.isTerraformResource(node) ||
            terraform_data_source_1.TerraformDataSource.isTerraformDataSource(node)) {
            if (node.terraformGeneratorMetadata) {
                this.providerNames.add(node.terraformGeneratorMetadata.providerName);
            }
        }
        if (terraform_provider_1.TerraformProvider.isTerraformProvider(node)) {
            this.foundProviders.push(node);
        }
        for (const child of constructs_1.Node.of(node).children) {
            this.check(child);
        }
    }
    /**
     * Run the validation
     */
    validate() {
        this.check(this.host);
        const missingProviders = Array.from(this.providerNames).filter((name) => {
            return !this.foundProviders.some((p) => p.terraformGeneratorMetadata?.providerName === name);
        });
        if (missingProviders.length === 0) {
            return [];
        }
        else {
            const stack = terraform_stack_1.TerraformStack.of(this.host);
            return [
                `Found resources without a matching provider construct. Please make sure to add provider constructs [e.g. new RandomProvider(...)] to your stack '${stack.node.id}' for the following providers: ${missingProviders.join(", ")}`,
            ];
        }
    }
}
exports.ValidateProviderPresence = ValidateProviderPresence;
//# sourceMappingURL=data:application/json;base64,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