import os
from functools import lru_cache

from pydantic_settings import BaseSettings


class AppSettings(BaseSettings):
    """
    应用配置
    """
    app_env: str = 'dev'
    app_name: str = 'FasAPI-Admin'
    app_root_path: str = '/dev-api'
    app_host: str = '0.0.0.0'
    app_port: int = 9099
    app_version: str = '1.4.0'
    app_reload: bool = True
    app_ip_location_query: bool = True
    app_same_time_login: bool = True
    app_cache_dir: str = 'data/cache'
    app_log_dir: str = 'logs'


class JwtSettings(BaseSettings):
    """
    Jwt配置
    """
    jwt_secret_key: str = 'b01c66dc2c58dc6a0aabfe2144256be36226de378bf87f72c0c795dda67f4d55'
    jwt_algorithm: str = 'HS256'
    jwt_expire_minutes: int = 144000
    jwt_redis_expire_minutes: int = 300


class DataBaseSettings(BaseSettings):
    """
    数据库配置
    """
    db_host: str = '127.0.0.1'
    db_port: int = 3306
    db_username: str = 'root'
    db_password: str = 'mysqlroot'
    db_database: str = 'dash-fastapi'
    db_echo: bool = True
    db_expire_on_commit: bool = False
    db_max_overflow: int = 10
    db_pool_pre_ping: bool = True
    db_pool_size: int = 50
    db_pool_recycle: int = 3600
    db_pool_timeout: int = 30


class RedisSettings(BaseSettings):
    """
    Redis配置
    """
    redis_host: str = '127.0.0.1'
    redis_port: int = 6379
    redis_username: str = ''
    redis_password: str = ''
    redis_database: int = 2


class CachePathConfig:
    """
    缓存目录配置
    """
    @staticmethod
    def get_path():
        """获取缓存完整路径"""
        from backlin.config.env import AppConfig
        cache_dir = AppConfig.app_cache_dir
        if os.path.isabs(cache_dir):
            return cache_dir
        return os.path.join(os.path.abspath(os.getcwd()), cache_dir)

    @staticmethod
    def get_path_str():
        """获取缓存路径字符串标识"""
        return 'cache'

    # 为了保持向后兼容,提供属性访问
    @property
    def PATH(self):
        return self.get_path()

    @property
    def PATHSTR(self):
        return self.get_path_str()


# 创建单例实例,供向后兼容使用
_cache_path_config_instance = None


def get_cache_path_config():
    """获取缓存路径配置单例"""
    global _cache_path_config_instance
    if _cache_path_config_instance is None:
        _cache_path_config_instance = CachePathConfig()
    return _cache_path_config_instance


class RedisInitKeyConfig:
    """
    系统内置Redis键名
    """
    ACCESS_TOKEN = {'key': 'access_token', 'remark': '登录令牌信息'}
    SYS_DICT = {'key': 'sys_dict', 'remark': '数据字典'}
    SYS_CONFIG = {'key': 'sys_config', 'remark': '配置信息'}
    CAPTCHA_CODES = {'key': 'captcha_codes', 'remark': '图片验证码'}
    ACCOUNT_LOCK = {'key': 'account_lock', 'remark': '用户锁定'}
    PASSWORD_ERROR_COUNT = {'key': 'password_error_count', 'remark': '密码错误次数'}
    SMS_CODE = {'key': 'sms_code', 'remark': '短信验证码'}


class GetConfig:
    """
    获取配置
    """

    @lru_cache()
    def get_app_config(self):
        """
        获取应用配置
        """
        # 实例化应用配置模型
        return AppSettings()

    @lru_cache()
    def get_jwt_config(self):
        """
        获取Jwt配置
        """
        # 实例化Jwt配置模型
        return JwtSettings()

    @lru_cache()
    def get_database_config(self):
        """
        获取数据库配置
        """
        # 实例化数据库配置模型
        return DataBaseSettings()

    @lru_cache()
    def get_redis_config(self):
        """
        获取Redis配置
        """
        # 实例化Redis配置模型
        return RedisSettings()

# 实例化获取配置类
get_config = GetConfig()
# 应用配置
AppConfig = get_config.get_app_config()
# Jwt配置
JwtConfig = get_config.get_jwt_config()
# 数据库配置
DataBaseConfig = get_config.get_database_config()
# Redis配置
RedisConfig = get_config.get_redis_config()
