# Enhanced Input Capture Guide

The `get_input()` function now captures comprehensive input information including mouse position, button types, hold duration, modifier keys, and timestamps.

## Features

### 1. Mouse Position ✓
Captures exact (x, y) coordinates where the mouse was when clicked.

### 2. Button Detection ✓
Identifies which button was clicked: `left`, `right`, or `middle`

### 3. Input Type ✓
Distinguishes between `mouse` clicks and `key` presses

### 4. Hold Duration ✓
Tracks how long a button or key was held down (optional)

### 5. Modifier Keys ✓
Detects state of Ctrl, Shift, Alt, and Win keys

### 6. Timestamps ✓
Records exact time when input occurred

### 7. Visual Feedback Spinner ✓
Shows an animated spinner during long holds (>0.5s) when tracking duration

### 8. Additional Data ✓
- Virtual key codes for keyboard
- Jitter tolerance for mouse movement
- Timeout support

---

## API Reference

```python
get_input(
    timeout=None,        # Maximum wait time (seconds)
    verbose=True,        # Print waiting message
    track_duration=False,  # Track hold duration
    detailed=False,      # Return dictionary instead of tuple
    show_spinner=True    # Show spinner during long holds (>0.5s)
)
```

### Parameters

- **timeout** (float, optional): Maximum time to wait for input in seconds. Raises `TimeoutError` if exceeded.
- **verbose** (bool): If `True`, prints "Waiting for input..." message.
- **track_duration** (bool): If `True`, measures how long the button/key was held down.
- **detailed** (bool): If `True`, returns a dictionary with all available information.
- **show_spinner** (bool): If `True`, shows an animated spinner during long holds (>0.5 seconds) when `track_duration=True`. Default: `True`.

### Return Values

#### Simple Mode (detailed=False) - Default
Returns a tuple:
- **Mouse**: `('mouse', x, y, button)`
- **Keyboard**: `('key', key_name)`

#### Detailed Mode (detailed=True)
Returns a dictionary with keys:

**Common keys:**
- `type`: `'mouse'` or `'key'`
- `timestamp`: Unix timestamp when input was detected
- `modifiers`: Dictionary with `ctrl`, `shift`, `alt`, `win` boolean states
- `duration`: Hold duration in seconds (only if `track_duration=True`)

**Mouse-specific keys:**
- `x`, `y`: Mouse coordinates (integers)
- `button`: `'left'`, `'right'`, or `'middle'`

**Keyboard-specific keys:**
- `key`: Key name (string)
- `vk_code`: Virtual key code (integer)

---

## Usage Examples

### Example 1: Simple Mode (Backwards Compatible)

```python
from alyios_windows import get_input

result = get_input()

if result[0] == 'mouse':
    _, x, y, button = result
    print(f"Clicked at ({x}, {y}) with {button} button")
else:
    _, key = result
    print(f"Pressed key: {key}")
```

### Example 2: Detailed Information

```python
from alyios_windows import get_input

result = get_input(detailed=True)

print(f"Type: {result['type']}")
print(f"Timestamp: {result['timestamp']}")
print(f"Modifiers: {result['modifiers']}")

if result['type'] == 'mouse':
    print(f"Position: ({result['x']}, {result['y']})")
    print(f"Button: {result['button']}")
else:
    print(f"Key: {result['key']}")
```

### Example 3: Track Hold Duration with Spinner

```python
from alyios_windows import get_input

print("Click and hold...")
# Spinner will appear after 0.5 seconds
result = get_input(track_duration=True, detailed=True, show_spinner=True)

print(f"Held for {result['duration']:.3f} seconds")
```

### Example 3b: Track Duration Without Spinner

```python
from alyios_windows import get_input

print("Click and hold...")
# No visual feedback during hold
result = get_input(track_duration=True, detailed=True, show_spinner=False)

print(f"Held for {result['duration']:.3f} seconds")
```

### Example 4: Detect Modifier Keys

```python
from alyios_windows import get_input

result = get_input(detailed=True)

if result['modifiers']['ctrl']:
    print("Ctrl was held!")

if result['modifiers']['shift']:
    print("Shift was held!")
```

### Example 5: Record Input Sequence

```python
from alyios_windows import get_input

# Record 10 inputs with full details
sequence = []
for i in range(10):
    inp = get_input(track_duration=True, detailed=True, verbose=False)
    sequence.append(inp)
    print(f"{i+1}. {inp['type']} - duration: {inp['duration']:.2f}s")

# Replay later...
for inp in sequence:
    if inp['type'] == 'mouse':
        click(inp['x'], inp['y'], button=inp['button'])
    else:
        press_key(inp['key'])
```

### Example 6: Timeout Handling

```python
from alyios_windows import get_input

try:
    result = get_input(timeout=5.0)
    print("Got input:", result)
except TimeoutError:
    print("No input within 5 seconds")
```

### Example 7: Keyboard Shortcuts Detection

```python
from alyios_windows import get_input

print("Press Ctrl+C, Ctrl+V, or another combo...")
result = get_input(detailed=True)

if result['type'] == 'key':
    mods = result['modifiers']
    key = result['key']

    if mods['ctrl'] and key == 'c':
        print("Copy command detected!")
    elif mods['ctrl'] and key == 'v':
        print("Paste command detected!")
    elif mods['ctrl'] and mods['shift'] and key == 'esc':
        print("Task Manager shortcut!")
```

---

## Use Cases

### 1. **Macro Recording**
```python
# Record user actions with timing
actions = []
for _ in range(10):
    action = get_input(track_duration=True, detailed=True, verbose=False)
    actions.append(action)
```

### 2. **Input Analytics**
```python
# Analyze user behavior
clicks = []
for _ in range(100):
    inp = get_input(detailed=True, verbose=False)
    if inp['type'] == 'mouse':
        clicks.append((inp['x'], inp['y'], inp['duration']))

# Calculate average click duration
avg_duration = sum(c[2] for c in clicks) / len(clicks)
print(f"Average click duration: {avg_duration:.3f}s")
```

### 3. **Accessibility Tools**
```python
# Detect long holds for accessibility features
result = get_input(track_duration=True, detailed=True)

if result['duration'] > 2.0:
    print("Long press detected - triggering accessibility menu")
```

### 4. **Gaming Input Capture**
```python
# Capture combo moves
combo = []
start_time = time.time()

while time.time() - start_time < 3.0:
    try:
        inp = get_input(timeout=0.5, detailed=True, verbose=False)
        combo.append(inp['key'] if inp['type'] == 'key' else inp['button'])
    except TimeoutError:
        break

print("Combo:", ' -> '.join(combo))
```

### 5. **Gesture Recognition**
```python
# Track mouse movement patterns
positions = []
for _ in range(5):
    inp = get_input(detailed=True, verbose=False)
    if inp['type'] == 'mouse':
        positions.append((inp['x'], inp['y']))

# Analyze pattern
if is_circular_pattern(positions):
    print("Circle gesture detected!")
```

---

## Migration Guide

### From Old API

**Before:**
```python
# Sequential - must do one then the other
x, y, button = get_click()
key = get_key()
```

**After:**
```python
# Either/or - whichever comes first
result = get_input()

if result[0] == 'mouse':
    _, x, y, button = result
else:
    _, key = result
```

### Getting More Information

**Simple:**
```python
result = get_input()  # Basic info
```

**Detailed:**
```python
result = get_input(detailed=True)  # All info
```

**With Duration:**
```python
result = get_input(track_duration=True, detailed=True)
print(f"Held for {result['duration']:.2f}s")
```

---

## Performance Notes

- **Default mode**: Very fast, minimal overhead
- **Detailed mode**: Adds ~1-2ms per capture
- **Duration tracking**: Blocking until release (by design)
- **Jitter tolerance**: 5 pixels for mouse movement during holds

## Thread Safety

The function uses Windows API polling and is thread-safe when called from different threads. However, only one input capture should be active per application to avoid race conditions.

## Platform

- **Supported**: Windows only (uses Windows API via ctypes)
- **Python**: 3.7+
- **Dependencies**: None (pure Python + Windows API)
