# Alyios Windows - Packaging & Distribution Guide

## What's Included

This package now includes the **Automation Studio** - a visual automation builder that generates Python code for the `alyios-windows` library.

## Package Structure

```
AlyiosWindowsFunctions/
├── src/
│   └── alyios_windows/
│       ├── __init__.py
│       ├── inputs.py
│       ├── mouse.py
│       ├── keyboard.py
│       ├── dialogs.py
│       ├── data.py
│       ├── bin/
│       │   └── automation-studio.exe  # Bundled Automation Studio
│       ├── helpers/
│       │   └── *.exe
│       └── studio.py  # Launcher script
├── setup.py
├── pyproject.toml
└── README.md
```

## Dependencies

The package automatically installs:
- **pandas** >= 2.0.0 (for data manipulation)
- **openpyxl** >= 3.0.0 (for Excel support)
- **chardet** >= 5.0.0 (for encoding detection)
- **pyperclip** >= 1.8.0 (for clipboard operations - used by generated scripts)

## Building the Package

### Step 1: Build Automation Studio

```bash
cd C:\Users\Willi\Documents\Alyios-Open-Source\AlyiosAutomation\automation-studio
powershell -ExecutionPolicy Bypass -File build_production.ps1
```

This creates: `automation-studio\src-tauri\target\release\automation-studio.exe`

### Step 2: Copy Executable to Package

```bash
copy automation-studio\src-tauri\target\release\automation-studio.exe ..\..\AlyiosWindowsFunctions\src\alyios_windows\bin\
```

### Step 3: Build Python Package

```bash
cd C:\Users\Willi\Documents\Alyios-Open-Source\AlyiosWindowsFunctions

# Clean previous builds
if exist dist rmdir /s /q dist
if exist build rmdir /s /q build
if exist src\alyios_windows.egg-info rmdir /s /q src\alyios_windows.egg-info

# Build the package
python -m build
```

This creates:
- `dist/alyios_windows-0.6.0-py3-none-any.whl`
- `dist/alyios-windows-0.6.0.tar.gz`

### Step 4: Test Locally (Optional but Recommended)

```bash
# Create a virtual environment for testing
python -m venv test_env
test_env\Scripts\activate

# Install your package from the wheel
pip install dist\alyios_windows-0.6.0-py3-none-any.whl

# Test launching the studio
alyios-studio

# Test importing the library
python -c "from alyios_windows import inputs, mouse, keyboard; print('Success!')"

# Deactivate when done
deactivate
```

### Step 5: Upload to PyPI

```bash
# Install twine if needed
pip install twine

# Upload to TestPyPI first (recommended)
python -m twine upload --repository testpypi dist/*

# If TestPyPI works, upload to real PyPI
python -m twine upload dist/*
```

## User Installation & Usage

### Installation from PyPI

```bash
pip install alyios-windows
```

### Launching Automation Studio

After installation, users can launch the Automation Studio in two ways:

**Method 1: Command Line**
```bash
alyios-studio
```

**Method 2: Python Script**
```python
from alyios_windows import studio
studio.main()
```

### Using the Generated Code

The Automation Studio generates Python scripts that use this library:

```python
#!/usr/bin/env python3
"""
Generated automation script
"""
import time
import pyperclip
from alyios_windows import inputs, mouse, keyboard
import pandas as pd  # If using CSV/Excel operations

def main():
    """Main automation script"""
    # Your automation actions here
    mouse.click(100, 200, button='left')
    keyboard.type('Hello World', interval=0.05)
    time.sleep(1.0)

if __name__ == "__main__":
    main()
```

## What Gets Installed

When users run `pip install alyios-windows`, they get:

1. **Core Library**: All input simulation, dialogs, clipboard, etc.
2. **Automation Studio**: Visual automation builder (Windows executable)
3. **CLI Command**: `alyios-studio` command to launch the studio
4. **All Dependencies**: pandas, openpyxl, chardet, pyperclip

## File Size Considerations

The Automation Studio executable adds ~3-5 MB to the package size. This is acceptable for the convenience it provides.

## Platform Support

- **Automation Studio**: Windows only (Tauri .exe)
- **Core Library**: Windows only (uses Windows API)
- **Generated Scripts**: Windows only (uses Windows-specific automation)

## Version Management

Current version: **0.6.0**

To update the version:
1. Update `version` in `setup.py` (line 10)
2. Update `version` in `pyproject.toml` (line 7)
3. Rebuild and re-upload

## Troubleshooting

### Executable Not Found

If users get "Automation Studio executable not found":
1. Verify the exe is in the package: `pip show -f alyios-windows | findstr automation-studio`
2. Reinstall: `pip install --force-reinstall alyios-windows`

### Missing Dependencies

If pyperclip or pandas is missing:
```bash
pip install --upgrade alyios-windows
```

This ensures all dependencies are installed.

## Development Notes

- **TypeScript Config**: `noUnusedLocals` and `noUnusedParameters` are set to `false` for production builds
- **Build Tools**: Requires Visual Studio Build Tools for Rust compilation
- **Bundle Type**: Using default Tauri bundles (no specific bundle type specified)
