# Alyios Windows Functions

A pure Python package for Windows-specific utilities with **zero external dependencies**. All functionality is implemented using native Python and Windows API via `ctypes` and C# helpers.

## Features

### 1. Interactive Console Interface
- Arrow key navigation (↑/↓)
- Visual selection indicators (→)
- Support for lists, dictionaries, and Enums
- Multi-selection with checkboxes
- Confirmation dialogs
- Text input with password masking

### 2. Mouse & Keyboard Input
**Capture:**
- Capture mouse clicks with position and button info
- Get current mouse position
- Capture keyboard key presses
- Continuous event listeners for both mouse and keyboard

**Simulation:**
- Simulate mouse clicks at specific coordinates
- Move mouse cursor programmatically
- Simulate keyboard key presses
- Type text automatically
- Send key combinations (Ctrl+C, Alt+Tab, etc.)

All implemented via Windows API

### 3. File Dialogs
- Native Windows file selection dialogs
- Folder selection dialogs
- Save file dialogs
- Support for file type filters
- Multi-file selection

### 4. Clipboard Operations
- Get/set text from clipboard
- Get/set file paths
- Clear clipboard
- Copy/paste helpers

### 5. Screen Capture & Display Info
- Take screenshots (full screen or region)
- Multi-monitor support
- Get display information (resolution, count, DPI)
- List all connected displays
- Pixel color detection and monitoring
- Wait for color changes
- Search for colors on screen

### 6. Notifications
- Windows toast notifications
- Custom icons (info, warning, error)
- Configurable timeout
- Helper functions for common notification types

### 7. Audio Control
- Get/set system volume
- Mute/unmute audio
- Increase/decrease volume
- Check if muted

### 8. Automation Studio
Visual automation builder for creating Python automation scripts without coding.

**Launch the Studio:**
```bash
# After installing alyios-windows
alyios-studio
```

**Or from Python:**
```python
from alyios_windows import studio
studio.main()
```

**Features:**
- Visual drag-and-drop interface for building automations
- Support for CSV/Excel data processing with pandas
- Variable substitution with `{column_name}` syntax
- Mouse and keyboard automation
- File dialogs, clipboard operations, and more
- Export to Python code using the alyios-windows library
- Instant clipboard paste for high-performance data entry

## Installation

**From PyPI (Recommended):**
```bash
pip install alyios-windows
```

**From Source:**
```bash
cd AlyiosWindowsFunctions
pip install -e .
```

## Usage Examples

### Console Interface

```python
from AlyiosWindowsFunctions import select_option, select_from_list, confirm

# Simple selection with arrow keys
choice = select_option(
    "Choose your favorite color:",
    ["Red", "Green", "Blue", "Yellow"]
)

# Using Enums
from enum import Enum

class Options(Enum):
    OPTION_A = "a"
    OPTION_B = "b"

choice = select_option("Select option:", Options)

# Multi-selection
items = select_from_list(
    "Choose items:",
    ["Item 1", "Item 2", "Item 3"],
    multi=True  # Use Space to toggle, Enter to confirm
)

# Yes/No confirmation
if confirm("Continue?", default=True):
    print("Confirmed!")
```

### Mouse & Keyboard Input

**Capture:**
```python
from AlyiosWindowsFunctions import get_click, get_key, get_mouse_position

# Get current mouse position
x, y = get_mouse_position()
print(f"Mouse at: ({x}, {y})")

# Wait for a mouse click
x, y, button = get_click()
print(f"Clicked at ({x}, {y}) with {button} button")

# Wait for a key press
key = get_key()
print(f"You pressed: {key}")

# Continuous event listener
from AlyiosWindowsFunctions import on_click_event

def handle_click(x, y, button, pressed):
    if pressed:
        print(f"Clicked at ({x}, {y})")

listener = on_click_event(handle_click)
# ... do work ...
listener.stop()
```

**Simulation:**
```python
from AlyiosWindowsFunctions import click, move_mouse, press_key, type_text, send_keys

# Simulate mouse click at coordinates
click(500, 300)  # Left click
click(500, 300, button='right')  # Right click
click(500, 300, clicks=2)  # Double-click

# Move mouse cursor
move_mouse(1000, 500)

# Simulate keyboard input
press_key('enter')
press_key('f5')

# Type text automatically
type_text("Hello, World!")

# Send key combinations
send_keys('ctrl+c')  # Copy
send_keys('ctrl+v')  # Paste
send_keys('alt+tab')  # Switch windows
```

### File Dialogs

```python
from AlyiosWindowsFunctions import select_file, select_folder, save_file_dialog

# Select a single file
filepath = select_file(
    title="Open File",
    filetypes=[("Text files", "*.txt"), ("All files", "*.*")]
)

# Select multiple files
files = select_file(title="Select Files", multiple=True)

# Select a folder
folder = select_folder(title="Choose Directory")

# Save file dialog
filepath = save_file_dialog(
    title="Save As",
    initialfile="document.txt",
    defaultextension=".txt",
    filetypes=[("Text files", "*.txt")]
)
```

### Clipboard Operations

```python
from AlyiosWindowsFunctions import clipboard

# Get text from clipboard
text = clipboard.get_text()
print(f"Clipboard: {text}")

# Set text to clipboard
clipboard.set_text("Hello, World!")

# Copy/paste shortcuts
clipboard.copy("Some text")
text = clipboard.paste()

# Clear clipboard
clipboard.clear()

# Get files from clipboard
files = clipboard.get_files()
if files:
    for file in files:
        print(f"File: {file}")

# Set files to clipboard
clipboard.set_files(["C:\\file1.txt", "C:\\file2.txt"])
```

### Screen Capture & Display

```python
from AlyiosWindowsFunctions import screen

# Take a screenshot of primary monitor
screen.screenshot("screenshot.png")

# Capture secondary monitor
screen.screenshot("monitor2.png", screen=1)

# Capture specific region (x, y, width, height)
screen.screenshot("region.png", region=(100, 100, 800, 600))

# Get primary display resolution
width, height = screen.get_primary_resolution()
print(f"Resolution: {width}x{height}")

# List all displays
displays = screen.list_displays()
for display in displays:
    print(display)

# Get detailed display info
info = screen.get_display_info()
print(f"Number of screens: {info['screen_count']}")

# Get screen count
count = screen.get_screen_count()
print(f"{count} displays connected")

# Get pixel color at coordinates
color = screen.get_pixel_color(100, 100)
if color:
    r, g, b = color
    print(f"Color at (100, 100): RGB({r}, {g}, {b})")

# Wait for color change at position (useful for detecting UI changes)
screen.wait_for_color_change(500, 500, timeout=30)

# Wait for specific color (useful for waiting for buttons to become enabled)
target_color = (0, 120, 215)  # Windows blue
screen.wait_for_color(500, 500, target_color, tolerance=10, timeout=30)

# Find color on screen
position = screen.find_color_on_screen(target_color, tolerance=10)
if position:
    x, y = position
    print(f"Found color at ({x}, {y})")

# Search in specific region for better performance
region = (0, 0, 800, 600)  # x, y, width, height
position = screen.find_color_on_screen(target_color, region=region, tolerance=10)
```

### Notifications

```python
from AlyiosWindowsFunctions import notifications

# Show a simple notification
notifications.notify("Hello", "This is a notification!")

# Show with different icons
notifications.notify("Warning", "Something needs attention", icon="warning")
notifications.notify("Error", "An error occurred", icon="error", timeout=10000)

# Helper functions
notifications.show_info("Success", "Operation completed")
notifications.show_warning("Warning", "Low disk space")
notifications.show_error("Error", "Failed to connect")

# Quick toast
notifications.toast("Task completed!")
```

### Audio Control

```python
from AlyiosWindowsFunctions import audio

# Get current volume
volume = audio.get_volume()
print(f"Current volume: {volume}%")

# Set volume
audio.set_volume(50)  # Set to 50%
audio.set_volume(100)  # Set to maximum

# Mute/unmute
audio.mute()
audio.unmute()

# Increase/decrease volume
audio.increase_volume(10)  # Increase by 10%
audio.decrease_volume(5)   # Decrease by 5%

# Check if muted
if audio.is_muted():
    print("Audio is muted")
```

## Technical Details

### No External Dependencies
This package uses **only**:
- Python standard library (`ctypes`, `getpass`, `os`, `sys`, `threading`, `time`, `enum`, `typing`, `subprocess`, `pathlib`)
- Windows API via `ctypes` (user32.dll, kernel32.dll)
- C# helpers compiled to native executables (DialogHelper.exe, WindowsHelper.exe)

### Architecture
The package uses two approaches:
1. **Direct ctypes bindings** for console and input operations
2. **C# helpers** for complex UI operations (dialogs, clipboard, screen, notifications, audio)

This hybrid approach provides:
- Zero external Python dependencies
- Native Windows integration
- Clean, maintainable code
- High performance

### Windows API Features Used
- **Console API**: For arrow key input and cursor control
- **Low-level hooks**: For mouse and keyboard capture
- **Windows Forms**: For modern file dialogs and clipboard operations
- **GDI+**: For screen capture
- **System.Drawing**: For image manipulation
- **NotifyIcon**: For toast notifications
- **WinMM**: For audio control
- **GDI32**: For pixel color detection (GetDC, GetPixel, ReleaseDC)

### Keyboard Controls

#### Single Selection:
- `↑/↓`: Navigate options
- `Enter` or `Space`: Select
- `Esc`: Cancel

#### Multi-Selection:
- `↑/↓`: Navigate options
- `Space`: Toggle selection
- `Enter`: Confirm selections
- `Esc`: Cancel

## Running the Demo

```bash
cd AlyiosWindowsFunctions
python example.py
```

The demo showcases all features with interactive examples.

## Requirements

- Python 3.7+
- Windows OS (uses Windows-specific APIs)

## License

MIT License
